#!/usr/bin/env ruby

require "logger"
require "nkf"
require "optparse"
require "socket"

params = {
  :BindAddress => "127.0.0.1",
  :Port => 10025,
}

parser = OptionParser.new
parser.on("--bind-address=ADDR", "Bind address") do |addr|
  params[:BindAddress] = addr
end
parser.on("--port=NUM", "Listening port number") do |num|
  params[:Port] = num.to_i
end

begin
  parser.parse!
rescue OptionParser::ParseError => err
  $stderr.puts err.message
  $stderr.puts parser.help
  exit 1
end

case $KCODE
when "SJIS"
  def decode(line)
    NKF.nkf("-ms", line)
  end
when "EUC"
  def decode(line)
    NKF.nkf("-me", line)
  end
else
  $KCODE = "u"
  def decode(line)
    NKF.nkf("-mw", line)
  end
end

STDOUT.sync = true
logger = Logger.new(STDOUT)

class FakeSmtpd
  def initialize(sock, logger)
    @sock = sock
    @logger = logger
  end

  def puts(line)
    line = line.gsub(/\r?$/, "\r\n")
    @logger.info { "S> #{line.inspect}" }
    @sock.write(line)
  end

  def gets
    line = @sock.gets
    line_decoded = decode(line)
    @logger.info "<C #{line.dump} (#{line_decoded.inspect})"
    line
  end

  def run
    mailname = "localhost"
    puts "220 #{mailname} ESMTP fake smtpd"
    while line = gets
      case line
      when /^(HELO|EHLO) /
        puts "250 #{mailname}"
      when /^MAIL FROM:/
        puts "250 2.1.0 Ok"
      when /^RCPT TO:/
        puts "250 2.1.5 Ok"
      when /^DATA/
        puts "354 End data with <CR><LF>.<CR><LF>"
        while line = gets
          if /^\.\r?\n/ =~ line
            puts "250 2.0.0 Ok: not queued"
            break
          end
        end
        return unless line
      when /^QUIT/
        puts "221 2.0.0 Bye"
        return
      else
        puts "502 5.5.2 Error: command not recognized"
        return
      end
    end
  end
end

trap("INT") { STDERR.puts "INT signal received."; exit!(0) }
if RUBY_PLATFORM =~ /mswin/
  Thread.start do
    while true
      sleep 1
    end
  end
end

logger.info { "Listen on #{params[:BindAddress]}:#{params[:Port]}" }
server = TCPServer.open(params[:BindAddress], params[:Port])
while true
  STDERR.puts "accept"
  Thread.start(server.accept) do |sock|
    logger.info { "accepted: #{sock.peeraddr.inspect}" }
    begin
      FakeSmtpd.new(sock, logger).run
    rescue Exception => e
      logger.error(e)
    ensure
      sock.close
    end
  end
end
