module ActiveRecord

  # 操作者の情報を記録するモジュール。
  module UserMonitor

    def self.included(base)
      base.alias_method_chain :create, :user
      base.alias_method_chain :update, :user
      base.extend(ClassMethods)
    end

    module ClassMethods
      def user_monitor
        class_eval do
          alias :created_by :created_by_with_user
          alias :updated_by :updated_by_with_user
        end
      end
    end

    # <tt>create</tt> を置き換える。
    # カラム <tt>created_by</tt> や <tt>updated_by</tt> が存在する場合は作成者を記録する。
    def create_with_user
      write_attribute("created_by", User.current_id) if self.class.column_names.include?("created_by") && created_by.nil?
      write_attribute("updated_by", User.current_id) if self.class.column_names.include?("updated_by")
      create_without_user
    end

    # <tt>update</tt> を置き換える。
    # カラム <tt>updated_by</tt> が存在する場合は更新者を記録する。
    def update_with_user
      write_attribute("updated_by", User.current_id) if self.class.column_names.include?("updated_by")
      update_without_user
    end

    # 作成者を <tt>User</tt> インスタンスとして返す。
    # 存在しない場合は nil を返す。
    def created_by_with_user
      begin
        User.find(read_attribute("created_by"))
      rescue ActiveRecord::RecordNotFound
        nil
      end
    end

    # 更新者を <tt>User</tt> インスタンスとして返す。
    # 存在しない場合は nil を返す。
    def updated_by_with_user
      begin
        User.find(read_attribute("updated_by"))
      rescue ActiveRecord::RecordNotFound
        nil
      end
    end
  end
end

ActiveRecord::Base.class_eval do
  include ActiveRecord::UserMonitor
end
