# -*- coding: utf-8 -*-
require 'iconv'

# PDF 文書のモデル。
class SimplePDF < FPDF
  class TextBox
    include DocumentStyles::StyleAlignment
    include DocumentStyles::StyleMargin

    def initialize(text, font_size=10, options={})
      @text = text
      @font_size = font_size

      options.each do |key,val|
        case(key)
        when :top_margin
          self.margin[:top] = val
        when :bottom_margin
          self.margin[:bottom] = val
        when :left_margin
          self.margin[:left] = val
        when :right_margin
          self.margin[:right] = val
        when :align
          self.align = val
        end
      end
    end

    attr_accessor :text, :font_size
  end

  # overrided method that is called automatically
  def Header
    unless @header.text.empty?
      SetXY(@header.margin[:left], @header.margin[:top])
      SetFontSize(@header.font_size)
      Cell(page_width - @header.margin[:left] - @header.margin[:right],
           @footer.font_size / @k,
           extract_fields(@header.text),
           0, 0,
           @header.align_expr)
      SetXY(left_margin, right_margin)
    end
  end

  # overrided method that is called automatically
  def Footer
    unless @footer.text.empty?
      SetXY(@footer.margin[:left], @footer.margin[:top])
      SetFontSize(@footer.font_size)
      Cell(page_width - @footer.margin[:left] - @footer.margin[:right],
           @footer.font_size / @k,
           extract_fields(@footer.text),
           0, 0,
           @footer.align_expr)
    end
  end

  # overrided the original method because of the case sensitive fault (@Page => @page)
  def Link(x, y, w, h, link)
    # Put a link on the page
    @PageLinks[@page]=Array.new unless @PageLinks.has_key?(@page)
    @PageLinks[@page].push([x*@k,@hPt-y*@k,w*@k,h*@k,link])
  end

  alias :output :Output

  def initialize(options={})
    orientation = options[:orientation] ? options[:orientation] : "Portrait"
    unit = "mm" # The unit of measure is fixed for simple implementation.
    format = options[:format] ? options[:format] : "A4"
    super(orientation, unit, format)

    set_default(options[:language])
    
    options.each do |key,val|
      case(key)
      when :title
        SetTitle(val)
      when :author
        SetAuthor(val)
      when :creator
        SetCreator(val)
      when :keywords
        SetKeywords(val.join(' '))
      when :subject
        SetSubject(val)
      when :top_margin
        SetTopMargin(val) # mm
      when :left_margin
        SetLeftMargin(val) # mm
      when :right_margin
        SetRightMargin(val) # mm
      when :zoom
        case(val)
        when "fullpage", "fullwidth", "real"
          SetDisplayMode(val)
        else
          SetDisplayMode("default")
        end
      when :header
        @header = val.dup
      when :footer
        @footer = val.dup
      end
    end

    AliasNbPages("%P")
  end

  # alternative methods
  def write(h, utf8_txt)
    Write(h, encode_text(utf8_txt))
  end

  def cell(w,h=0,utf8_txt='',border=0,ln=0,align='',fill=0,link='')
    Cell(w, h, encode_text(utf8_txt), border, ln, align, fill, link)
  end

  def multi_cell(w,h,utf8_txt,border=0,align='J',fill=0)
    MultiCell(w, h, encode_text(utf8_txt), border, align, fill)
  end

  # color setting methods
  def set_draw_color(rgb)
    set_color(:SetDrawColor, rgb)
  end

  def set_text_color(rgb)
    set_color(:SetTextColor, rgb)
  end

  def set_fill_color(rgb)
    set_color(:SetFillColor, rgb)
  end

  private

  def set_color(setter, rgb)
    method(setter).call(rgb[0], rgb[1], rgb[2]) if rgb.kind_of?(Array)
  end

  def set_default(language)
    case language
    when "ja", "Japanese"
      extend(PDF_Japanese)
      AddSJISFont()
      SetFont('SJIS')
      @ic = Iconv.new('CP932', 'UTF-8')
    when "zh", "Chinese"
      extend(PDF_Chinese)
      AddBig5Font()
      SetFont('Big5')
      @ic = Iconv.new('Big5', 'UTF-8')
    when "ko", "Korean"
      extend(PDF_Korean)
      AddUHCFont()
      SetFont('UHC')
      @ic = Iconv.new('UHC', 'UTF-8')
    else
      SetFont('Arial')
      @ic = nil
    end

    @header = SimplePDF::TextBox.new("%t", 8,
                                     :align => :left,
                                      :top_margin => 5.0,
                                      :left_margin => 10.0,
                                      :right_margin => 10.0)
    @footer = SimplePDF::TextBox.new("%p", 8,
                                     :align => :center,
                                      :top_margin => -20.0,
                                      :left_margin => 10.0,
                                      :right_margin => 10.0)
  end

  def encode_text(utf8_str)
    return utf8_str unless @ic
    res = @ic.iconv(utf8_str)
    res << @ic.iconv(nil)
    res
  end

  # encode UTF-8 string to UCS-2 (Big Endian) to get it fit document
  # information property fields like <FEFF4E2D672C>.
  def encode_info_text(utf8_str)
    str = "<FEFF"
    Iconv.iconv("UTF-16BE", "UTF-8", utf8_str).join.unpack('C*').each do |c|
      if c < 0xf
        str << "0" << c.to_s(16).upcase
      else
        str << c.to_s(16).upcase
      end
    end
    return str << ">"
  end

  # override the super class's method to output UTF-8 string directly
  def putinfo
    out('/Producer '+textstring('Ruby FPDF '+FPDF_VERSION));
    out('/Title'+encode_info_text(@title)) if @title
    out('/Subject'+encode_info_text(@subject)) if @subject
    out('/Author'+encode_info_text(@author)) if @author
    out('/Keywords'+encode_info_text(@keywords)) if @keywords
    out('/Creator'+encode_info_text(@creator)) if @creator
    out('/CreationDate'+textstring('D:'+creation_date(Time.now)))
  end

  private

  def creation_date(time)
    ret = time.strftime("%Y%m%d%H%M%S")

    # time zone
    offset_min = time.utc_offset.abs / 60 % 60
    offset_hour = time.utc_offset.abs / 60 / 60

    ret << (time.utc_offset>=0 ? "+" : "-")
    ret << '0' if offset_hour < 10
    ret << offset_hour.to_s
    ret << '\''
    ret << '0' if offset_min < 10
    ret << offset_min.to_s
    ret << '\''

    return ret
  end

  def page_width
    return @w
  end

  def page_height
    return @h
  end

  def left_margin
    return GetMargins()[0]
  end

  def top_margin
    return GetMargins()[1]
  end

  def right_margin
    return GetMargins()[2]
  end

  def drawable_width
    return page_width - left_margin - right_margin
  end

  def extract_fields(text)
    ret = text.dup
    ret.gsub!(/\%t/){ encode_text(@title ? @title : "") }
    ret.gsub!(/\%a/){ encode_text(@author ? @author : "") }
    ret.gsub!(/\%s/){ encode_text(@subject ? @subject : "") }
    ret.gsub!(/\%p/){ PageNo().to_s }

    # %P (total page number) will be replaced when the document is closed
    # by FPDF library.

    return ret
  end
end
