# PDF 文書のスタイルに関するモジュール。
module DocumentStyles

  # 色のためのクラス。
  class Color
    def initialize(r = 0, g = 0, b = 0)
      @red   = r.to_i
      @green = g.to_i
      @blue  = b.to_i
      validate
    end

    attr_reader :red, :green, :blue

    def red=(val)
      @red = val.to_i
      validate
    end

    def green=(val)
      @green = val.to_i
      validate
    end

    def blue=(val)
      @blue = val.to_i
      validate
    end

    def to_a
      return [red, green, blue]
    end

    private

    def validate
      to_a.each do |val|
        if val < 0 || val > 255
          raise "Color element value #{val} is out of bound (0-255)"
        end
      end
    end
  end

  # 長方形を表現するモジュール。
  module Rectangle
    def []=(side, val)
      case side
      when :top, :bottom, :left, :right
        @rectangle[side] = val
      else
        raise "Invalid side #{side.inspect}"
      end
    end

    def [](side)
      case side
      when :top, :bottom, :left, :right
        return @rectangle[side]
      else
        raise "Invalid side #{side.inspect}"
      end
    end
  end

  # ボーダー幅のモデル。
  class BorderWidth
    include Rectangle

    def initialize
      @rectangle = {
        :top    => 0.2,
        :bottom => 0.2,
        :left   => 0.2,
        :right  => 0.2
      }
    end

    def common_width
      if (@rectangle[:top] == @rectangle[:bottom]) && (@rectangle[:top] == @rectangle[:right]) && (@rectangle[:top] == @rectangle[:left])
        return @rectangle[:top]
      else
        return nil
      end
    end
  end

  # ボーダー幅のスタイルを表すモジュール。
  module StyleBorderWidth
    def border_width
      @border_width = BorderWidth.new unless @border_width
      return @border_width
    end

    def border_width=(val)
      if val.kind_of?(BorderWidth)
        @border_width = val
      else
        raise "Invalid instance was specified."
      end
    end
  end

  # マージンを表現するモジュール。
  class Margin
    include Rectangle

    def initialize
      @rectangle = {
        :top    => 0.0,
        :bottom => 0.0,
        :left   => 0.0,
        :right  => 0.0
      }
    end
  end

  # マージンのスタイルを表すモジュール。
  module StyleMargin
    def margin
      @margin = Margin.new unless @margin
      return @margin
    end

    def margin=(val)
      if val.kind_of?(Margin)
        @margin = val
      else
        raise "Invalid instance was specified."
      end
    end
  end

  # 位置合わせのスタイルを表すモジュール。
  module StyleAlignment
    DEFAULT_ALIGN = :left
    EXPRESSION = {
      :left   => 'L',
      :center => 'C',
      :right  => 'R'
    }

    def align=(val)
      case(val)
      when :left, :center, :right
        @align = val
      else
        raise "Invalid alignment was specified #{val.inspect}"
      end
    end

    def align
      return @align ? @align : DEFAULT_ALIGN
    end

    def align_expr
      return EXPRESSION[self.align]
    end
  end
end

