# -*- coding: utf-8 -*-
# /usr/bin/env ruby

#
# access_log_backup.rb
#
# DB上のアクセスログをファイルにバックアップする
# バックアップが終わったログはDBから削除される
#
# バックアップの形式は以下のようになる
#
#   --------------------------------------------------
#   ID = 12
#   UUID = e31680b5-9fe2-4a71-99cf-8e5d602663e0
#   Login ID = 5
#   Access Time = 2009-03-05 13:57:08
#   IP Address = 127.0.0.1
#   Response time = 1.734834
#   Parameters:
#     action = index
#     controller = menu
#   --------------------------------------------------
#


module AccessLogBackup

  DIRECTORY = 'log/accesslog'

  class Filename

    @@count = 0

    def self.to_s
      serial = (@@count == 0) ? '' : "_#{@@count}"
      date   = Date.today.strftime("%y%m%d")
      "#{DIRECTORY}/accesslog_#{date}#{serial}.log"
    end

    def self.increment; @@count += 1 end
    def self.count; @@count end
  end

  class << self
    def run
      ActiveRecord::Base.establish_connection(RAILS_ENV.to_sym)

      ActiveRecord::Base.transaction do
        File.open(Filename.to_s, 'wb') do |io|
          io.puts header
          AccessLog.all.each do |log|
            io.puts backup log
            log.destroy
          end
          io.puts footer
        end
      end
    end

    private
    def separator(mark)
      mark * 50 + "\n"
    end

    def header
      result = ''
      result << separator('=')
      result << "AccessLog backup file\n"
      result << "#{Date.today} - #{Filename.count}\n"
      result << separator('=')
      result << "\n"
      result
    end

    def footer
      result = ''
      result << separator('=')
      result << "\n"
      result
    end

    def backup(log)
      result = ''
      result << separator('-')
      result << format_without_parameters(log)
      result << format_parameters(log)
      result << separator('-')
      result << "\n"
      result
    end

    def format_without_parameters(log)
      result = ''
      [
        ['ID',            :id           ],
        ['UUID',          :uuid         ],
        ['Login ID',      :login_id     ],
        ['Accessed At',   :accessed_at  ],
        ['IP Address',    :ip_address   ],
        ['Response time', :response_time],
      ].each do |name, key|
        if value = log.__send__(key)
          result << "#{name} = #{value}\n"
        end
      end
      result
    end

    def format_parameters(log)
      result = ''
      if params = log.__send__(:parameters)
        result <<  "Parameters:\n"
        parameters = YAML.load(params)
        if parameters.kind_of? Hash
          parameters.each do |key, value|
            result << "  #{key} = #{value}\n"
          end
        else
          result << "#{params}\n"
        end
      end
      result
    end
  end
end

