# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: storage_objects
#
#  id           :integer       not null, primary key
#  domain_id    :integer       not null
#  name         :string(255)
#  mime_type    :string(255)   default("application/octet-stream"), not null
#  uri          :string(255)
#  length       :integer
#  created_at   :string(14)
#  updated_at   :string(14)
#  created_by   :integer
#  updated_by   :integer
#  created_in   :integer
#  updated_in   :integer
#  lock_version :integer       default(0), not null
#

# サーバ上に保存された添付ファイルのモデル。
class StorageObject < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor
  has_one :attachment, :as => :file, :dependent => :destroy

  after_destroy :unlink

  # 保存先としてのラベルの文字列を返す。
  def location
    s_("rfw|StorageObject|Server")
  end

  # 保存先のパスを返す。
  def path
    storage = Storage.current
    storage.root + storage.separator + id.to_s
  end

  # ファイル <em>x</em> を保存する。
  def store(x)
    ConfigAttachment.current.validate_filename(self.name)
    if Storage.current.available?
      if x.is_a?(Tempfile)
        FileUtils.mv(x.path, path)
      else
        File.open(path, "wb") do |f|
          f.write x.read
        end
      end
      File.chmod(0644, path)
      self.length = File.size(path)
      begin
        ConfigAttachment.current.validate_length(self.length)
      rescue ConfigAttachment::MaximumLengthError
        unlink
        raise
      end
    else
      self.length = x.length
    end
    # x has instance method 'original_filename'.
    # cf. http://www.ruby-doc.org/stdlib/libdoc/cgi/rdoc/classes/CGI.html
    if x.respond_to?("original_path") # Rails 2.0.x -- http://dev.rubyonrails.org/changeset/7759
      fpath = x.original_path
    elsif x.respond_to?("full_original_filename") # Rails 1.2.6 -- http://dev.rubyonrails.org/changeset/2345
      fpath = x.full_original_filename
    else
      fpath = x.original_filename
    end
    epath = URI.encode(fpath.gsub('\\','/'))
    self.uri = "file:///#{epath}"
    return save
  end

  # ファイルの内容を読み出す。
  def content
    begin
      File.open(path, "rb", &:read)
    rescue Errno::ENOENT
      ""
    end
  end

  # 複製操作のためにコピーする。失敗した場合は <tt>false</tt> を返す。
  def copy
    copied = self.class.new
    copied.attributes = attributes
    if copied.save
      if length == File.open(copied.path, "wb") {|f| f.write File.open(path, "rb", &:read)}
        return copied
      end
    end
    return false
  end

  private

  def unlink
    unless new_record?
      begin
        File.unlink(path)
      rescue SystemCallError
        # ignored
      end
    end
    return true
  end
end
