# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: people
#
#  id                      :integer       not null, primary key
#  domain_id               :integer       not null
#  name                    :string(255)   not null
#  code                    :string(255)
#  mail_address            :string(255)
#  last_language           :string(255)
#  default_company_id      :integer
#  default_organization_id :integer
#  default_post_id         :integer
#  receive_mail            :boolean       default(TRUE)
#  created_at              :string(14)
#  updated_at              :string(14)
#  created_by              :integer
#  updated_by              :integer
#  created_in              :integer
#  updated_in              :integer
#  lock_version            :integer       default(0), not null
#

# 個人のモデル。
class Person < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor
  belongs_to :domain
  has_one :user, :dependent => :destroy
  has_many :company_members, :dependent => :destroy
  has_many :posted_tos, :dependent => :destroy
  has_many :organization_members, :dependent => :destroy
  has_many :group_members, :dependent => :destroy
  has_many :personal_group_members, :dependent => :destroy
  has_many :grant_ons, :as => :roleable, :dependent => :destroy

  validates_presence_of :name

  attr_accessor :keep_original_membership

  before_update DefaultMembership.new(:company      => CompanyMember,
                                      :organization => OrganizationMember,
                                      :post         => PostedTo)
  after_create DefaultMembership.new(:company      => CompanyMember,
                                     :organization => OrganizationMember,
                                     :post         => PostedTo)

  # <em>p</em> と <em>q</em> が同一かどうかを判定する。
  def self.exists_with_people?(p, q)
    return (p == q)
  end

  # 所属する企業を配列で返す。
  def companies
    company_members.map(&:company).flatten
  end

  # 役職を配列で返す。
  def posts
    posted_tos.map(&:post).flatten
  end

  # 所属する組織を配列で返す。
  def organizations
    organization_members.map(&:organization).flatten
  end

  # 所属するグループを配列で返す。
  def groups
    group_members.map(&:group).flatten
  end

  # 同じ組織に属する個人を配列で返す。
  def colleagues
    organizations.inject([]) do |seed, organization|
      seed | organization.people
    end
  end

  # 同じグループに属する個人を配列で返す。
  def neighbors
    groups.inject([]) do |seed, group|
      seed | group.people
    end
  end

  # 名前および企業の名前と組織の名前を返す。
  def name_with_in
    if company = preferred_company
      if organization = preferred_organization
        "#{name} (#{preferred_company.name}/#{preferred_organization.name})"
      else
        "#{name} (#{preferred_company.name})"
      end
    end
    return name
  end

  # 優先される所属企業を返す。
  def preferred_company
    get_preferred("company")
  end

  # 優先される役職を返す。
  def preferred_post
    get_preferred("post")
  end

  # 優先される所属組織を返す。
  def preferred_organization
    get_preferred("organization")
  end

  # 個人の集合を返す。
  def people
    [self]
  end

  # メールの受信が可能である場合は true、さもなくば false を返す。
  def can_be_recipient?
    !mail_address.blank? && receive_mail
  end

  private

  def get_preferred(column)
    items = __send__(column.pluralize)
    prefer = __send__("default_#{column}_id")
    items.each do |it|
      return it if it.id == prefer
    end
    return items[0]
  end

end
