# 画面生成を制御する。
class ProductGeneratorController < ApplicationController
  before_filter :admin_only

  # 表示する言語を設定する。
  def apply_language
    if params[:language] && Language.find_by_code(params[:language])
      session[:language] = params[:language]
    end
    if session[:language]
      params[:lang] = session[:language]
    end
  end
  private :apply_language
  if ENV["RAILS_ENV"] != "test"
    before_init_gettext :apply_language
  end

  # <tt>list</tt> へリダイレクトする。
  def index
    x_redirect_to "main", nil, :action => "list"
  end

  # 機能の一覧を表示する。
  def list
    view_in :main
    sync_fragment "main", :page, 1
    @product_pages, @products = paginate :products, :order => "id"
  end

  # 機能を表示する。
  def show
    view_in :side
    @product = Product.find(params[:id])
  end

  # 機能を新規作成する。
  def new
    view_in :side
    unless request.post? && params[:product]
      @product = Product.new
      render :action => "product_name_form"
      return
    end

    product_class = params[:product]["type"].constantize
    unless product_class < Product
      # should not happen via usual UI
      raise "invalid product type"
    end
    product_class.transaction do
      @product = product_class.create!({
                                         :code => params[:product]["code"],
                                         :name_po => 0,
                                         :model_name => params[:product]["model_name"],
                                       })
      rest_attributes = params[:product].dup
      [:code, :model_name].each {|k| rest_attributes.delete(k)}
      @product.attributes = rest_attributes
      @product.save!
    end
    flash[:notice] = s_("rfw|flash|notice|Product was successfully created.")
    x_redirect_to "side", :close, :action => "list"
  rescue ActiveRecord::RecordInvalid
    unless @product
      @product = product_class.for_error_message(params[:product])
    end
    render :action => "product_name_form"
  end

  # 機能の編集を行う。
  def edit
    view_in :product
    @sub_view = nil
    @product = Product.find(params[:id])
    unless request.post? && params[:product]
      render :action => "product_name_form"
      return
    end

    @product.attributes = params[:product]
    @product.save!
    flash[:notice] = s_("rfw|flash|notice|Product was successfully updated.")
    x_redirect_to "product", :close, :action => "list"
  rescue ActiveRecord::ActiveRecordError
    @product ||= Product.new
    render :action => "product_name_form"
  end

  # 機能に紐づいた画面を編集する。
  def manage
    view_in :product
    @product = Product.find(params[:id])
    @displays = @product.displays.find(:all,
                                       :conditions => {:type => ['DisplayToList','DisplayToShow','DisplayToEdit','DisplayToNew']},
                                       :order => "id")
    @displays_count = @displays.size
    unless request.post? && params[:product]
      render :action => "product_content_form"
      return
    end

    failed = false
    @product.attributes = params[:product]
    @displays.each do |display|
      display.attributes = params[:display][display.id.to_s]
      failed = true unless display.save
    end
    if !failed && @product.save
      flash[:notice] = s_("rfw|flash|notice|Product was successfully updated.")
      x_redirect_to "product", :close, :action => "show"
    else
      @product ||= Product.new
      render :action => "product_content_form"
    end
  rescue ActiveRecord::ActiveRecordError
    @product ||= Product.new
    render :action => "product_content_form"
  end

  # 一覧画面を新規作成する。
  def new_list
    view_in :product
    @product = Product.find(params[:id])
    unless request.post? && @product
      render
      return
    end

    # ':name_po => 0' indicates a transitive status.
    DisplayToList.create! :product_id => @product.id, :code => "NEW_LIST_#{rand.to_s[(2..-1)]}", :name_po => 0
    flash_message = s_("rfw|flash|notice|A list was successfully added.")

    if request.xhr?
      flash.now[:notice] = flash_message
      manage
    else
      flash[:notice] = flash_message
      redirect_to :action => "manage", :id => params[:id]
    end
  end

  # 画面を編集する。
  def display_edit
    view_in :display
    setup_display
    unless request.post? && params[:display]
      render
      return
    end

    @display.attributes = params[:display]
    if @display.save
      flash[:notice] = s_("rfw|flash|notice|Product was successfully updated.")
      x_redirect_to "display", :close, :action => "manage", :id => @product.id
    end
  end

  # 画面を削除する。
  def display_destroy
    view_in :display
    setup_display

    unless @params[:confirm_destroy]
      render :action => "display_edit"
      return
    end

    @display = Display.find(params[:id])
    if @display && @display.destroy
      flash[:notice] = s_("rfw|flash|notice|Display was successfully destroyed.")
    else
      flash[:warning] = s_("rfw|flash|warning|Display was failed to destroy.")
    end
    x_redirect_to @current_view, :close, :action => "manage", :id => @product.id
  end

  # 項目を編集する。
  def item_edit
    view_in :item
    @item = Item.find(params[:id])
    setup_item

    if params[:item]
      @item.attributes = params[:item]
      if @item.save
        flash[:notice] = s_("rfw|flash|notice|Item was successfully updated.")
        x_redirect_to @current_view, :close, :action => "display_edit", :id => @display.id
        return
      end
    end

    if @item.is_a?(ItemProper)
      render :action => "item_proper"
    else
      render :action => "item_pseudo"
    end
  end

  # 擬似項目を新規作成する。
  def item_new_pseudo
    view_in :item
    @item = ItemPseudo.new :display_id => params[:id]
    setup_item

    if params[:item]
      if @item.name_po_message = PoMessageSingular.create(:msgctxt => "", :msgid => params[:item]["name_en"])
        @item.attributes = params[:item]
        if @item.save
          flash[:notice] = s_("rfw|flash|notice|Item was successfully created.")
          x_redirect_to @current_view, :close, :action => "display_edit", :id => @display.id
          return
        end
      end
    end

    render :action => "item_pseudo"
  end

  private

  def view_in(key)
    case key
    when :main
      if request.xhr?
        @current_view = "view_m"
        @parent_view = "view_main"
      else
        @current_view = "view_main"
        @parent_view = nil
      end
      @sub_view = "view_side"
    when :side
      @parent_view = "view_main"
      @current_view = "view_side"
      @sub_view = "view_product"
    when :product
      @parent_view = "view_side"
      @current_view = "view_product"
      @sub_view = "view_display"
    when :display
      @parent_view = "view_product"
      @current_view = "view_display"
      @sub_view = "view_item"
    when :item
      @parent_view = "view_display"
      @current_view = "view_item"
      @sub_view = nil
    end
  end

  def setup_display
    @display = Display.find(params[:id], :include => :items, :order => "items.id")
    @product = @display.product
    @items = @display.items
  end

  def setup_item
    @display = @item.display
    @product = @display.product
  end
end
