# -*- coding: utf-8 -*-
# プロダクトの制御を行う:
# - 一覧画面
# - 詳細画面
# - 新規作成画面
# - 編集画面
# - 削除画面
class ProductController < ApplicationController
  include PaginationConfiguration
  include RandomId

  # <tt>list</tt> へリダイレクトする。
  def index
    redirect_to :action => :list
  end

  # 一覧画面を表示する。
  def list
    view_in :m
    prepare_display_to_list
    if @display.narrowing?
      if params[:narrowing_text]
        session[:display_narrowing] ||= HashWithIndifferentAccess.new
        if /\A[0-9]+\z/ =~ params[:display_narrowing_id].to_s
          display_narrowing = DisplayNarrowing.find(params[:display_narrowing_id].to_i)
        else
          display_narrowing = @display.active_display_narrowing(session)
        end
        dns = DisplayNarrowing.find(:all, :select => "display_id", :conditions => {:narrowing_id => display_narrowing.narrowing_id})
        dns.map(&:display_id).each do |display_id|
          session[:display_narrowing][display_id] ||= HashWithIndifferentAccess.new
          session[:display_narrowing][display_id][:id] = display_narrowing.narrowing_id
        end
        @display.narrowing_value_keys.each do |key|
          session[key] = nil
        end
        @display.narrowing_value_keys(session).each do |key|
          session[key] = params[:"narrowing_value_#{key}"]
        end
      end
      @display.narrow_to(session)
    end
    @items.each{|item|
      next unless item.condition_pattern == 'custom'
      keys = item.condition_value.split(';').last.split(',').map{|v| v.strip.to_sym }
      hash = {}
      keys.each{|k| hash[k] = params[k] }
      if hash.values.reject{|v| v.blank? }.empty?
        item.condition_hash = session["#{@display.id}_#{item.id}}"]
      else
        item.condition_hash = hash
        session["#{@display.id}_#{item.id}}"] = item.condition_hash
      end
    }
    options = @display.query_options(session)
    setup_list_id
    setup_per_page(@model_class, options) do |default_per_page|
      sync_fragment("m", :per, default_per_page)
    end
    sync_fragment("m", :page, 1)
    @pages, @things = paginate(@model_class, options)
    @header_per_line = User.list_header_per_line

    # 再表示ボタンやperのonchangeのときにpageの変更と同じように
    # fragmentのpopup部分を削除する。
    if params["dummy_button"]
      delete_sub_view_fragment(@current_view.sub(/\Aview_/, ""))
    end
  end

  # 詳細画面を表示する。
  def show
    view_in :detail
    prepare_display_to :show
    begin
      @it = @model_class.find(params[:id], :readonly => true)
    rescue ActiveRecord::RecordNotFound
      flash[:notice] = s_("rfw|flash|notice|It does not exist.")
      x_close_or_redirect_to :action => "list"
      return
    end
    prepare_display_to_workflow
    if @product.is_a?(ProductDetailed)
      prepare_details
    end

    if @display.document?
      @tab_label = @it.__send__(@product.document_name_method)
      @id_suffix = "#{@it.id}of#{@product.id}_#{random_id}"
    end

    return if use_picker

    if @display.mail?
      @disable_mailsend = true
      if params[:mail] && !params[:mail][:recipients].blank?
        @disable_mailsend = false
      end
      if !@disable_mailsend && params[:mailsend]
        mail_to_queue(@it, true)
      end
    end
  end

  # 詳細画面(関連文書用)を表示する。
  def show_only
    @parent_view = "view_detail"
    # @current_view,@sub_view は後で設定
    prepare_display_to :show
    begin
      @it = @model_class.find(params[:id], :readonly => true)
    rescue ActiveRecord::RecordNotFound
      flash[:notice] = s_("rfw|flash|notice|It does not exist.")
      x_close_or_redirect_to :action => "list"
      return
    end
    prepare_display_to_workflow
    if @product.is_a?(ProductDetailed)
      prepare_details
    end
    @id_suffix = "#{@it.id}of#{@product.id}_#{random_id}"
    @current_view = "view_detail#{@it.id}of#{@product.id}"
    @sub_view = "view_subofdetail#{@it.id}of#{@product.id}"

    return if use_picker

    if @display.mail?
      @disable_mailsend = true
      if params[:mail] && !params[:mail][:recipients].blank?
        @disable_mailsend = false
      end
      if !@disable_mailsend && params[:mailsend]
        mail_to_queue(@it, true)
      end
    end

    if params[:tab_id]
      render :update do |page|
        page.replace_html params[:tab_id], :partial => 'detail'
      end
    end
  end

  # 削除を行う。
  def destroy
    view_in :detail
    prepare_display_to :show
    raise PermissionDenied, "product is not modifiable" unless @product.modifiable?
    raise DisabledException, "disabled with respect to button" unless @display.button_delete?
    unless request.post? && params[:id] && (params[:confirm_destroy] || request.xhr?)
      begin
        @it = @model_class.find(params[:id])
      rescue ActiveRecord::RecordNotFound
        flash[:notice] = s_("rfw|flash|notice|It has been already destroyed.")
        x_close_or_redirect_to :action => "list"
        return
      end
      # @title : same as show
      @display = Object.new
      def @display.method_missing(sym, *args)
        return false
      end
      def @display.button_back?
        return true
      end
      render :action => :confirm_destroy
      return
    end
    begin
      @it = @model_class.find(params[:id])
      if @product.is_a?(ProductDetailed)
        @details = @it.details
        @details.each {|d| instance_variable_set("@details_#{d.id}", d)}
      end
      begin
        if with_logic(@it, :destroy)
          flash[:notice] = s_("rfw|flash|notice|It was successfully destroyed.")
        else
          flash[:warning] = s_("rfw|flash|warning|It was failed to destroy.")
        end
      rescue PermissionDenied
        @alert_message = s_("rfw|flash|warning|It could not be destroyed because of its permission.")
        render :action => "show"
        return
      rescue BusinessLogic::Failure
        @alert_message = s_("rfw|flash|warning|It could not be destroyed because of its business logic.")
        render :action => "show"
        return
      rescue BusinessLogic::RecordInvalid => e
        @alert_message = e.message
        render :action => "show"
        return
      rescue WorkflowError
        @alert_message = s_("rfw|flash|warning|It could not be destroyed because of its workflow.")
        render :action => "show"
        return
      end
    rescue ActiveRecord::RecordNotFound
      flash[:notice] = s_("rfw|flash|notice|It has been already destroyed.")
    end
    x_close_or_redirect_to :action => "list"
  end

  # 新規作成画面を表示し、新規作成を行う。
  def new
    @copying = !!params[:id]
    view_in :detail
    create("form") do
      flash[:notice] = s_("rfw|flash|notice|It was successfully created.")
      x_close_or_redirect_to :action => "list"
    end
  end

  # 編集画面を表示し、編集を行う。
  def edit
    view_in :detail
    prepare_display_to :edit
    begin
      @it = @model_class.find(params[:id])
    rescue ActiveRecord::RecordNotFound
      flash[:notice] = s_("rfw|flash|notice|It has been already destroyed.")
      x_close_or_redirect_to :action => "list"
      return
    end
    if @it.respond_to?(:writable?) && !@it.writable?
      flash[:warning] = s_("rfw|flash|warning|Can not edit it because of its permission.")
      x_close_or_redirect_to :action => "list"
      return
    end

    prepare_display_to_workflow

    return if use_picker

    if @product.is_a?(ProductDetailed)
      prepare_details
      @details.each {|d| instance_variable_set("@details_#{d.id}", d)}
      begin
        @old_details = ActiveSupport::JSON.decode(params[:old_details] || "[]")
      rescue ActiveSupport::JSON::ParseError
        @old_details = []
      end
      set_new_details
      set_order_details
    end

    if params[:it]
      set_attributes
      if request.post? && params[:update]
        # update
        begin
          if with_logic(@it, :update)
            flash[:notice] = s_("rfw|flash|notice|It was successfully updated.")
            x_close_or_redirect_to :action => "list"
            return
          end
        rescue ActiveRecord::StaleObjectError
          @stale_object_error = true
        rescue BusinessLogic::Failure => failure
          set_attributes
          @business_logic_failure = failure.message
        rescue WorkflowError
          set_attributes
          @workflow_error = true
        end
      end
    end

    render :action => "form"
  end

  # タブ内に編集画面を表示する
  # 編集する
  def edit_only
    prepare_display_to :edit
    begin
      @it = @model_class.find(params[:id])
    rescue ActiveRecord::RecordNotFound
      flash[:notice] = s_("rfw|flash|notice|It has been already destroyed.")
      x_close_or_redirect_to :action => "list"
      return
    end
    if @it.respond_to?(:writable?) && !@it.writable?
      flash[:warning] = s_("rfw|flash|warning|Can not edit it because of its permission.")
      x_close_or_redirect_to :action => "list"
      return
    end

    prepare_display_to_workflow

    @current_view = "view_detail#{@it.id}of#{@product.id}"
    @sub_view = "view_picker"

    return if use_picker

    if @product.is_a?(ProductDetailed)
      prepare_details
      @details.each {|d| instance_variable_set("@details_#{d.id}", d)}
      begin
        @old_details = ActiveSupport::JSON.decode(params[:old_details] || "[]")
      rescue ActiveSupport::JSON::ParseError
        @old_details = []
      end
      set_new_details
      set_order_details
    end

    if params[:it]
      set_attributes
      if request.post? && params[:update]
        # update
        begin
          if with_logic(@it, :update)
            flash[:notice] = s_("rfw|flash|notice|It was successfully updated.")
            redirect_to :action => 'show_only', :id => params[:id], :tab_id => params[:tab_id]
            return
          end
        rescue ActiveRecord::StaleObjectError
          @stale_object_error = true
        rescue BusinessLogic::Failure => failure
          set_attributes
          @business_logic_failure = failure.message
        rescue WorkflowError
          set_attributes
          @workflow_error = true
        end
      end
    end

    render :update do |page|
      page.replace_html params[:tab_id], :partial => 'form'
    end
  end

  # 関連文書を選択するための一覧を表示する。
  def select_document
    view_in :document
    prepare_display_to_list
    options = @display.query_options(session)
    setup_per_page(@model_class, options) do |default_per_page|
      params[:per] ||= default_per_page
    end
    @pages, @things = paginate(@model_class, options)
    @header_per_line = User.list_header_per_line
    begin
      @relatable = params[:type].constantize.find(params[:id])
    rescue
      @relatable = nil
    end
  end

  # 関連文書向けに新規作成する
  def create_document
    @copying = false
    view_in :document
    create("create_document") do
      p = {
        :type => params[:type],
        :product_id => params[:relatable_product_id],
        :target_id => @it.id,
        :target_type => @it.class.to_s,
        :target_product_id => @product.id,
      }
      p[:id] = params[:id] unless params[:id].blank?
      render :update do |page|
        page << "new Ajax.Updater('document_table_base', '#{url_for(:controller => "document", :action => "select")}', {onComplete: function(){$('view_dp').innerHTML = '';}, parameters: #{p.to_json}})"
      end
    end
  end

  private

  def view_in(key)
    case key
    when :m
      @parent_view = "view_main"
      @current_view = "view_m"
      @sub_view = "view_detail"
    when :detail
      @parent_view = "view_m"
      @current_view = "view_detail"
      @sub_view = "view_picker"
    when :document
      @parent_view = "view_de"
      @current_view = "view_dp"
      @sub_view = "view_dpp"
    end
    params[:id] ||= fetch_fragment(@current_view.sub(/\Aview_/, ""), "id", nil)
  end

  def display_list
    return @display_list if defined?(@display_list)
    x = @product.displays.select do |display|
      (display.is_a?(DisplayToListPrivate) && display.person_id == User.current.person_id) ||
        (display.class == DisplayToList && display.enabled?)
    end
    y, z = x.partition {|display| display.class == DisplayToList}
    @display_list = y.sort_by {|display| display.position.to_i} + z.sort_by {|display| display.position.to_i}
  end

  def prepare_display_to(display_type, display_class="display_to_#{display_type}".classify)
    @display_type = display_type
    @product ||= Product.find(params[:product_id])
    case @display_type
    when :show, :list
      raise PermissionDenied, "product is invisible" unless @product.visible?
    when :edit, :new
      raise PermissionDenied, "product is not modifiable" unless @product.modifiable?
    else
      raise ArgumentError, "unknown display type: #{@display_type}"
    end
    @display ||= @product.displays.find_by_type(display_class)
    raise NotFoundException, "display not found" unless @display
    raise DisabledException, "display disabled" unless @display.enabled?
    Display.current = @display
    @title ||= @display.name
    @title = "no title" if @title.blank?
    @model_class = @product.model_class
    @items = @display.items.find(:all, :conditions => ["layout > 0"], :order => "layout,id")
    @items = @items.select(&:selected?) if @display.is_a?(DisplayToListPrivate)
    if @product.is_a?(ProductDetailed)
      @detail_items, @items = @items.partition(&:detail?)
    end
  end

  def prepare_display_to_list
    begin
      @product = Product.find(params[:product_id])
    rescue ActiveRecord::RecordNotFound
      raise NotFoundException, "product not found"
    end
    raise NotFoundException, "display not found" if display_list.empty?
    list_id = params[:list] || fetch_fragment("m", :list, nil)
    if user_id = User.current_id
      if @default_list = DefaultList.find_by_product_id_and_user_id(@product.id, user_id)
        # use the default list unless specified
        list_id ||= @default_list.display_id
      end
    end
    if list_id
      params[:list] = list_id
      list_id = list_id.to_i
      @display = display_list.find {|d| d.id == list_id }
    end
    @title = @product.name
    prepare_display_to(:list)
  end

  def prepare_display_to_workflow
    return unless @display.workflow_enabled?
    client = WorkflowClient.client('probe')
    @wf_config = client.configuration(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                                      WorkflowStruct::Message.new(:body => wf_message_body),
                                      @product.workflow)
    unless @wf_config.issueable?
      wf_items = @wf_config.items
      @items.each do |item|
        if wf_items && (i = wf_items.detect{|v| v.id == item.id })
          item.validates_presence = i.validates_presence
        else
          item.instance_variable_set(:@writable, false)
        end
      end
    end
    params[:workflow] ||= {}
    params[:workflow][:option] ||= {}
  rescue Errno::ECONNREFUSED => ex
    logger.error("ERROR: workflow server down")
    logger.error(ex)
    logger.error(ex.backtrace.join("\n"))
    raise ServiceUnavailable, "Workflow server down"
  end

  def prepare_details
    if @product.is_a?(ProductDetailedForGrantData)
      @details = GrantOnWithoutPeriodic.find(
        :all,
        :conditions => {
          :grant_targettable_id => @it.id,
          :grant_targettable_type => @it.class.name,
        }
      )
    else
      @details = @it.details
    end
  end

  def set_new_details
    begin
      @new_details = ActiveSupport::JSON.decode(params[:new_details] || "[]")
    rescue ActiveSupport::JSON::ParseError
      @new_details = []
    end
    @new_details.each do |n|
      d = @product.detail_class.new
      d.header = @it
      instance_variable_set("@new_details_#{n}", d)
    end
  end

  def set_order_details
    begin
      @order_details = ActiveSupport::JSON.decode(params[:order_details] || "[]")
    rescue ActiveSupport::JSON::ParseError
      @order_details = []
    end
  end

  def create(template, &block)
    prepare_display_to :new
    params[:mail] ||= {}
    @it = @model_class.new
    @details = []
    @old_details = []

    #work flow
    prepare_display_to_workflow

    return if use_picker

    if @product.is_a?(ProductDetailed)
      set_new_details
      set_order_details
    end

    if params[:it]
      set_attributes
      if request.post? && params[:create]
        # create
        begin
          if with_logic(@it, :create)
            yield
            return
          end
        rescue BusinessLogic::Failure => failure
          set_attributes
          @business_logic_failure = failure.message
        rescue WorkflowError
          set_attributes
          @workflow_error = true
        end
      end
      return render(:action => template)
    end

    # start creating a brand new it
    if @display.attachment?
      # clean up uploaded_attachments
      session[:uploaded_attachments] ||= []
      session[:uploaded_attachments].delete_if do |attachment|
        @it.is_a?(attachment.attachable_type.constantize) && !attachment.attachable_id && attachment.destroy
      end
    end
    if @display.document?
      # clean up added documents
      session[:added_documents] ||= []
      session[:added_documents].delete_if do |document|
        @it.is_a?(document.relatable_type.constantize) && !document.relatable_id
      end
    end
    if @copying
      begin
        orig = @model_class.find(params[:id])
      rescue ActiveRecord::RecordNotFound
        flash[:notice] = s_("rfw|flash|notice|It has been already destroyed.")
        x_close_or_redirect_to :action => "list"
        return
      end
      @items.each do |i|
        next unless i.readable?
        key = i.column_name
        next if key == "id" # do not copy id
        case i
        when ItemProper, ItemPlural, ItemPolymorphic
          @it.__send__("#{key}=", orig.__send__(key)) unless i.fill_column_of(@it)
        end
      end
      if @product.is_a?(ProductDetailed)
        @new_details = []
        orig.details.each_with_index do |d, k|
          copied_detail = @product.detail_class.new
          @detail_items.each do |i|
            next unless i.readable?
            key = i.column_name
            next if key == "id" # do not copy id
            case i
            when ItemProper, ItemPlural, ItemPolymorphic
              copied_detail.__send__("#{key}=", d.__send__(key)) unless i.fill_column_of(copied_detail)
            end
          end
          instance_variable_set("@new_details_#{k + 1}", copied_detail)
          @new_details.push(k + 1)
        end
      end
    else
      @new_details = []
      @items.each do |i|
        i.initialize_column_of(@it)
        if i.input_type == 'picker'
          if /\A(matter|project|project_segment)/ =~ i.input_parameter
            @it[i.column_name] = initialize_id_with_narrowing(i)
          end
        end
      end
    end
    render :action => template
  end
  
  def initialize_id_with_narrowing(item)
    return nil unless item.is_a?(ItemProper)
    id = item.default_id_with_narrowing(session)
    return id if id
    return nil if item.input_initializer.blank?
    case item.input_initializer.strip
    when /\Anarrowing\((\d+)\)\z/
      narrowing = NarrowingById.find($1.to_i)
      key = narrowing.value_keys.first
      return session[key]
    end
  rescue ActiveRecord::RecordNotFound
    # fall through
  ensure
    nil
  end

  # options[:per_page] は正の値に保つ
  def setup_per_page(model_class, options, &block)
    @allowed_per_page_options = ALLOWED_PER_PAGE.map do |n|
      if n == 0
        [s_("rfw|select|option|ALL"), n]
      else
        [(ns_("rfw|select|option|per %{n} page", "per %{n} pages", n) % {:n => n}), n]
      end
    end

    default_per_page = User.list_default_per_page
    yield(default_per_page) if block_given?
    per_page = params[:per].to_i
    if per_page == 0
      options[:per_page] = count_collection_for_pagination(model_class, options)
    elsif ALLOWED_PER_PAGE.include?(per_page)
      options[:per_page] = per_page
    else
      options[:per_page] = per_page = default_per_page
    end
    options[:per_page] = 1 if options[:per_page] <= 0
  end

  def setup_list_id
    params[:list] ||= fetch_fragment("m", :list, nil)
    update_fragment("m", :list, @default_list && @default_list.display_id)
  end

  def default_url_options(options)
    h = super || {}
    # testのときにはparamsがnilになっているため。
    if params
      h.merge({
          :product_id => params[:product_id],
        })
    end
    return h
  end

  def set_attributes
    @items.each {|i| i.set_attributes(@it, params[:it])}
    @it.lock_version = params[:it][:lock_version]
    add_singleton_validations(@it, @items)
    if @product.is_a?(ProductDetailed)
      @details.each do |d|
        unless @old_details.include?(d.id)
          detail = instance_variable_get("@details_#{d.id}")
          @detail_items.each {|i| i.set_attributes(detail, params["details_#{d.id}".to_sym])}
          add_singleton_validations(detail, @detail_items)
        end
      end
      @new_details.each do |n|
        if detail = instance_variable_get("@new_details_#{n}")
          @detail_items.each {|i| i.set_attributes(detail, params["new_details_#{n}".to_sym])}
          add_singleton_validations(detail, @detail_items)
          if @product.respond_to?(:pivot)
            unless @product.pivot.blank?
              detail["#{@product.pivot}_id".to_sym] = @product.id
              detail["#{@product.pivot}_type".to_sym] = @product.class.name
            end
          end
        end
      end
    end
  end

  def add_singleton_validations(x, items, action = :save)
    x.extend ::CustomValidations::SingletonValidateable
    items.each {|i| i.apply_validations(x, action) } unless items.blank?
  end

  def valid?(it)
    v = it.valid?
    if @product.is_a?(ProductDetailed)
      @details.each do |d|
        unless @old_details.include?(d.id)
          detail = instance_variable_get("@details_#{d.id}")
          v &= detail.valid?
        end
      end
      @new_details.each do |n|
        detail = instance_variable_get("@new_details_#{n}")
        v &= detail.valid?
      end
    end
    return v
  end

  def with_before_and_after(detail, action, pool)
    message = (action == :destroy) ? :destroy : :save
    case message
    when :destroy
      add_singleton_validations(detail, @detail_items, :destroy)
      unless detail.valid_on_destroy?
        raise BusinessLogic::Failure, "validation on destroy was failed"
      end
    end
    unless detail = @display.inject_logic(:before, detail, action, pool)
      raise BusinessLogic::Failure, "beforeprocess returns #{detail}, so reverted"
    end
    unless detail.__send__(message)
      raise "#{detail} could not process, so reverted"
    end
    unless detail = @display.inject_logic(:after, detail, action, pool)
      raise BusinessLogic::Failure, "afterprocess returns #{detail}, so reverted"
    end
  end

  def with_logic(it, action)
    if it.respond_to?(:writable?) && !it.writable?
      raise PermissionDenied, "it is not writable: #{it.inspect}"
    end
    message = (action == :destroy) ? :destroy : :save
    newbie = it.new_record?
    case message
    when :save
      return false unless valid?(it)
    when :destroy
      add_singleton_validations(it, @items, :destroy)
      return false unless it.valid_on_destroy?
    end
    # execute
    pool = HashWithIndifferentAccess.new
    it.class.transaction do
      unless it = @display.inject_logic(:pre, it, action, pool)
        raise BusinessLogic::Failure, "preprocess returns #{it}, so reverted"
      end
      unless it.__send__(message)
        raise "#{it} could not process, so reverted"
      end
      if @product.is_a?(ProductDetailed)
        unless mid = @display.inject_logic(:mid, it, action, pool)
          raise BusinessLogic::Failure, "midprocess returns #{mid}, so reverted"
        end
        if message == :save
          details = []
          @details.each do |d|
            detail = instance_variable_get("@details_#{d.id}")
            if @old_details.include?(d.id)
              with_before_and_after(detail, :destroy, pool)
            else
              with_before_and_after(detail, action, pool)
              if k = params[:order_details].index("details_#{d.id}")
                details[k] = detail
              end
            end
          end
          @new_details.each do |n|
            detail = instance_variable_get("@new_details_#{n}")
            with_before_and_after(detail, action, pool)
            if k = params[:order_details].index("new_details_#{n}")
              details[k] = detail
            end
          end
          # order details
          details.each_with_index {|detail,k| detail.insert_at(k + 1) if detail}
        else
          @details.each do |d|
            detail = instance_variable_get("@details_#{d.id}")
            with_before_and_after(detail, action, pool)
          end
        end
      end
      if message == :save
        commit_attachments(it, newbie) if @display.attachment?
        commit_documents(it, newbie) if @display.document?
        mail_to_queue(it) if @display.mail?
      end
      unless post = @display.inject_logic(:post, it, action, pool)
        raise BusinessLogic::Failure, "postprocess returns #{post}, so reverted"
      end
      # call workflow web service
      if @display.workflow_enabled? && !params[:workflow].blank?
        case message
        when :save
          wf_action_name = params[:workflow][:action]
          if wf_action_name && WorkflowStruct::Configuration::POSSIBLE_RESPONSES.include?(wf_action_name)
            @wf_response = __send__("wf_#{wf_action_name}".to_sym)
            raise WorkflowError if @wf_response.failure?
          end
        when :destroy
          @wf_response = User.current.admin? ? wf_delete : wf_destroy
          raise WorkflowError if @wf_response.failure?
        end
      end
    end
    return true
  end

  # return true when rendered
  def use_picker
    session[:picker] ||= {}
    picker_session = session[:picker]
    if picker_session[:it_before_picker] &&
        picker_session[:workflow_before_picker] &&
        picker_session[:return_to] &&
        picker_session[:return_to][:controller] == params[:controller] &&
        picker_session[:return_to][:product_id] == params[:product_id] &&
        picker_session[:return_to][:action] == params[:action] &&
        picker_session[:return_to][:id] == params[:id]
      params[:workflow] = picker_session[:workflow_before_picker]
      # TODO : 他のも戻す
      unless @it.readonly?
        params[:it] = picker_session[:it_before_picker]
        @items.each {|i| i.set_attributes(@it, params[:it])}
        @it.lock_version = params[:it][:lock_version]
      end
      params[:mail] = picker_session[:mail_before_picker]
      if @product.is_a?(ProductDetailed)
        %w|new_details old_details order_details|.each do |k|
          params[k.to_sym] = picker_session["#{k}_before_picker".to_sym]
        end
        picker_session.each do |k,v|
          if /\A((?:new_)?details_\d+)_before_picker\z/ =~ k.to_s
            params[$1.to_sym] = v
          end
        end
      end
      session[:picker] = nil
      if flash[:pick]
        if flash[:pick][:mail]
          params[:mail] = flash[:pick][:mail]
        end
      end
    elsif params[:picker]
      picker, = params[:picker].keys
      field, item_id = get_picker_fields(picker)
      if field
        if "calendar" == field
          item = Item.find(item_id)
          if /\A(\d{4})\D?(\d{1,2})/ =~ params[:it][item.column_name]
            flash[:year], flash[:month] = $1, $2
          end
        end
        picker_session[:return_to] = {
          :controller => params[:controller],
          :product_id => params[:product_id],
          :action     => params[:action],
          :id         => params[:id],
          :tab_id     => params[:tab_id],
        }
        picker_session[:it_before_picker] = params[:it] || {}
        picker_session[:workflow_before_picker] = params[:workflow] || {}
        # TODO: picker で取得したデータを保存する
        picker_session[:mail_before_picker] = params[:mail]
        if @product.is_a?(ProductDetailed)
          %w|new_details old_details order_details|.each do |k|
            picker_session["#{k}_before_picker".to_sym] = params[k.to_sym]
          end
          params.each do |k,v|
            if /\A(?:new_)?details_\d+\z/ =~ k.to_s
              picker_session["#{k}_before_picker".to_sym] = v
            end
          end
        end
        x_redirect_to @sub_view, picker, {
          :controller   => "picker",
          :action       => field,
          :return_field => field,
        }
        return true
      end
    end
    return false
  end
  
  def get_picker_fields(key)
    # modify: add param(destination|bizcardcompany|bizcardbranch|bizcardperso) 2009/2/27 h.nakamura
    if /\A(?:mail|(company|organization|person|post|group|calendar|lump|destination|bizcardcompany|bizcardorganization|bizcardbranch|bizcardperson)(\d+)(?:_(?:new_)?details_\d+)?)\z/ =~ key
      return $1 || 'mail', $2
    elsif (/\Aworkflow/ === key)
      return 'person', nil
    else
      nil
    end
  end

  def commit_attachments(x, newbie)
    if session[:deleted_attachments].is_a?(Array)
      session[:deleted_attachments].delete_if do |attachment|
        attachment.attachable == x && attachment.destroy
      end
    end
    if session[:uploaded_attachments].is_a?(Array)
      if newbie
        session[:uploaded_attachments].delete_if do |a|
          if x.is_a?(a.attachable_type.constantize) && !a.attachable_id
            a.attachable_id = x.id
            a.save
          else
            false
          end
        end
      else
        session[:uploaded_attachments].delete_if {|a| x == a.attachable && a.save}
      end
    end
  end

  def commit_documents(x, newbie)
    if !newbie && session[:deleted_documents].is_a?(Array)
      session[:deleted_documents].delete_if(&:destroy)
    end
    if session[:added_documents].is_a?(Array)
      if newbie
        session[:added_documents].delete_if do |d|
          if x.is_a?(d.relatable_type.constantize) && !d.relatable_id
            d.relatable_id = x.id
            d.save
          else
            false
          end
        end
      else
        session[:added_documents].delete_if {|d| x == d.relatable && d.save}
      end
    end
  end

  def mail_to_queue(it, flash_now=false)
    mail = params[:mail]
    if mail && !mail[:recipients].blank?
      fragment = FragmentHash.new(params[:fragment])
      menu_id = fragment.motion("m").to_i
      mail_queue = MailQueue.create!({
          :menu_id => menu_id,
          :product_id => @product.id,
          :document => it,
          :from => User.current.person,
          :recipient_ids => mail[:recipients],
          :comment_message => mail[:comment],
          :field_type => mail[:field_type],
          :has_attachment => mail[:attachment],
          :mail_mode => "auto login",
          :copy_to_sender => false,
          :processed => false,
          :auto_login_url => url_for({
              :controller => "user",
              :action => "auto",
              # :t => "/", # "/" is default
              :f => fragment_for(:menu => Menu.find(menu_id), :document => it),
            }),
        })
      url = url_for(:only_path => true, :controller => "mail_sender", :action => "post", :id => mail_queue.id)
#       MiddleMan.new_worker(:class => :mail_sender_worker, :args => {:post => [mail_queue.id, url]})
      ap4r.async_to({
          :controller => "mail_sender",
          :action => "post",
        }, {
          :id => mail_queue.id,
        })
      if flash_now
        flash.now[:message] = s_("rfw|flash|message|Mail queued for delivery.")
      else
        flash[:message] = s_("rfw|flash|message|Mail queued for delivery.")
      end
    end
  end

  # workflow:起票 (申請)
  def wf_issue
    client = WorkflowClient.client('action')
    assignment = wf_get_assignments(params[:workflow][:option][:assignment])
    candidates = wf_get_assignments(params[:workflow][:option][:candidates])
    issue_option = params[:workflow][:option].dup
    issue_option[:assignment] = assignment.first
    issue_option[:candidates] = candidates
    response = client.issue(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                            WorkflowStruct::Department.new(:id => params[:workflow][:department_id]),
                            WorkflowStruct::Message.new(:body => wf_message_body),
                            WorkflowStruct::Route.new(:workflow => @product.workflow, :workflow_aux => params[:workflow][:workflow_aux]),
                            WorkflowStruct::IssueOption.new(issue_option))
  end

  # workflow:承認
  def wf_accept
    client = WorkflowClient.client('action')
    assignment = wf_get_assignments(params[:workflow][:option][:assignment])
    accept_option = params[:workflow][:option].dup
    accept_option[:assignment] = assignment.first
    accept_option[:condition_parameters] = wf_get_condition_parameters
    response = client.accept(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                             WorkflowStruct::Issue.new(:id => params[:workflow][:issue_id]),
                             WorkflowStruct::AcceptOption.new(accept_option))
  end

  # workflow:差戻し
  def wf_reject
    client = WorkflowClient.client('action')
    response = client.reject(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                             WorkflowStruct::Issue.new(:id => params[:workflow][:issue_id]))
  end

  # workflow:取消
  def wf_cancel
    client = WorkflowClient.client('action')
    response = client.cancel(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                             WorkflowStruct::Issue.new(:id => params[:workflow][:issue_id]))
  end

  # workflow:棄却
  def wf_withdraw
    client = WorkflowClient.client('action')
    response = client.withdraw(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                               WorkflowStruct::Issue.new(:id => params[:workflow][:issue_id]))
  end

  # workflow:削除
  def wf_destroy
    client = WorkflowClient.client('action')
    response = client.destroy(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                              WorkflowStruct::Issue.new(:id => params[:workflow][:issue_id]))
  end

  # workflow:管理者による削除
  def wf_delete
    client = WorkflowClient.client('action')
    response = client.delete(WorkflowStruct::Account.new(:person_id => User.current.person_id),
                             WorkflowStruct::Message.new(:body => wf_message_body))
  end

  # params[:workflow][:option] から必要なデータを取り出して加工する
  def wf_get_assignments(assignments)
    return [] if assignments.blank?
    assignments.map{|k, v|
      WorkflowStruct::Assignment.new(:id => k.to_i,
                                     :authorizers => v.values.reject{|id| id.blank? }.map{|person_id|
                                       WorkflowStruct::Account.new(:person_id => person_id.to_i)
                                     })
    }
  end

  def wf_get_condition_parameters
    return nil if @wf_config.condition_parameters.blank?
    @wf_config.condition_parameters.map{|c|
      v = ItemMethodChain.resolve_method_chain(@it, c.key)
      to_x = case v
             when String  then 'to_s'
             when Integer then 'to_i'
             when Float   then 'to_f'
             else              'to_s'
             end
      WorkflowStruct::ConditionParameter.new(:key   => c.key,
                                             :value => v,
                                             :to_x  => to_x)
    }
  end

  def wf_message_body
    # __send__ できるメソッドを制限する
    unless @it.class.column_names.include?(@product.workflow_body_method)
      raise ClientError, "Invalid workflow_body_method: #{@product.workflow_body_method}"
    end
    @it.__send__(@product.workflow_body_method)
  end

end
