class ApplicationTableGenerator < Rails::Generator::NamedBase
  attr_accessor :specs
  default_options :quiet => true, :skip_migration => false

  def initialize(*runtime_args)
    super
    @specs = args.collect {|a| TableColumnSpec.new(*a.split(':'))}
  end

  def manifest
    record do |m|
      # Check for class naming collisions.
      m.class_collisions class_path, class_name, "#{class_name}Test"

      # Model, test, and fixture directories.
      m.directory File.join('app/models', class_path)
      m.directory File.join('test/unit', class_path)
      m.directory File.join('test/fixtures', class_path)

      # Model class, unit test, and fixtures.
      m.template 'fixtures.yml',  File.join('test/fixtures', class_path, "#{table_name}.yml"), :collision => :skip, :quiet => true

      unless options[:skip_migration]
        m.migration_template 'migration.rb', 'db/migrate', :assigns => {
          :migration_name => "Create#{class_name.pluralize.gsub(/::/, '')}"
        }, :migration_file_name => "create_#{file_path.gsub(/\//, '_').pluralize}"
      end
    end
  end

  protected
    def banner
      "Usage: #{$0} generate TableName [field:type, field:type]"
    end

    def add_options!(opt)
      opt.separator ''
      opt.separator 'Options:'
      opt.on("--skip-migration", 
             "Don't generate a migration file for this table") { |v| options[:skip_migration] = v }
    end
end

class TableColumnSpec < Rails::Generator::GeneratedAttribute
  delegate :limit, :to => :column
  
  def initialize(name, type, limit=nil)
    @name, @type = name, type.to_sym
    @column = ActiveRecord::ConnectionAdapters::Column.new(name, nil, @type)
    @column.instance_variable_set("@limit", limit) if limit
  end
end
