# -*- coding: utf-8 -*-
# 有効期間があるモデルを抽象化するモジュール。
module ActiveRecord
  module Acts
    module Periodic
      def self.included(base)
        base.extend(ClassMethods)
      end

      module ClassMethods
        # <tt>acts_as_periodic</tt> を呼び出すことで、
        # <tt>find</tt>、<tt>count</tt> および <tt>with_scope</tt> を置き換える。
        # そのモデルでは2つのカラム
        # [<tt>inception</tt>] 開始日付
        # [<tt>expiry<tt>] 終了日付
        # を基準にして、この期間外で置き換えられたメソッドを呼び出した場合には
        # 透過的に存在しないものとして扱う。
        def acts_as_periodic(options = {})
          return if periodic?
          class << self
            alias_method_chain :validate_find_options, :period
            alias_method :underlying_find, :find
            alias_method :count_with_period, :count
            alias_method :clobbering_with_scope, :with_scope
          end
          include InstanceMethods
        end

        def periodic?
          self.included_modules.include? ActiveRecord::Acts::Periodic::InstanceMethods
        end

        protected

        VALID_FIND_OPTIONS_WITH_PERIOD =
          class << ::ActiveRecord::Base
            VALID_FIND_OPTIONS
          end + [:with_period]

        def validate_find_options_with_period(options)
          options.assert_valid_keys(VALID_FIND_OPTIONS_WITH_PERIOD)
        end
      end

      module InstanceMethods
        def self.included(base)
          base.extend(ClassMethods)
        end

        module ClassMethods
          def find(*args)
            options = args.extract_options!
            call_underlying_find = lambda { underlying_find(*(args << options)) }
            if options[:with_period]
              return call_underlying_find.call
            else
              return with_period_scope { call_underlying_find.call }
            end
          end

          def find_with_period(*args)
            return underlying_find(*(args << args.extract_options!.merge(:with_period => true)))
          end

          def count(*args)
            return with_period_scope { count_with_period(*args) }
          end

          protected

          def with_period_scope
            now = Time.now.utc.strftime("%Y%m%d")
            periodic_cond = "#{self.table_name}.inception <= '#{now}' AND '#{now}' <= #{self.table_name}.expiry"
            clobbering_with_scope({:find => {:conditions => periodic_cond}}) do
              yield
            end
          end
        end
      end
    end
  end
end
