# -*- coding: utf-8 -*-
# app/controllers/product_controller.rb のクラス ProductController をテストする。

require File.dirname(__FILE__) + '/../test_helper'
require 'product_controller'

# Re-raise errors caught by the controller.
class ProductController; def rescue_action(e) raise e end; end

class ProductControllerTest < Test::Unit::TestCase
  fixtures(:products, :displays, :items, :expenses, :expense_details, :matter_reports,
           :search_conditions, :search_condition_items)

  def setup
    @controller = ProductController.new
    @request    = ActionController::TestRequest.new
    @response   = ActionController::TestResponse.new

    # set logged in
    @request.session[:user_id] = User.find_by_login("tesla")

    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # Web Service のモックオブジェクトを作成する
  # Web Service API の個々のアクションはそれぞれのテストで設定する
  def setup_webservice_mock(options = {})
    hash = {
      :issueable => false, :configurable => false, :cacheable => false, :candidateable => false,
      :possible_responses => [], :buttons => [], :items => [],
      :origins => [], :routes => [], :assignments => [] }
    hash.update(options)
    @clients = { :action => mock('action'), :probe => mock('probe') }
    ActionWebService::Client::Soap.expects(:new).
      with(WorkflowProbeApi, anything).at_most_once.returns(@clients[:probe])
    ActionWebService::Client::Soap.expects(:new).
      with(WorkflowActionApi, anything).at_most_once.returns(@clients[:action])
    @config = mock()
    @clients[:probe].expects(:configuration).with(anything, anything, anything).returns(@config)
    @config.expects(:issue_id).at_most_once.returns(hash[:issue_id])
    @config.expects(:items).at_most_once.returns(hash[:items])
    @config.expects(:issueable?).at_most(2).returns(hash[:issueable])
    @config.expects(:buttons).at_most(1).returns(hash[:buttons])
    @config.expects(:origins).at_most(1).returns(hash[:origins])
    @config.expects(:routes).at_most(1).returns(hash[:routes])
    @config.expects(:assignments).at_most(1).returns(hash[:assignments])
    @config.expects(:cacheable).at_most(1).returns(hash[:cacheable])
    @config.expects(:candidateable).at_most(1).returns(hash[:candidateable])
    @config.expects(:possible_responses).at_most(10).returns(hash[:possible_responses])
    unless hash[:issueable]
      @config.expects(:configurable?).at_most(10).returns(hash[:configurable])
    end
    @config.expects(:notice).at_most_once.returns('notice message')
  end

  def workflow_button_mock(action, label = action)
    WorkflowStruct::Button.new(:key => action, :value => label)
  end

  # ProductController#index が機能ごとの既定の一覧画面へリダイレクトすることをテストする。
  def test_index
    get :index, :product_id => 8
    assert_response :redirect
    assert_redirected_to :action => "list"
  end

  # ProductController#list が機能ごとの一覧画面を出力することをテストする。
  def test_list
    get :list, :product_id => 8
    assert_response :success
    assert_template "list"
    assert_select "div"
    assert_equal displays(:display_to_list_private_1), assigns(:display) # the default list
  end

  # ProductController#list が XMLHTTPRequest の POST メソッドに応じて
  # 機能ごとの一覧画面を出力することをテストする。
  def test_xhr_list
    xhr :post, :list, :product_id => 8
    assert_response :success
    assert_template "list"
    assert_select "div"
    assert_equal displays(:display_to_list_private_1), assigns(:display) # the default list
  end

  # 再表示ボタンでfragmentのpopup部分が消えることをテストする。
  def test_list_reload
    post :list, :product_id => 8, :dummy_button => "", :fragment => "m=8.per_20,detail=show.id_1", :per => 20
    assert_response :success
    assert_template "list"
    assert_select "input.fragment_setter" do
      assert_select "[name=?]", "sync"
      assert_select "[value=?]", "m=8.per_20"
    end

    xhr :post, :list, :product_id => 8, :dummy_button => "", :fragment => "m=8.per_20,detail=show.id_1"
    assert_response :success
    assert_template "list"
    assert_select "input.fragment_setter" do
      assert_select "[name=?]", "sync"
      assert_select "[value=?]", "m=8.per_20"
    end
  end

  # ProductController#show が機能ごとの詳細画面を出力することをテストする。
  def test_show
    setup_webservice_mock(:issueable => true)
    get :show, :product_id => 8, :id => 1
    assert_response :success
    assert_template "show"
    assert_select "div"
    assert_equal displays(:show8), assigns(:display)
  end

  # ProductController#show_only が機能ごとの詳細画面(関連文書用)を出力することをテストする。
  def test_show_only
    setup_webservice_mock(:issueable => true)
    get :show_only, :product_id => 8, :id => 1
    assert_response :success
    assert_template "show_only"
    assert_select "div"
    assert_equal displays(:show8), assigns(:display)
  end

  # ProductController#destroy が機能ごとの削除確認画面を出力することをテストする。
  def test_get_destroy
    get :destroy, :product_id => 8, :id => 1
    assert_response :success
    assert_template "confirm_destroy"
    assert_select "div"
    assert !assigns(:display).kind_of?(Display)
  end

  # ProductController#destroy が XMLHTTPRequest の GET メソッドに応じて
  # 機能ごとの削除確認画面を出力することをテストする。
  def test_xhr_get_destroy
    xhr :get, :destroy, :product_id => 8, :id => 1
    assert_response :success
    assert_template "confirm_destroy"
    assert_select "div"
    assert !assigns(:display).kind_of?(Display)
  end

  # ProductController#destroy が POST メソッドに応じて
  # 機能ごとの削除完了画面へリダイレクトされることをテストする。
  def test_post_destroy
    post :destroy, :product_id => 8, :id => 1, :confirm_destroy => true
    assert_response :redirect
    assert_redirected_to :action => :list
  end

  # ProductController#destroy が XMLHTTPRequest の POST メソッドに応じて
  # 機能ごとの削除完了画面を出力することをテストする。
  def test_xhr_post_destroy
    xhr :post, :destroy, :product_id => 8, :id => 1
    assert_response :success
    assert_equal "text/javascript", @response.content_type
  end

  # ProductController#destroy の validation のテスト
  # 親レコードの validates_isolated_1 の条件
  def test_xhr_post_destroy_with_validates_isolated_of__ng
    @request.session[:user_id] = User.find_by_login("admin")
    xhr :post, :destroy, :product_id => 40, :id => 1
    assert_response :success
    assert_equal "text/javascript", @response.content_type
    assert Expense.exists?(1) # 削除されないので存在する
  end

  # ProductController#destroy でビジネスロジックによって削除を阻まれるテスト
  def test_xhr_post_destroy_but_alert
    @request.session[:user_id] = User.find_by_login("admin")
    xhr :post, :destroy, :product_id => 1000, :id => 1
    assert_response :success
    assert_template "show"
  end

  # ProductController#edit が機能ごとの編集画面を出力することをテストする。
  def test_edit
    setup_webservice_mock(:issueable => true,
                          :items => [WorkflowStruct::Item.new(:id => 832, :validates_presence => true)],
                          :origins => [WorkflowStruct::Department.new(:id => 1, :name => 'aaa')])
    get :edit, :product_id => 8, :id => 1
    assert_response :success
    assert_template "form"
    assert_select "div"
    assert_equal displays(:edit8), assigns(:display)
    assigns(:items).each do |item|
      if item.id == 832
        assert item.validates_presence, "item.id = #{item.id}"
      else
        assert item.writable?, "item.id = #{item.id}"
      end
    end
  end

  # 子レコードを削除するときの validation のテスト
  def test_edit_with_validates_isolated_of__ng
    @request.session[:user_id] = User.find_by_login("admin")
    post :edit, :product_id => 40, :id => 1, :it => expenses(:e201801a),
      :old_details => [1, 2].to_json, :update => '更新',
      :details_1 => expense_details(:e201801a0),
      :details_2 => expense_details(:e201801a1),
      :details_3 => expense_details(:e201801a2),
      :order_details => ["details_1", "details_2", "details_3"].to_json

    # assert_response :redirect
    assert ExpenseDetail.exists?(1)
    assert ExpenseDetail.exists?(2)
    assert ExpenseDetail.exists?(3)
  end

  # 子レコードを削除するときの validation のテスト
  def test_edit_with_validates_isolated_of__ok
    item = Item.find(1036)
    item.validates_isolated = false
    assert item.save(false)
    @request.session[:user_id] = User.find_by_login("admin")
    assert_nothing_raised do
      post :edit, :product_id => 40, :id => 2, :it => expenses(:e201801b),
      :old_details => [4, 5, 6, 7].to_json, :update => '更新',
      :details_4 => expense_details(:e201801b0),
      :details_5 => expense_details(:e201801b1),
      :details_6 => expense_details(:e201801b2),
      :details_7 => expense_details(:e201801b3),
      :order_details => ["details_4", "details_5", "details_6", "details_7"].to_json
    end

    assert_response :redirect
    assert !ExpenseDetail.exists?(4)
    assert !ExpenseDetail.exists?(5)
    assert !ExpenseDetail.exists?(6)
    assert !ExpenseDetail.exists?(7)
  end

  # ProductController#new が機能ごとの新規作成画面を出力することをテストする。
  def test_new
    get :new, :product_id => 8
    assert_response :success
    assert_template "form"
    assert_select "div"
    assert_equal displays(:new8), assigns(:display)
  end

  # ProductController#new で ItemPolymorphic のインスタンスとともに複写できることをテストする。
  def test_new__copy_with_item_polymorphic
    @request.session[:user_id] = User.find_by_login("admin")
    CacheEachRequest.clear
    get :new, :product_id => 888, :id => 1
    assert_response :success
    assert_template "form"
    assert_select "input[name=picker_lump88833]" do
      assert_select "[value=?]", "ミッドガルドシステムズ"
    end
  end

  # ProductController#select_document が関連文書の選択画面を出力することをテストする。
  def test_select_document
    xhr :post, :select_document, :product_id => 8, :id => 1, :type => "Matter"
    assert_response :success
    assert_template "select_document"
    assert_select "table"
  end

  # ProductController#create_document が関連文書の新規作成画面を出力することをテストする。
  def test_create_document
    get :create_document, :product_id => 8, :id => 2
    assert_response :success
    assert_template "create_document"
    assert_select "div"
  end

end
