# -*- coding: utf-8 -*-
require "action_controller"
require "action_view"
require "digest/sha1"

# ユーザー管理に関するモジュール。
module UserSystem

  protected

  # authenticate_user filter. add
  #
  #   before_filter :authenticate_user
  #
  def authenticate_user
    return true if authenticated_user?{ password_user? }
    if false # disable returning to clicked url because menu may not revive
      session[:return_to] = request.request_uri
    end
    access_denied
    return false
  end

  # overwrite if you want to have special behavior in case the user is not authorized
  # to access the current operation.
  # the default action is to redirect to the login screen
  # example use :
  # a popup window might just close itself for instance
  def access_denied
    redirect_url = { :controller => "/user", :action => "login" }
    if request.xhr?
      render :update do |page|
        page.redirect_to url_for(redirect_url)
      end
    else
      redirect_to redirect_url
    end
  end

  def redirect_back_or_default
    if session[:return_to].nil?
      redirect_to CONFIG[:default_url]
    else
      redirect_to session[:return_to]
      session[:return_to] = nil
    end
  end

  def reset_login_session
    session[:user_id] = nil
    session[:login_time] = nil
    @current_user = nil
    CacheEachRequest.clear
  end

  def init_session_expiry
    session[:expires_at] = Clock.now + CONFIG[:session_times_out_in]
  end

  def session_has_timed_out?
    Time.now > session[:expires_at]
  end

  def init_login_session(user)
    User.current = user
    session[:user_id] = user.id
    session[:login_time] = Time.now
    session[:remote_ip] = request.remote_ip
    session[:user_agent] = request.env["HTTP_USER_AGENT"]
    Person.transaction do
      person = user.person
      last_language = person.last_language
      current_language = GetText.locale.language[0, 2]
      if last_language.blank? || last_language != current_language
        person.last_language = current_language
        person.save!
      end
    end
    init_session_expiry
    # ユーザ別の絞り込みの設定をセッションに読み込む
    defaults = DefaultNarrowing.find(:all, :conditions => {:user_id => user.id})
    defaults.each do |d|
      keys = d.value_keys
      keys.each do |k|
        session[k.to_sym] = d.restore(k)
      end
      d.display_narrowings.each do |dn|
        display = dn.display
        if display
          session[:display_narrowing] ||= {}
          session[:display_narrowing][display.id] ||= {}
          session[:display_narrowing][display.id][:id]   = d.narrowing_id
          session[:display_narrowing][display.id][:text] = d.narrowing_text
        end
      end
    end
  end

  def authenticated_user?
    # skip auth in rails side
    unless request.env["REMOTE_USER"].blank?
      return skip_authentication(request.env["REMOTE_USER"])
    end

    if session[:user_agent] && session[:user_agent] != request.env["HTTP_USER_AGENT"]
      logger.info "::: User-Agent changed, resetting session."
      user = User.find_by_id(session[:user_id])
      create_login_history(user, LoginHistory::RESULT_USER_AGENT_CHANGED)
      reset_login_session
      flash[:message] = s_("rfw|flash|message|Invalid request detected. Please login again.")
      return false
    end

    if session[:expires_at]
      if session_has_timed_out?
        logger.info "::: Session has expired, resetting session."
        user = User.find_by_id(session[:user_id])
        create_login_history(user, LoginHistory::RESULT_SESSION_TIMED_OUT)
        reset_login_session
        flash[:message] = s_("rfw|flash|message|Session has expired. Please login again.")
        return false
      end
      logger.info "::: Session has not expired. Reinitializing."
      init_session_expiry
    else
      logger.info "::: Session expiry not initialized"
      init_session_expiry
    end

    if session[:user_id]
      @current_user = User.find_by_id(session[:user_id])
      User.current = @current_user
      return false if @current_user.nil?
      if @current_user.password_expire?
        path_parameters = request.path_parameters
        unless path_parameters["controller"] == "user" && path_parameters["action"] == "change_password"
          redirect_to :controller => "user", :action => "change_password"
        end
      end
      return true
    end

    return true if block_given? && yield

    # Everything failed
    return false
  end

  def password_user?
    # If not, is the user being authenticated by a token (created by signup/forgot password actions)?
    return false if not params['user']
    id = params['user']['id']
    key = params['key']
    if id and key
      @current_user = User.authenticate_by_token(id, key)
      session[:user_id] = @current_user ? @current_user.id : nil
      return true if not @current_user.nil?
    end
    return false
  end

  def skip_authentication(login)
    @current_user = User.find_by_login(login)
    if session[:user_id] && session[:user_id] != @current_user.id
      logger.info "invalid session session[:user_id]=#{session[:user_id]} != #{@current_user.id}"
      reset_login_session
    end
    return false if @current_user.nil?
    init_login_session(@current_user)
    flash["notice"] = s_("rfw|flash|notice|Login skipped")
    create_login_history(@current_user, LoginHistory::RESULT_LOGIN_SKIPPED)
    return true
  end

  def create_login_history(user, result, program_type=::LoginHistory::PROGRAM_TYPE_WEB, login=nil, password=nil)
    history = LoginHistory.new
    if params["user"]
      login ||= params["user"]["login"]
      password ||= params["user"]["password"]
    end
    history.login = login if login
    history.password = password if password
    history.result = result
    history.remote_address = request.remote_ip
    history.program_type = program_type
    if user
      history.user_id = user.id
      history.domain_id = user.domain_id
    end
    history.save!
  end
end

::ActionController::Base.__send__(:include, ::UserSystem)
