# テーブルを描画する PDF のクラス。
class SimpleTablePDF < SimplePDF
  def initialize(options={})
    super

    # default value
    @page_order = 0  # 0 => "Down, then over", else => "Over, then down"
    @bottom_margin = 25.0 # mm

    # retrieve options
    options.each do |key,val|
      case(key)
      when :table
        @table = val # an instance of SimpleTable
      when :page_order
        @page_order = val # 0 => "Down, then over", else => "Over, then down"
      when :bottom_margin
        @bottom_margin = val # mm
      end
    end

    SetAutoPageBreak(false)
  end

  attr_accessor :table

  def output
    draw_table
    super
  end

  private

  attr_reader :bottom_margin

  def drawable_height
    return page_height - top_margin - bottom_margin
  end

  # construct primitive text
  def construct_primitive(txt, options)
    if txt.empty?
      return false
    else
      txt = txt.gsub(')', '\\)')
      txt.gsub!('(', '\\(')
      txt.gsub!('\\', '\\\\\\')
      s = sprintf('BT %.2f %.2f Td (%s) Tj ET', options[:offset_x], options[:offset_y], txt)
      s = "q #{options[:text_color]} #{s} Q" if options[:color_flag]
      return s
    end
  end

  def draw_string(txt)
    s = construct_primitive(txt,
                            :color_flag => @ColorFlag,
                            :text_color => @TextColor,
                            :offset_x => @x*@k,
                            :offset_y => (@h-@y-@FontSize)*@k)
    out(s) if s
  end

  # calculate cell width and height
  def form_table
    @table.each_index do |r|
      @table[r].each_index do |c|
        # set font size
        SetFontSize(@table[r][c].style.font_size)

        # calculate cell width
        max_line = @table[r][c].text(:array => true).max do |line1, line2|
          w1 = GetStringWidth(encode_text(line1))
          w2 = GetStringWidth(encode_text(line2))
          w1 <=> w2
        end
        w = @table[r][c].style.margin[:left] +
          GetStringWidth(encode_text(max_line)) +
          @table[r][c].style.margin[:right]
        if @table.column_width[c] == nil || w > @table.column_width[c]
          @table.column_width[c] = w
        end

        # calculate cell height
        h = @table[r][c].style.margin[:top] +
          @table[r][c].line_num * @table[r][c].style.line_height +
          @table[r][c].style.margin[:bottom]
        if @table.row_height[r] == nil || h > @table.row_height[r]
          @table.row_height[r] = h
        end
      end
    end
  end

  def draw_table
    form_table # an uniformed big table
    ts = @table.split(drawable_width, drawable_height)

    # set page order
    ts = ts.transpose if @page_order == 0  # "Down, then over"
                                      # else "Over, then down"
    ts.flatten!

    # draw table
    AddPage()
    x = GetX()
    y = GetY()
    ts.each_index do |p|
      @PageLinks[@page] = Array.new
      ts[p].each_index do |r|
        ts[p][r].each_index do |c|
          # set style
          style = ts[p][r][c].style

          # set font size
          SetFontSize(style.font_size)

          # set color
          set_draw_color(style.color[:border].to_a)
          set_text_color(style.color[:text].to_a)
          set_fill_color(style.color[:fill].to_a) if style.color[:fill]

          # set link
          if ts[p][r][c].link && !(ts[p][r][c].link.to_s.empty?)
            Link(x, y,
                 ts[p].column_width[c], ts[p].row_height[r],
                 ts[p][r][c].link.to_s)
          end

          # draw border
          if style.border_width.common_width == nil
            if style.color[:fill]
              Rect(x, y,
                   ts[p].column_width[c],
                   ts[p].row_height[r],
                   'F')
            end

            if style.border_width[:top] != 0
              SetLineWidth(style.border_width[:top])
              Line(x, y,
                   x + ts[p].column_width[c], y)
            end

            if style.border_width[:bottom] != 0
              SetLineWidth(style.border_width[:bottom])
              Line(x, y + ts[p].row_height[r],
                   x + ts[p].column_width[c], y + ts[p].row_height[r])
            end

            if style.border_width[:left] != 0
              SetLineWidth(style.border_width[:left])
              Line(x, y,
                   x, y + ts[p].row_height[r])
            end

            if style.border_width[:right] != 0
              SetLineWidth(style.border_width[:right])
              Line(x + ts[p].column_width[c], y,
                   x + ts[p].column_width[c], y + ts[p].row_height[r])
            end
          elsif style.border_width.common_width != 0
            SetLineWidth(style.border_width.common_width)
            Rect(x, y,
                 ts[p].column_width[c],
                 ts[p].row_height[r],
                 style.color[:fill] ? 'DF' : 'D')
          end


          # draw cell contents
          line_num = 0
          ts[p][r][c].each_line do |line|
            # determine drawing location
            case(style.align)
            when :left
              SetXY(x + style.margin[:left],
                    y + style.margin[:top] + line_num * style.line_height)
            when :center
              SetXY(x + style.margin[:left] + 
                    (ts[p].column_width[c] -
                     (style.margin[:left] + style.margin[:right]) -
                     GetStringWidth(encode_text(line))) / 2,
                    y + style.margin[:top] + line_num * style.line_height)
            when :right
              SetXY(x + ts[p].column_width[c] -
                    (style.margin[:right] + GetStringWidth(encode_text(line))),
                    y + style.margin[:top] + line_num * style.line_height)
            end

            # draw the contents
            draw_string(encode_text(line))

            line_num += 1
          end

          x += ts[p].column_width[c]
        end

        x = left_margin
        y += ts[p].row_height[r]
      end

      if p < ts.size - 1
        AddPage()
        SetXY(x = left_margin, y = top_margin)
      end
    end
  end
end
