# 一覧テーブルのモデル。
class SimpleTable
  attr_reader :row_size, :column_size
  attr_accessor :row_height, :column_width

  def initialize(row=1, col=1, row_height = 0.0, column_width = 0.0)
    @row_size = row
    @column_size = col

    @table = Array.new(@row_size){ Array.new(@column_size){ SimpleCell.new("") } }
    @row_height = Array.new(@row_size){ row_height }
    @row_line_nums = Array.new(@row_size){ 1 }
    @column_width = Array.new(@column_size){ column_width }
  end

  # 行 <em>row</em>、および真ならば列 <em>col</em> にアクセスする。
  def [](row, col=nil)
    return col ? @table[row][col] : @table[row]
  end

  # 行および列に代入する。
  def []=(row, val_or_col, val=nil)
    if val
      @table[row][val_or_col] = val.dup
    else
      @table[row] = val_or_col.dup
    end
  end

  def each_index(&block)
    @table.each_index(&block)
  end

  def set_contents
    @table.each_index do |r|
      @table[r].each_index do |c|
        @table[r][c] = yield(r, c)
      end
    end
  end

  # split a big table into several pages
  #
  # +------------+----------------+
  # |            |                |
  # |  ts[0][0]  |  ts[0][1]      |
  # |            |                |
  # |------------+----------------|
  # |            |                |
  # |  ts[1][0]  |  ts[1][1]      |
  # |            |                |
  # |            |                |
  # |------------+----------------|
  # |            |                |
  # |  ts[2][0]  |  ts[2][1]      |
  # |            |                |
  # +------------+----------------+
  #
  def split(page_width, page_height)
    ts = split_horizontally(page_height)
    ts.collect!{ |t| t.split_vertically(page_width) }

    return ts
  end

  # split this table horizontally into several tables
  #
  # +------------+
  # |            |
  # |  tv[0]     |
  # |            |
  # |------------+
  # |            |
  # |  tv[1]     |
  # |            |
  # |            |
  # |------------+
  # |            |
  # |  tv[2]     |
  # |            |
  # +------------+
  #
  def split_horizontally(page_height)
    tv = []
    h = 0 # current height

    rows = @table.dup
    stacked_rows = []
    row_height = @row_height.dup
    stacked_row_height = []

    proc = lambda do
      t = SimpleTable.new(stacked_rows.size, @column_size)
      t.row_height = stacked_row_height
      t.column_width = @column_width.dup
      t.each_index {|r| t[r] = stacked_rows[r]}
      tv << t
    end

    while row = rows.shift
      dh = row_height.shift

      if (h + dh) < page_height
        stacked_row_height << dh
        stacked_rows << row
        h += dh
      else
        need_tail_row = false
        tail_row = Array.new(@column_size)
        tail_row_height = 0
        head_row = Array.new(@column_size)
        head_row_height = 0

        @column_size.times do |c|
          style    = row[c].style
          line_num = row[c].line_num

          if line_num < 1
            tail_style = style.dup
            tail_style.margin[:bottom] =  0
            tail_row[c] = SimpleCell.new("", style.dup)

            head_style = style.dup
            head_style.margin[:top] = 0
            head_row[c] = SimpleCell.new("", style.dup)
            ddh = 0
          else
            text = row[c].text(:array => true)
            dht = style.margin[:top]
            dhb = style.margin[:top] + style.line_height * line_num

            if (h + dht) >= page_height
              tail_style = style.dup
              tail_style.margin[:bottom] = 0
              tail_row[c] = SimpleCell.new("", tail_style)

              head_style = style.dup
              head_style.margin[:top] = 0 # ToDo: need to set correct value
              head_row[c] = SimpleCell.new(text, head_style)
              ddh = 0
            elsif (h + dhb) >= page_height
              need_tail_row = true
              ln = 0
              dhc = style.margin[:top] + style.line_height

              while (h + dhc) < page_height
                dhc += style.line_height
                ln += 1
              end

              tail_style = style.dup
              tail_style.margin[:bottom] = 0
              tail_row[c] = SimpleCell.new(text[0...ln], tail_style)

              head_style = style.dup
              head_style.margin[:top] = 0 # ToDo: need to set correct value
              head_row[c] = SimpleCell.new(text[ln...line_num], head_style)
              ddh = 0 # ToDo: need to set
            else
              need_tail_row = true
              tail_style = style.dup
              tail_style.margin[:bottom] = 0
              tail_row[c] = SimpleCell.new(text, tail_style)

              head_style = style.dup
              head_style.margin[:top] = 0 # ToDo: need to set correct value
              head_row[c] = SimpleCell.new("", head_style)

              ddh = 0 # ToDo: need to set
            end
          end
        end

        if need_tail_row
          tail_height = 0
          tail_row.each do |cell|
            # calculate height
            h = cell.style.margin[:top] +
              cell.line_num * cell.style.line_height +
              cell.style.margin[:bottom]
            tail_height = h if h > tail_height

            # erase bottom border line
            cell.style.border_width[:bottom] = 0
          end
          stacked_row_height << tail_height
          stacked_rows << tail_row
        end

        # copy the clipped table
        proc.call

        h = 0
        stacked_row_height = []
        stacked_rows = []

        head_height = 0
        head_row.each do |cell|
          # calculate height
          h = cell.style.margin[:top] +
            cell.line_num * cell.style.line_height +
            cell.style.margin[:bottom]
          head_height = h if h > head_height

          # erase top border line
          cell.style.border_width[:top] = 0
        end
        row_height.unshift(head_height)
        rows.unshift(head_row)
      end
    end

    # tail rows
    proc.call

    return tv
  end

  # split this table vertically into several tables
  #
  # +---------+-----------+--------+
  # |         |           |        |
  # |  th[0]  |   th[1]   |  th[2] |
  # |         |           |        |
  # +---------+-----------+------- +
  #
  def split_vertically(page_width)
    th = []
    w = 0 # current width
    start_col = 0
    proc = lambda do |c|
      t = SimpleTable.new(@row_size, c - start_col)
      t.row_height = @row_height.dup
      t.column_width = @column_width[start_col...c]
      t.each_index {|r| t[r] = @table[r][start_col...c]}
      th << t
    end

    @column_size.times do |c|
      w += @column_width[c]
      if w >= page_width
        # copy the clipped table
        proc.call(c)
        w = @column_width[c]
        start_col = c
      end
    end
    # tail columns
    proc.call(@column_size)

    return th
  end
end
