# -*- coding: utf-8 -*-
module Rfw
  module Date

    # 日付の書式を扱う。
    module Format

      # 日付の表現として妥当であればその日付を返す。
      # さもなくば false を返す。
      def self.valid_date(x)
        x = x.to_s
        if /\A(\d{4})\d{4}\z/ =~ x || /\A(\d{4})([\/-])\d{1,2}\2\d{1,2}\z/ =~ x
          year = $1.to_i
          return false unless (1..9999).include?(year)
          begin
            return x.to_s.to_date
          rescue ArgumentError
            return false
          end
        end
        return false
      end

    end

    # 項目の日付に関する振る舞いを扱う。
    # 以下のメッセージに respond_to? であることが前提:
    # * calendar?
    # * calendar_format
    # * validates_year_month_day?
    module Item

      def date?
        validates_year_month_day?
      end

      def fit_into_internal_date_format(x)
        with_date_format(x) do |date|
          return x unless respond_to?(:model_class)
          return x unless model_class.respond_to?(:columns_hash)
          column = model_class.columns_hash[column_name]
          return x if column.nil?
          format = (column.limit.nil? || App::INTERNAL_DATE_FORMAT.length <= column.limit) ? App::INTERNAL_DATE_FORMAT : "yyyymmdd"
          CustomFormatTranslator.format_date(date, format)
        end
      end

      def fit_into_external_date_format(x)
        with_date_format(x) do |date|
          CustomFormatTranslator.format_date(date, calendar? ? calendar_format : App::EXTERNAL_DATE_FORMAT)
        end
      end

      def fit_into_internal_date_sub(x)
        case App::INTERNAL_DATE_FORMAT
        when /\//
          x.to_s.gsub("-", "/")
        when /-/
          x.to_s.gsub("/", "-")
        else
          x.to_s.gsub(/[\/\-]/, "")
        end
      end

      private

      def with_date_format(x, &block)
        return x if x.blank?
        case x
        when ::Date
          date = x
        else
          date = Format.valid_date(x)
          return x unless date
        end
        block.call(date)
      end

    end
  end
end
