# -*- coding: utf-8 -*-
# URL の fragment を構成するためのハッシュクラス。
class FragmentHash < ActiveSupport::OrderedHash
  def initialize(fragment)
    @motion = {}
    @default_params = {}
    if fragment
      fragment.scan(/([a-z0-9]+)=([._a-z0-9]+)/) do |key, value|
        values = value.split(/\./) # restSeparator in application.js
        @motion[key] = values.shift
        h = self[key] = {}.with_indifferent_access
        values.each do |v|
          k, v = v.split(/_/, 2)
          h[k] = v
        end
      end
    end
  end

  def [](key)
    return super(key.to_s) || {}.with_indifferent_access
  end

  DUMMY_MOTION = "0".freeze

  # <em>view</em> にともなる motion を返す。
  def motion(view)
    return @motion.fetch(view, DUMMY_MOTION)
  end

  # 既定のパラメータを参照する。
  def default_params(view)
    @default_params[view] ||= {}
    return @default_params[view]
  end

  # fragment を文字列として返す。
  def to_fragment
    parts = []
    self.each do |view, hash|
      next if /\A\[/ =~ view
      a = ["#{view}=#{motion(view)}"]
      hash.keys.sort.each do |key|
        a << "#{key}_#{hash[key]}"
      end
      parts << a.join(".")
    end
    return parts.join(",")
  end

  module ControllerHelper
    private

    # use in before_filter
    def convert_fragment
      @fragment_hash = FragmentHash.new(params[:fragment])
    end

    # sync params[key] with fragment
    def sync_fragment(view, key, default)
      params[key.to_sym] ||= fetch_fragment(view, key, default)
      update_fragment(view, key, default)
    end

    # fetch value from fragment
    def fetch_fragment(view, key, default)
      return @fragment_hash[view][key] || default
    end

    # set value to fragment or remove when default
    def update_fragment(view, key, default, value=params[key])
      raise "invalid view #{view.inspect}" unless /\A[a-z0-9]+\z/ =~ view
      raise "invalid key #{key.inspect}" unless /\A[a-z0-9]+\z/ =~ key.to_s
      raise "invalid value #{value.inspect}" unless /\A[a-z0-9]*\z/ =~ value.to_s

      @fragment_hash.default_params(view)[key] = default

      hash = @fragment_hash[view]
      if value.blank? || value.to_s == default.to_s
        hash.delete(key)
      else
        hash[key] = value
      end
      @fragment_hash[view] = hash
      new_fragment = @fragment_hash.to_fragment
      return if params[:fragment].nil?
      if new_fragment != params[:fragment]
        session[:fragment] = [:sync, new_fragment]
      end
    end

    # fragmentのsub viewの部分を削除する。
    def delete_sub_view_fragment(current_view)
      found = false
      @fragment_hash.keys.each do |view|
        if found
          @fragment_hash.delete(view)
        elsif current_view == view
          found = true
        end
      end
      if found
        session[:fragment] = [:sync, @fragment_hash.to_fragment]
      end
    end
  end

  module UrlHelper
    def fragment_for(options)
      options = options.with_indifferent_access
      menu = options[:menu]
      product = options[:product]
      it = options[:document]
      action = options[:action] || "show"
      if menu.blank?
        if product
          # product -> menu は一意に決まらない可能性がある
          menu = Menu.find(:first,
                           :conditions => ["product_id = ?", product.id],
                           :order => "id ASC")
        else
          return ""
        end
      end
      return "m=#{menu.id},detail=#{action}.id_#{it.id}"
    end
  end
end
