# -*- coding: utf-8 -*-
########################################################################
#
# Struct
# 

# Design Web サービスで一般的に利用するデータ構造を定義する。
module DesignStruct

  class LanguageLabel < ActionWebService::Struct
    member :language_id, :int
    member :language_name, :string
  end

  # リクエストおよびレスポンスの単位として共通で利用する。
  class Book < ActionWebService::Struct
    member :client_identifier_x, :int
    member :client_identifier_y, :int
    member :client_identifier_z, :string
    member :message_code, :string
    member :message_content, :string

    # 処理が成功したことを表す。
    def succeed
      self.message_code = "success"
      return self
    end
  end
end

# ログイン時に利用するデータ構造を定義する。
module LoginStruct
  # ログイン要求を構成する。
  class Proposal < ActionWebService::Struct
    member :parameter_0, :string
    member :parameter_1, :string
  end

  # ログイン要求に対するレスポンスを構成する。
  class Approval < DesignStruct::Book
  end
end

# ログアウト時に利用するデータ構造を定義する。
module LogoutStruct
  # ログアウト要求を構成する。
  class Proposal < DesignStruct::Book
  end

  # ログアウト要求に対するレスポンスを構成する。
  class Approval < DesignStruct::Book
  end
end

# MenuService で利用するデータ構造を定義する。
module MenuStruct
  
  class MenuName < ActionWebService::Struct#追加
    member :language_id, :int
    member :menu_name, :string#改修
  end
  
  class MenuItem < ActionWebService::Struct
    member :menu_id, :int#追加
    member :product_id, :int
    (1..3).each do |i|
      member "menu_hierarchy_level_#{i}", :int
    end
    member :menu_code, :string#改修
    member :menu_names, [MenuName]
   #member :product_name, :string削除
    member :menu_type, :boolean
  end

  class MainSheet < ActionWebService::Struct
    member :menu_items, [MenuItem]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
    member :language_labels, [DesignStruct::LanguageLabel]#追加 
  end
end

# LanguageService で利用するデータ構造を定義する。
module ConfigLanguageStruct

  class LanguageConfItem < ActionWebService::Struct
    member :language_id, :int
    member :language_suffix, :string
    member :language_name, :string
  end

  class MainSheet < ActionWebService::Struct
    member :language_conf_items, [LanguageConfItem]

    def push(x) #:nodoc:
      unless x.is_a?(LanguageConfItem)
        x = x.to_language_conf_item
      end
      self.language_conf_items ||= []
      self.language_conf_items << x
    end
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
  end

  def to_language_conf_item #:nodoc:
    item = LanguageConfItem.new
    item.language_id = id
    item.language_suffix = code
    item.language_name = name_(Language.default_code)
    return item
  end
end

# MailService で利用するデータ構造を定義する。
module MailFormatStruct

  class MailConfItem < ActionWebService::Struct
    member :mail_id, :int
    member :mail_mode, :string
    member :from_format, :string
    member :recipients_format, :string
    member :subject_format, :string
    member :body_format, :string

    TABLE = {
      :mail_id           => :id,
      :mail_mode         => :mail_mode,
      :from_format       => :from_format,
      :recipients_format => :recipients_format,
      :subject_format    => :subject_format,
      :body_format       => :body_format,
    }
  end

  class MainSheet < ActionWebService::Struct
    member :mail_conf_items, [MailConfItem]

    def push(x) #:nodoc:
      unless x.is_a?(MailConfItem)
        x = x.to_mail_conf_item
      end
      self.mail_conf_items ||= []
      self.mail_conf_items << x
    end
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
  end

  def to_mail_conf_item #:nodoc:
    item = MailConfItem.new
    MailConfItem::TABLE.each {|k,v| item.__send__("#{k}=", __send__(v))}
    return item
  end
end

# MenuService で利用するデータ構造を定義する。
module ProductStruct

  class ProductName < ActionWebService::Struct
    member :language_id, :int
    member :product_name, :string
  end

  class ProductItem < ActionWebService::Struct
    h = {
      :workflow_enabled => :boolean,
      :workflow => :string,
      :workflow_body_method => :string,
      :mail => :boolean,
      :mail_skip_auth => :boolean,
      :document => :boolean,
      :document_name_method => :string,
      :document_number_method => :string,
      :document_content_method => :string,
      :attachment => :boolean,
      :search => :boolean,
      :csv => :boolean,
      :initial_roleable_type => :string,
      :scope_roleable_type => :string,
    }
    REGULAR_MEMBER = h

    member :product_id, :int
    member :product_code, :string
    member :product_model, :string
    member :product_detail_model, :string#追加
    member :table_name, :string#追加
    member :product_detail_table, :string#追加
    member :product_parent_succeeded, :boolean#追加
    member :product_type, :string    #追加
    member :product_names, [ProductName]
    REGULAR_MEMBER.each do |k,v|
        member k, v
    end
  end

  class MainSheet < ActionWebService::Struct
    member :product_items, [ProductItem]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
    member :language_labels, [DesignStruct::LanguageLabel]
  end
end

# DisplayService で利用するデータ構造を定義する。
module DisplayStruct

  # MainSheet の構成要素のクラス。
  class DisplayName < ActionWebService::Struct
    member :language_id, :int
    member :display_name, :string
  end

  # MainSheet の構成要素のクラス。
  class DisplayItem < ActionWebService::Struct
    h = {
       :code => :string,
       :position => :int,
       :enabled => :boolean,
       :button_new => :boolean,
       :button_edit => :boolean,
       :button_delete => :boolean,
       :button_copy => :boolean,
       #追加:logic_mid => :boolean,
       #追加:logic_befor => :boolean,
       #追加:logic_after => :boolean,
       :logic_path => :string,
       :logic_pre => :string,
       :logic_post => :string,
    }
    REGULAR_MEMBER = h

    member :display_id, :int
    member :display_names, [DisplayName]
    member :type, :string
    REGULAR_MEMBER.each do |k,v|
      member k, v
    end
  end

  class MainSheet < ActionWebService::Struct
    member :display_items, [DisplayItem]
  end

  # Item の構成要素のクラス。
  class ItemName < ActionWebService::Struct
    member :language_id, :int
    member :item_name, :string
  end

  # ItemSheet の構成要素のクラス。
  class Item < ActionWebService::Struct
    member :item_id, :int
    member :display_id, :int     # FYI
    member :model_name, :string#追加   
    member :column_name, :string # FYI
    member :item_names, [ItemName]
    member :item_code, :string

    # <em>item</em> を変更する。
    def label_item
      return if !item_id || item_id == 0
      hash = Language.to_h
      item = ::Item.find(item_id)
      item.code = item_code
      item_names.each do |n|
        item.__send__("name_#{hash[n.language_id]}=", n.item_name)
      end
      item.save!
    end
  end

  class ItemSheet < ActionWebService::Struct
    member :items, [Item]
  end

  # NewSheet および EditSheet の構成要素のクラス。
  class EntryItem < ActionWebService::Struct
    h = {
      :code => :string,
      :layout  => :int,
      :width => :string,
      :align => :string,
      :decorator => :string,
      :decorator_parameter => :string,
      :position => :string,
      :control => :string,
      :input_type => :string,
      :field_size => :string,
      :input_parameter => :string,
      :input_initializer => :string,
      :input_initial_value => :string,
      :column_name => :string,
      :method_chain => :string,
      :adapter_name => :string,
    }
    ::Item::VALIDATES_COLUMN_NAME.each do |name|
      h[name] = :boolean
    end
    ::Item::VALIDATES_COLUMN_PARAMETER_COUNT.each do |name, count|
      (1..count).each do |i|
        h["#{name}_#{i}"] = :string
      end
    end
    REGULAR_MEMBER = h
    
    member :model_name, :string#追加
    member :display_id, :int#追加
    member :item_id, :int
    member :type, :string
    REGULAR_MEMBER.each do |k,v|
      member k, v
    end
  end

  class NewSheet < ActionWebService::Struct
    member :new_items, [EntryItem]
  end

  class EditSheet < ActionWebService::Struct
    member :edit_items, [EntryItem]
  end

  # ShowSheet の構成要素のクラス。
  class ShowItem < ActionWebService::Struct
    h = {
      :code => :string,
      :layout => :string,
      :width => :string,
      :align => :string,
      :input_parameter => :string,
      :decorator => :string,
      :decorator_parameter => :string,
      :linked => :boolean,
      :link_parameter => :string,
      :control => :string,
      :column_name => :string,
      :method_chain => :string,
      :adapter_name => :string,
    }
    REGULAR_MEMBER = h
    
    member :model_name, :string#追加
    member :display_id, :int#追加
    member :item_id, :int
    member :type, :string
    REGULAR_MEMBER.each do |k,v|
      member k,v
    end
  end

  class ShowSheet < ActionWebService::Struct
    member :show_items, [ShowItem]
  end

  # ListSheet の構成要素のクラス。
  class ListItem < ActionWebService::Struct
    h = {
      :code => :string,
      :layout => :string,
      :width => :string,
      :align => :string,
      :input_parameter => :string,
      :decorator => :string,
      :decorator_parameter => :string,
      :direction => :string,
      :condition_value => :string,
      :condition_pattern => :string,
      :selected => :boolean,
      :ordered => :boolean,
      :control => :string,
      :search => :boolean,
      :column_name => :string,
      :method_chain => :string,
      :adapter_name => :string,
    }
    REGULAR_MEMBER = h

    member :model_name, :string#追加
    member :display_id, :int#追加
    member :item_id, :int
    member :type, :string
    REGULAR_MEMBER.each do |k,v|
      member k, v
    end
  end

  class ListSheet < ActionWebService::Struct
    member :list_items, [ListItem]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
    member :item_sheet, ItemSheet
    member :new_sheet,  NewSheet
    member :edit_sheet, EditSheet
    member :show_sheet, ShowSheet
    member :list_sheet, ListSheet
    member :language_labels, [DesignStruct::LanguageLabel]
    member :product_id, :int
    member :product_code, :string
    member :product_name, :string
  end
end

# MessageService で利用するデータ構造を定義する。
module MessageStruct

  class MessageValue < ActionWebService::Struct
    member :language_id, :int
    member :message_value, :string
  end

  class Message < ActionWebService::Struct
    member :message_msgid, :string
    member :message_values, [MessageValue]
  end

  class MainSheet < ActionWebService::Struct
    member :messages, [Message]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
    member :language_labels, [DesignStruct::LanguageLabel]
  end
end

# CommonLabelService で利用するデータ構造を定義する。
module CommonLabelStruct

  class CommonValue < ActionWebService::Struct
    member :language_id, :int
    member :common_value, :string
  end

  class CommonLabel < ActionWebService::Struct
    member :type, :string
    member :msgid, :string
    member :msgid_plural, :string
    member :msgid_n, :string
    member :common_values, [CommonValue]
  end

  class MainSheet < ActionWebService::Struct
    member :common_labels, [CommonLabel]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
    member :language_labels, [DesignStruct::LanguageLabel]
  end
end

# ApplicationTableService で利用するデータ構造を定義する。
module ApplicationTableStruct

  class Column < ActionWebService::Struct
    member :name, :string
    member :type, :string
    member :length, :int
    member :relation_table_name, :string
    member :relation_item_name, :string
    member :polymorphic, :boolean
    member :dependent, :string
    member :application_key, :boolean
  end

  class Table < ActionWebService::Struct
    member :name, :string
    member :columns, [Column]
  end

  class MainSheet < ActionWebService::Struct
    member :tables, [Table]
  end

  class Book < DesignStruct::Book
    member :main_sheet, MainSheet
  end
end

########################################################################
# 
# API
#

# Design Web サービスで共通に利用する API を定義する。
# 現時点で利用する予定の API は以下の通り:
# - ログイン
class DesignApi < ActionWebService::API::Base
  api_method :login, :expects => [LoginStruct::Proposal], :returns => [LoginStruct::Approval]
  api_method :logout, :expects => [LogoutStruct::Proposal], :returns => [LogoutStruct::Approval]
end

# Design Web サービスで共通に利用する API を実装する。
class DesignService < ActionWebService::Base
  web_service_api DesignApi

  # <em>proposal</em> に従ってログイン処理を行う。
  def login(proposal)
    p0, p1 = proposal.parameter_0, proposal.parameter_1
    if p0.empty? || p1.empty? # failed
      return false
    end
    if user = User.authenticate(p0, p1, true)
      if ticket = DesignTicket.give(user)
        approval = LoginStruct::Approval.new(:client_identifier_x => ticket.user_id,
                                             :client_identifier_y => ticket.domain_id,
                                             :client_identifier_z => ticket.token)
        return approval
      end
    end
    # failed
    return false
  end

  # <em>proposal</em> に従ってログアウト処理を行う。
  def logout(proposal)
    if ticket = DesignTicket.take(:user_id   => proposal.client_identifier_x,
                                  :domain_id => proposal.client_identifier_y,
                                  :token     => proposal.client_identifier_z,
                                  :invalidate => true)
      approval = LogoutStruct::Approval.new(:client_identifier_x => ticket.user_id,
                                            :client_identifier_y => ticket.domain_id,
                                            :client_identifier_z => ticket.token)
      return approval
    end
    return false
  end
end

Language.class_eval do
  def self.language_labels #:nodoc:
    result = []
    Language.all.each do |language|
      label = DesignStruct::LanguageLabel.new(:language_id => language.id,
                                              :language_name => language.code)
      result << label
    end
    return result
  end
end

class DesignSubService < ActionWebService::Base
  before_invocation :authenticate

  def replace_with(book, klass, prefix, hash) #:nodoc:
    ids = []
    items = book.main_sheet.__send__ "#{prefix}_items"
    items.each do |item|
      item_id = item.__send__("#{hash[:id]}")
      if item_id && item_id != 0
        x = klass.find(item_id)
        # update existing one.
        hash.each do |k,v|
          unless k == :id
            x.__send__("#{k}=", item.__send__(v))
          end
        end
        x.save!
        ids << item_id
      else
        # added newly one.
        spec = {}
        hash.each {|k,v| spec[k] = item.__send__(v)}
        spec[:domain_id] = book.client_identifier_y if klass.column_names.include?('domain_id')
        spec[:name_po] = 0 if klass.column_names.include?('name_po')
        newbie = klass.create!(spec)
        ids << newbie.id
      end
    end
    existing_ids = klass.find(:all, :order => "id").map(&:id)
    klass.delete(existing_ids - ids)
    return ids
  end

  private

  def authenticate(name, args) #:nodoc:
    u, d, t = args[0].client_identifier_x, args[0].client_identifier_y, args[0].client_identifier_z
    user = User.find_by_id(u)
    return [false, "no such user"] unless user
    return [false, "user is not admin"] unless user.admin?
    unless DesignTicket.valid?(:user_id   => u,
                               :domain_id => d,
                               :token     => t)
      return [false, "no such ticket"]
    end
    User.current = User.find(u)
  end
end

# Design Web サービスで多言語設定を扱う API を定義する。
class LanguageApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [ConfigLanguageStruct::Book], :returns => [ConfigLanguageStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [ConfigLanguageStruct::Book], :returns => [ConfigLanguageStruct::Book]
end

# Design Web サービスで多言語設定を扱う API を実装する。
class LanguageService < DesignSubService
  web_service_api LanguageApi

  # languages テーブルの内容をクライアントに送る。
  def import(book)
    return Language.to_book.succeed
  end

  # <em>book</em> として与えられたデータで languages テーブルの内容を置き換える。
  def export(book)
    hash = {
      'id' => 'language_id',
      'code' => 'language_suffix',
      "name_#{Language.default_code}" => 'language_name'
    }.with_indifferent_access
    ids = replace_with(book, Language, :language_conf, hash)
    return Language.to_book(ids).succeed
  end
end

# Design Web サービスでフレームワークの多言語対応のための翻訳を扱う API を定義する。
class CommonLabelApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [CommonLabelStruct::Book], :returns => [CommonLabelStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [CommonLabelStruct::Book], :returns => [CommonLabelStruct::Book]
end

PoMessage.class_eval do
  def self.to_book #:nodoc:
    hash = Language.to_h.invert
    result = CommonLabelStruct::Book.new
    result.language_labels = Language.language_labels
    result.main_sheet = CommonLabelStruct::MainSheet.new
    result.main_sheet.common_labels = []
    # all messages but of msgctxt "Application|"
    find(:all, :conditions => ["msgctxt <> ?", "Application|"], :order => "id").each do |po_message|
      label = CommonLabelStruct::CommonLabel.new
      label.type = po_message.attributes["type"]
      [:msgid, :msgid_plural, :msgid_n].each do |k|
        label.__send__("#{k}=", po_message.__send__(k))
      end
      label.common_values = []
      # po_translations -> common_values
      PoTranslation.find(:all, :conditions => {:po_message_id => po_message.id}, :order => "id").each do |po_translation|
        type = po_translation.attributes["type"]
        if type =~ /PoTranslation(.+)$/
          code = $1.downcase
          if id = hash[code]
            common_value = CommonLabelStruct::CommonValue.new(:language_id => id,
                                                              :common_value => po_translation.msgstr)
            label.common_values << common_value
          else
            raise Language::LanguageNotFound, "code \"#{code}\" not found"
          end
        else
          raise Language::LanguageNotFound, "malformed type #{type}"
        end
      end
      result.main_sheet.common_labels << label
    end
    return result
  end
end

# Design Web サービスでフレームワークの多言語対応のための翻訳を扱う API を実装する。
class CommonLabelService < DesignSubService
  web_service_api CommonLabelApi

  # フレームワークに関する po_messages および po_translations の情報を返す。
  def import(book)
    return PoMessage.to_book.succeed
  end

  # <em>book</em> として与えられるデータで po_messages および po_translations を置き換える。
  def export(book)
    hash = Language.to_h
    mids = []
    tids = []
    book.main_sheet.common_labels.each do |common_label|
      message_class = common_label.type.constantize
      msgid         = common_label.msgid
      msgid_plural  = common_label.msgid_plural
      msgid_n       = common_label.msgid_n
      po_message = message_class.find_by_msgid(msgid)
      unless (po_message &&
              po_message.msgid_plural == msgid_plural &&
              po_message.msgid_n      == msgid_n)
        # added newly msgid.
        po_message = message_class.create!(:domain_id    => book.client_identifier_y,
                                           :msgctxt      => "",
                                           :msgid        => msgid,
                                           :msgid_plural => msgid_plural,
                                           :msgid_n      => msgid_n)
      end
      mids << po_message.id
      # update po_messages.
      common_label.common_values.each do |value|
        klass = "PoTranslation#{hash[value.language_id].to_s.camelize}".constantize
        if po_translation = klass.find_by_po_message_id(po_message.id)
          # update the existing msgstr.
          po_translation.msgstr = value.common_value
          po_translation.save!
        else
          # added newly msgstr.
          po_translation = klass.create!(:domain_id => book.client_identifier_y,
                                         :po_message_id => po_message.id,
                                         :msgstr => value.common_value)
        end
        tids << po_translation.id
      end
    end
    emids = PoMessage.find(:all, :conditions => ["msgctxt <> ?", "Application|"], :order => "id").map(&:id)
    PoMessage.delete(emids - mids)
    etids = PoTranslation.find(:all, :conditions => {:po_message_id => emids}, :order => "id").map(&:id)
    PoTranslation.delete(etids - tids)
    return PoMessage.to_book.succeed
  end
end

# Design Web サービスでアプリケーションの多言語対応のための翻訳を扱う API を定義する。
# po_messages および po_translations を対象にする。
class MessageApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [MessageStruct::Book], :returns => [MessageStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [MessageStruct::Book], :returns => [MessageStruct::Book]
end

PoMessageSingular.class_eval do
  def self.to_book #:nodoc:
    hash = Language.to_h.invert
    result = MessageStruct::Book.new
    result.language_labels = Language.language_labels
    result.main_sheet = MessageStruct::MainSheet.new
    result.main_sheet.messages = []
    # messages both of form singular and of msgctxt "Application|" only available for now
    find(:all, :conditions => {:msgctxt => "Application|"}, :order => "id").each do |po_message|
      message = MessageStruct::Message.new
      message.message_msgid = po_message.msgid
      message.message_values = []
      # po_translations -> message_values
      PoTranslation.find(:all, :conditions => {:po_message_id => po_message.id}, :order => :id).each do |po_translation|
        type = po_translation.attributes["type"]
        if type =~ /PoTranslation(.+)$/
          code = $1.downcase
          if id = hash[code]
            message_value = MessageStruct::MessageValue.new(:language_id => id,
                                                            :message_value => po_translation.msgstr)
            message.message_values << message_value
          else
            raise Language::LanguageNotFound, "code \"#{code}\" not found"
          end
        else
          raise Language::LanguageNotFound, "malformed type #{type}"
        end
      end
      result.main_sheet.messages << message
    end
    return result
  end
end

# Design Web サービスでアプリケーションの多言語対応のための翻訳を扱う API を実装する。
class MessageService < DesignSubService
  web_service_api MessageApi

  # アプリケーションに関する po_messages および po_translations の情報を返す。
  def import(book)
    return PoMessageSingular.to_book.succeed
  end

  # <em>book</em> として与えられるデータで po_messages および po_translations を置き換える。
  def export(book)
    hash = Language.to_h
    mids = []
    tids = []
    book.main_sheet.messages.each do |message|
      msgid = message.message_msgid
      po_message = PoMessageSingular.find_by_msgctxt_and_msgid("Application|", msgid)
      unless po_message
        # added newly msgid.
        po_message = PoMessageSingular.create!(:domain_id => book.client_identifier_y,
                                               :msgctxt => "Application|",
                                               :msgid => msgid)
      end
      mids << po_message.id
      # update po_messages.
      message.message_values.each do |value|
        klass = "PoTranslation#{hash[value.language_id].to_s.camelize}".constantize
        if po_translation = klass.find_by_po_message_id(po_message.id)
          # update the existing msgstr.
          po_translation.msgstr = value.message_value
          po_translation.save!
        else
          # added newly msgstr.
          po_translation = klass.create!(:domain_id => book.client_identifier_y,
                                         :po_message_id => po_message.id,
                                         :msgstr => value.message_value)
        end
        tids << po_translation.id
      end
    end
    emids = PoMessageSingular.find(:all, :conditions => {:msgctxt => "Application|"}, :order => "id").map(&:id)
    PoMessageSingular.delete(emids - mids)
    etids = PoTranslation.find(:all, :conditions => {:po_message_id => emids}, :order => "id").map(&:id)
    PoTranslation.delete(etids - tids)
    return PoMessageSingular.to_book.succeed
  end
end

Menu.class_eval do
  def self.to_menu_book(domain_id) #:nodoc:
    lang_code = Language.default_code
    book = MenuStruct::Book.new
    book.language_labels = Language.language_labels
    book.main_sheet = MenuStruct::MainSheet.new
    book.main_sheet.menu_items = []
    find(:all, :conditions => {:domain_id => domain_id}, :order => "lft").each do |menu|
      level = menu.ancestors.size
      if 1 <= level && level <= 3 # only of proper level
        menu_item = MenuStruct::MenuItem.new
        menu_item.menu_id = menu.id
        menu_item.product_id = menu.product_id
        menu_item.menu_code = menu.code
        #item.menu_type = product.is_a?(ProductPseudo)
        menu_item.menu_names = []
        Language.all.each do |lang|
          mn = MenuStruct::MenuName.new(:language_id => lang.id)
          mn.menu_name = menu.name_(lang.code)
          menu_item.menu_names << mn
        end
        (1..3).each do |i|
          menu_item.__send__("menu_hierarchy_level_#{i}=", (i == level) ? 1 : 0)
        end
        book.main_sheet.menu_items << menu_item
        #menu.save!
      end
    end
    return book
  end
end


# Design Web サービスでメニューを扱う API を定義する。
class MenuApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [MenuStruct::Book], :returns => [MenuStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [MenuStruct::Book], :returns => [MenuStruct::Book]
end

# Designメニューを扱う API を実装する。
class MenuService < DesignSubService
  web_service_api MenuApi

  class MissingMenuRoot < StandardError; end#改修
  class InvalidHierarchyLevel < StandardError; end

  # products テーブルの内容を Book にして返す。
  def import(book)
    domain_id = book.client_identifier_y
    unless Menu.root#改修
      Menu.create!(:domain_id => domain_id,#改修
                          :code => "menu_root_#{domain_id}",#改修
                          :name_po => 0)
    end
    return Menu.to_menu_book(domain_id).succeed#改修
  end

  # <em>book</em> として与えられるデータで products テーブルの内容を置き換える。
  def export(book)
    root = Menu.root
	domain_id = book.client_identifier_y
	hash = Language.to_h
	ids = []
	items = book.main_sheet.menu_items
	#menu_item = MenuStruct::MenuItem.new     
    #items.each do |menu_item|
    items.inject([false, false, false]) do |(x, y, z),menu_item|
	    menu_id = menu_item.menu_id        
       	proc = lambda do |menu|
		  menu.domain_id = domain_id
          menu.code = menu_item.menu_code
          product = Product.find_by_code(menu.code)
          unless product == nil
             menu.product_id = product.id
             menu.motion = product.motion
          end   
		  menu.save! if menu.new_record?
          menu_item.menu_names.each do |mn|
              language_id = mn.language_id
          	  menu.__send__("name_#{hash[language_id]}=", mn.menu_name)
          end
          menu.save!
          ids << menu.id
    	end
		if menu_id && menu_id != 0
        	# update the existing one
        	menu = Menu.find(menu_id)
        	proc.call(menu)
    	else
        	menu = Menu.new(:domain_id => book.client_identifier_y,
        	                :name_po => 0)
        	proc.call(menu)
    	end
    	
            if menu_item.menu_hierarchy_level_1 == 1
              if x
                 menu.move_to_right_of(x)
              else
                 menu.move_to_child_of(root)
              end
             [menu, false, false]
            elsif menu_item.menu_hierarchy_level_2 == 1
              if y
                 menu.move_to_right_of(y)
              else
                 menu.move_to_child_of(x)
              end
              [x, menu, false]
            elsif menu_item.menu_hierarchy_level_3 == 1
              if z
                 menu.move_to_right_of(z)
              else
                 menu.move_to_child_of(y)
              end
              [x, y, menu]
            else
              raise InvalidHierarchyLevel
            end
    end	
	eids = Menu.find(:all, 
	:conditions => 'parent_id != "null"',
	:order => "id").map(&:id)
    Menu.delete(eids - ids)
    return Menu.to_menu_book(domain_id).succeed
  end
end

# Design Web サービスで機能を扱う API を定義する。
class ProductApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [ProductStruct::Book], :returns => [ProductStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [ProductStruct::Book], :returns => [ProductStruct::Book]
end

Product.class_eval do
  def self.to_product_book(domain_id) #:nodoc:
    book = ProductStruct::Book.new
    book.language_labels = Language.language_labels
    book.main_sheet = ProductStruct::MainSheet.new
    book.main_sheet.product_items = []
    find(:all, :conditions => {:domain_id => domain_id}, :order => "id").each do |product|
      product_item = ProductStruct::ProductItem.new
      product_item.product_id = product.id
      product_item.product_code = product.code
      product_item.product_model = product.model_name
      product_item.product_detail_model = product.detail_model_name#追加
      product_item.table_name = product.table_name#追加
      product_item.product_detail_table = product.detail_table_name#追加
      product_item.product_parent_succeeded = product.parent_succeeded#追加
      product_item.product_type = product.type#追加
      product_item.product_names = []
      Language.all.each do |lang|
        pn = ProductStruct::ProductName.new(:language_id => lang.id)
        pn.product_name = product.name_(lang.code) 
        product_item.product_names << pn
      end
      ProductStruct::ProductItem::REGULAR_MEMBER.each do |k,v|
        product_item.__send__("#{k}=", product.__send__((v == :boolean) ? "#{k}?" : k))
      end
      book.main_sheet.product_items << product_item
    end
    return book
  end
end

# Design Web サービスで機能を扱う API を実装する。
class ProductService < DesignSubService
  web_service_api ProductApi

  # products テーブルの内容を返す。
  def import(book)
    domain_id = book.client_identifier_y
    unless Product.root
      ProductRoot.create!(:domain_id => domain_id,
                          :code => "product_root_#{domain_id}",
                          :name_po => 0)
    end
    return Product.to_product_book(domain_id).succeed
  end

  # <em>book</em> として与えられたデータで products テーブルの内容を置き換える。
  def export(book)
    domain_id = book.client_identifier_y
    hash = Language.to_h
    ids = []
    moids = []
    book.main_sheet.product_items.each do |product_item|
      product_id = product_item.product_id
      proc = lambda do |product|
        product.domain_id = domain_id
        product.code = product_item.product_code
        product.model_name = product_item.product_model
        product.detail_model_name = product_item.product_detail_model#追加
        product.table_name = product_item.table_name#追加
        product.detail_table_name = product_item.product_detail_table#追加
        product.parent_succeeded = product_item.product_parent_succeeded#追加
        product.type = product_item.product_type#追加
        ProductStruct::ProductItem::REGULAR_MEMBER.each do |k,|
          product.__send__("#{k}=", product_item.__send__(k))
        end
        #application_modelのデータ処理
        if product.type == "ProductDetailed"
          ApplicationTable.find(:all, :conditions => {:name => product.detail_table_name}, :order => "id").each do |application_detail_table|
            application_model = ApplicationModel.find(:first, :conditions => {:name => product.detail_model_name}, :order => "id")
            unless application_model
              application_model = ApplicationModel.new
            end
            application_model.application_table_id = application_detail_table.id
            application_model.name = product.detail_model_name
            application_model.save!
            if product.parent_succeeded == true
              TableColumn.find(:all, :conditions => {:table_id => application_detail_table.id, :application_key => true}, :order => "id").each do |table_column|
                application_model.parent_method_name = table_column.name.sub(/_id\z/ , "")
                application_model.save!
              end
            end
            moids << application_model.id
          end
        end
        ApplicationTable.find(:all, :conditions => {:name => product.table_name}, :order => "id").each do |application_table|
          application_model = ApplicationModel.find(:first, :conditions => {:name => product.model_name}, :order => "id")
          unless application_model
            application_model = ApplicationModel.new
          end
          application_model.application_table_id = application_table.id
          application_model.name = product.model_name
          application_model.save!
          if product.parent_succeeded == true
            TableColumn.find(:all, :conditions => {:table_id => application_table.id, :application_key => true}, :order => "id").each do |table_column|
              application_model.parent_method_name = table_column.name.sub(/_id\z/ , "")
              application_model.save!
            end
          end
          moids << application_model.id
        end
        product.save! if product.new_record?
        product_item.product_names.each do |pn|
          language_id = pn.language_id
          product.__send__("name_#{hash[language_id]}=", pn.product_name)
        end
        product.save!
        ids << product.id
      end
      if product_id && product_id != 0
        # update the existing one
        product = Product.find(product_id)
        proc.call(product)
      else
        #使用タイプがProductDetailedの場合
        if product_item.product_type == "ProductDetailed"
           #add a new one
           product = ProductDetailed.new(:name_po => 0)
           proc.call(product)
        else 
        # add a new one
        # TODO: other subclasses of Product should be available.
        product = ProductSingle.new(:name_po => 0)
        proc.call(product)
        end
        Menu.create!(:domain_id => book.client_identifier_y,
                     :code => product.code,
                     :product_id => product.id,
                     :parent_id => product.parent_id,
                     :motion => product.motion,
                     :name_po => 0)                   
      end
    end
    emoids = ApplicationModel.find(:all, :order => "id").map(&:id)
      ApplicationModel.find(:all, :conditions => {:id => (emoids - moids)},:order => "id").each do |drop_model|
        drop_model.drop_model
      end
    ApplicationModel.delete(emoids - moids)
    eids = Product.find(:all, :order => "id").map(&:id)
    Product.delete(eids - ids)
    mids = Menu.find(:all, :conditions => ['product_id IN (?)', eids - ids]).map(&:id)
    Menu.delete(mids)
    dids = Display.find(:all, :conditions => ['product_id IN (?)', eids - ids]).map(&:id)
    Display.delete(dids)
    itids = Item.find(:all, :conditions => ['display_id IN (?)', dids]).map(&:id)
    Item.delete(itids)
    
    #belongs_toのデータ処理
    BelongsTo.destroy_all
    Product.find(:all, :conditions => {:domain_id => 1}, :order => "id").each do |product|
      if product.type == "ProductDetailed"
        ApplicationModel.find(:all, :conditions => {:name => product.detail_model_name}, :order => "id").each do |application_model_id|
          ApplicationTable.find(:all, :conditions => {:name => product.detail_table_name}, :order => "id").each do |application_table|
            TableColumn.find(:all, :conditions => {:table_id => application_table.id, :application_key => true}, :order => "id").each do |table_column|
              if table_column.polymorphic == true && table_column.relation_table_name
                belongs_to = BelongsTo.new
                belongs_to.domain_id = product.domain_id
                belongs_to.application_table_id = table_column.table_id
                belongs_to.name = table_column.name.sub(/_id\z/ , "")
                belongs_to.polymorphic = table_column.polymorphic
                belongs_to.application_model_id = application_model_id.id
                if BelongsTo.find(:first,:donditions =>{:application_table_id => belongs_to.application_table_id, :name => belongs_to.name, :polymorphic => belongs_to.polymorphic, :application_model_id => belongs_to.application_model_id}) == nil
                  belongs_to.save!
                end
              end
              if table_column.polymorphic == false && table_column.relation_table_name
                if table_column.relation_table_name == product.table_name
                  belongs_to = BelongsTo.new
                  belongs_to.domain_id = product.domain_id
                  belongs_to.application_table_id = table_column.table_id
                  belongs_to.name = "header"
                  belongs_to.class_name = product.model_name
                  belongs_to.forein_key = table_column.name
                  belongs_to.polymorphic = table_column.polymorphic
                  belongs_to.application_model_id = application_model_id.id
                  if BelongsTo.find(:first,:conditions =>{:application_table_id => belongs_to.application_table_id, :name => belongs_to.name, :class_name => belongs_to.class_name, :forein_key => belongs_to.forein_key, :polymorphic => belongs_to.polymorphic, :application_model_id => belongs_to.application_model_id}) == nil
                    belongs_to.save!
                  end
                else
                  ApplicationTable.find(:all, :conditions => {:name => table_column.relation_table_name}, :order => "id").each do |application_table_b|
                    ApplicationModel.find(:all, :conditions => {:application_table_id => application_table_b.id}, :order => "id").each do |application_model|
                      belongs_to = BelongsTo.new
                      belongs_to.domain_id = product.domain_id
                      belongs_to.application_table_id = table_column.table_id
                      belongs_to.name = table_column.name.sub(/_id\z/ , "")
                      belongs_to.class_name = application_model.name
                      belongs_to.forein_key = table_column.name
                      belongs_to.polymorphic = table_column.polymorphic
                      belongs_to.application_model_id = application_model_id.id
                      if BelongsTo.find(:first,:conditions =>{:application_table_id => belongs_to.application_table_id, :name => belongs_to.name, :class_name => belongs_to.class_name, :forein_key => belongs_to.forein_key, :polymorphic => belongs_to.polymorphic, :application_model_id => belongs_to.application_model_id}) == nil
                        belongs_to.save!
                      end
                    end
                  end
                end
              end
            end
          end
        end
      end
      ApplicationModel.find(:all, :conditions => {:name => product.model_name}, :order => "id").each do |application_model_id|
        ApplicationTable.find(:all, :conditions => {:name => product.table_name}, :order => "id").each do |application_table|
          TableColumn.find(:all, :conditions => {:table_id => application_table.id, :application_key => true}, :order => "id").each do |table_column|
            if table_column.polymorphic == true && table_column.relation_table_name
              belongs_to = BelongsTo.new
              belongs_to.domain_id = product.domain_id
              belongs_to.application_table_id = table_column.table_id
              belongs_to.name = table_column.name.sub(/_id\z/ , "")
              belongs_to.polymorphic = table_column.polymorphic
              belongs_to.application_model_id = application_model_id.id
              if BelongsTo.find(:first,:conditions =>{:application_table_id => belongs_to.application_table_id, :name => belongs_to.name, :polymorphic => belongs_to.polymorphic, :application_model_id => belongs_to.application_model_id}) == nil
                belongs_to.save!
              end
            end
            if table_column.polymorphic == false && table_column.relation_table_name
              ApplicationTable.find(:all, :conditions => {:name => table_column.relation_table_name}, :order => "id").each do |application_table_b|
                ApplicationModel.find(:all, :conditions => {:application_table_id => application_table_b.id}, :order => "id").each do |application_model|
                  belongs_to = BelongsTo.new
                  belongs_to.domain_id = product.domain_id
                  belongs_to.application_table_id = table_column.table_id
                  belongs_to.name = table_column.name.sub(/_id\z/ , "")
                  belongs_to.class_name = application_model.name
                  belongs_to.forein_key = table_column.name
                  belongs_to.polymorphic = table_column.polymorphic
                  belongs_to.application_model_id = application_model_id.id
                  if BelongsTo.find(:first,:conditions =>{:application_table_id => belongs_to.application_table_id, :name => belongs_to.name, :class_name => belongs_to.class_name, :forein_key => belongs_to.forein_key, :polymorphic => belongs_to.polymorphic, :application_model_id => belongs_to.application_model_id}) == nil
                    belongs_to.save!
                  end
                end
              end
            end
          end
        end
      end
    end
    #has_manyのデータ処理
    HasMany.destroy_all
    Product.find(:all, :conditions => {:domain_id => 1}, :order => "id").each do |product|
      if product.type == "ProductDetailed"
        ApplicationModel.find(:all, :conditions => {:name => product.detail_model_name}, :order => "id").each do |application_model_id|
          TableColumn.find(:all, :conditions => ["relation_table_name like ?","%#{product.detail_table_name}%"], :order => "id").each do |table_column|
            if table_column.relation_table_name.split(/,/).index(product.detail_table_name) != nil
              if table_column.polymorphic == true
                ApplicationModel.find(:all, :conditions => {:application_table_id => table_column.table_id}, :order => "id").each do |application_model_h|
                  has_many = HasMany.new
                  has_many.domain_id = product.domain_id
                  has_many.application_model_id = application_model_id.id
                  has_many.name = application_model_h.name.underscore.pluralize
                  has_many.as_name = table_column.name.sub(/_id\z/ , "")
                  if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :as_name =>has_many.as_name}) == nil
                    has_many.save!
                  end
                end
              end
              if table_column.polymorphic == false
                ApplicationModel.find(:all, :conditions => {:application_table_id => table_column.table_id}, :order => "id").each do |application_model_h|
                  has_many = HasMany.new
                  has_many.domain_id = product.domain_id
                  has_many.application_model_id = application_model_id.id
                  if table_column.relation_item_name == nil
                    has_many.name = application_model_h.name.underscore.pluralize
                  else
                    has_many.name = table_column.relation_item_name
                  end
                  has_many.class_name = application_model_h.name
                  has_many.foreign_key = table_column.name
                  has_many.dependent = table_column.dependent
                  if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :class_name =>has_many.class_name, :foreign_key => has_many.foreign_key, :dependent =>has_many.dependent}) == nil
                    has_many.save!
                  end
                end
              end
            end
          end
        end
      end
      ApplicationModel.find(:all, :conditions => {:name => product.model_name}, :order => "id").each do |application_model_id|
        TableColumn.find(:all, :conditions => ["relation_table_name like ?","%#{product.table_name}%"], :order => "id").each do |table_column|
          if table_column.relation_table_name.split(/,/).index(product.table_name) != nil
            if table_column.polymorphic == true
              ApplicationModel.find(:all, :conditions => {:application_table_id => table_column.table_id}, :order => "id").each do |application_model_h|
                has_many = HasMany.new
                has_many.domain_id = product.domain_id
                has_many.application_model_id = application_model_id.id
                has_many.name = application_model_h.name.underscore.pluralize
                has_many.as_name = table_column.name.sub(/_id\z/ , "")
                if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :as_name =>has_many.as_name}) == nil
                  has_many.save!
                end 
              end
            end
            if table_column.polymorphic == false
              ApplicationModel.find(:all, :conditions => {:application_table_id => table_column.table_id}, :order => "id").each do |application_model_h| 
                BelongsTo.find(:all, :conditions => {:application_model_id => application_model_h.id}, :order => "id").each do |belongs_to|                   
                  if product.type == "ProductSingle"
                    has_many = HasMany.new
                    has_many.domain_id = product.domain_id
                    has_many.application_model_id = application_model_id.id
                    if table_column.relation_item_name == nil
                      has_many.name = application_model_h.name.underscore.pluralize
                    else
                      has_many.name = table_column.relation_item_name
                    end
                    has_many.class_name = application_model_h.name
                    has_many.foreign_key = belongs_to.name.+("_id")
                    has_many.dependent = table_column.dependent                  
                    if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :class_name =>has_many.class_name, :foreign_key => has_many.foreign_key, :dependent =>has_many.dependent}) == nil
                      has_many.save!
                    end
                  end
                  if product.type == "ProductDetailed"
                    if application_model_h.name == product.detail_model_name
                      has_many = HasMany.new
                      has_many.domain_id = product.domain_id
                      has_many.application_model_id = application_model_id.id
                      has_many.name = "details"
                      has_many.class_name = application_model_h.name
                      has_many.foreign_key = belongs_to.name.+("_id")
                      has_many.dependent = table_column.dependent
                      if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :class_name =>has_many.class_name, :foreign_key => has_many.foreign_key, :dependent =>has_many.dependent}) == nil
                        has_many.save!
                      end
                    else
                      has_many = HasMany.new
                      has_many.domain_id = product.domain_id
                      has_many.application_model_id = application_model_id.id
                      if table_column.relation_item_name == nil
                        has_many.name = application_model_h.name.underscore.pluralize
                      else
                        has_many.name = table_column.relation_item_name
                      end
                      has_many.class_name = application_model_h.name
                      has_many.foreign_key = belongs_to.name.+("_id")
                      has_many.dependent = table_column.dependent
                      if HasMany.find(:first, :conditions => {:application_model_id => has_many.application_model_id,:name => has_many.name, :class_name =>has_many.class_name, :foreign_key => has_many.foreign_key, :dependent =>has_many.dependent}) == nil
                        has_many.save!
                      end
                    end
                  end
                end
              end
            end
          end
        end
      end
    end
    ApplicationModel.find(:all, :order => "id").each do |create_models|
      create_models.create_model
    end
    return Product.to_product_book(domain_id).succeed
  end
end

# Design Web サービスで画面を扱う API を定義する。
# 対象:
# - 画面
# - 項目
class DisplayApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [DisplayStruct::Book], :returns => [DisplayStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [DisplayStruct::Book], :returns => [DisplayStruct::Book]
end

Item.class_eval do
  def to_item #:nodoc:
    item = DisplayStruct::Item.new
    item.item_id = id
    item.display_id = display_id
    item.model_name = model_name#追加
    item.column_name = column_name
    item.item_names = to_item_names
    item.item_code = code
    return item
  end

  def to_item_names #:nodoc:
    item_names = []
    Language.all.each do |lang|
      item_name = DisplayStruct::ItemName.new
      item_name.language_id = lang.id
      item_name.item_name = name_(lang.code)
      item_names << item_name
    end
    return item_names
  end

  [:entry, :show, :list].each do |x|
    class_eval <<-EOV, __FILE__, __LINE__+1
      def to_#{x}_item
        result = DisplayStruct::#{x.to_s.camelize}Item.new(:item_id => id)
        result.type = attributes["type"]
        result.display_id = display_id#追加
        #result.save!
        result.model_name = model_name#追加
        DisplayStruct::#{x.to_s.camelize}Item::REGULAR_MEMBER.each do |k,v|
          result.__send__("\#{k}=", __send__(v == :boolean ? "\#{k}?" : k))
        end
        return result
     end
    EOV
  end
end

Display.class_eval do
  def to_display_item #:nodoc:
    result = DisplayStruct::DisplayItem.new
    result.display_id = id
    result.type = attributes["type"]
    DisplayStruct::DisplayItem::REGULAR_MEMBER.each do |k,v|
      result.__send__("#{k}=", __send__(v == :boolean ? "#{k}?" : k))
    end
    result.display_names = []
    Language.all.each do |lang|
      dn = DisplayStruct::DisplayName.new(:language_id => lang.id)
      dn.display_name = name_(lang.code)
      result.display_names << dn
    end
    return result
  end
end

Product.class_eval do
  def to_item_sheet #:nodoc:
    sheet = DisplayStruct::ItemSheet.new
    sheet.items = []
    master_displays.each do |display|
      proper_items, pseudo_items = display.items.partition {|item| item.is_a?(ItemProper)}
      proper_items.sort_by(&:column_name).each do |item|
        sheet.items << item.to_item
      end
      pseudo_items.sort_by(&:adapter_name).each do |item|
        sheet.items << item.to_item
      end
    end
    return sheet
  end

  def to_book #:nodoc:
    lang_code = Language.default_code
    book = DisplayStruct::Book.new
    book.language_labels = Language.language_labels
    book.product_id = id
    book.product_code = code
    book.product_name = name_(lang_code)
    # construct main_sheet
    book.main_sheet = DisplayStruct::MainSheet.new
    displays_list =[]
    Display.find(:all , :conditions => {:product_id => id,
                                        :type => %w|DisplayToList DisplayToShow DisplayToEdit DisplayToNew|}, 
                                        :order => "id" ).each do |dsp|
      dsp_item = DisplayStruct::DisplayItem.new  
      dsp_item.display_id = dsp.id
      dsp_item.type = dsp.type
      DisplayStruct::DisplayItem::REGULAR_MEMBER.each do |k,v|
        dsp_item.__send__("#{k}=", dsp.__send__(v == :boolean ? "#{k}?" : k))
      end
      dsp_item.display_names = []
      Language.all.each do |lang|
        dn = DisplayStruct::DisplayName.new(:language_id => lang.id)
        dn.display_name = dsp.name_(lang.code)
        dsp_item.display_names << dn
      end
      displays_list << dsp_item
    end
    book.main_sheet.display_items = displays_list
    #book.main_sheet.display_items = displays.map(&:to_display_item)
    book.item_sheet = to_item_sheet
    # construct new_sheet, edit_sheet, show_sheet, and list_sheet
    {
      :new  => :entry,
      :edit => :entry,
      :show => :show,
      :list => :list,
    }.each do |k, v|
      sheet_class = "DisplayStruct::#{k.to_s.camelize}Sheet".constantize
      sheet = sheet_class.new
      items = []
      displays = []
      if displays = Display.find(:all, :conditions => {:type => "DisplayTo#{k.to_s.camelize}", :product_id => id})
        displays.each do |display| 
          display.items.each do |item|
            items << item.__send__("to_#{v}_item")
          end
        end
      end
      sheet.__send__("#{k}_items=", items)
      book.__send__("#{k}_sheet=", sheet)
    end
    return book
  end
end

# Design Web サービスで画面を扱う API を実装する。
class DisplayService < DesignSubService
  web_service_api DisplayApi

  class NoSuchDisplay < StandardError; end
  class NoSuchProduct < StandardError; end

  # <em>book.product_id</e> を id にもつ product に関連する画面情報を返す。
  def import(book)
    product = Product.find(book.product_id)
    return product.to_book.succeed
  end

  # <em>book</em> として与えられたデータで関連する画面情報を置き換える。
  def export(book)
    raise NoSuchProduct if !book.product_id || book.product_id == 0
    domain_id = book.client_identifier_y
    product = Product.find(book.product_id)
    # update or add displays
    #did = false
    dids = []
    book.main_sheet.display_items.each do |display_item|
      display_id = display_item.display_id
      proc = lambda do |display|
        DisplayStruct::DisplayItem::REGULAR_MEMBER.each do |k,|
          display.__send__("#{k}=", display_item.__send__(k))
        end
        #if display.new_record?
        # display.save!
        #  did = display.id   the latest display is expected to have new items as followed.
          # copy items from the existing display.
        #  source_display = display.class.find_by_product_id(book.product_id)
        # if source_display
        #    source_display.items.each {|item| item.private_copy(did)}
        #  end
        #end
        display_item.display_names.each do |n|
          lang = Language.find(n.language_id)
          display.__send__("name_#{lang.code}=", n.display_name)
          #"name_#{hash[language_id]}=" 
        end
        display.save! 
        dids << display.id
      end
      if !display_id || display_id == 0
        # add a new one
        display = "#{display_item.type}".constantize.new(:domain_id => domain_id,
                                                         :product_id => book.product_id,
                                                         :name_po => 0,
                                                         :code => display_item.code)
        display.save!                                                                                                  
        proc.call(display)
      else
        display = Display.find(display_id)
        # update the existing one
        type = display.attributes["type"]
        unless type == display_item.type
          raise NoSuchDisplay, "type mismatch: #{type} vs #{display_item.type}"
        end
        proc.call(display)
      end
    end
    # delete the inherent ones
    edids = Display.find(:all, :conditions => {
                           :product_id => book.product_id,
                           :type => %w|DisplayToList DisplayToShow DisplayToEdit DisplayToNew|,
                         }).map(&:id)
    Display.delete(edids - dids)
    itids = Item.find(:all, :conditions => ['display_id IN (?)', edids - dids]).map(&:id)
    Item.delete(itids)
    
    # update or add items
    iids = []
    [:new, :edit, :show, :list].each_with_index do |k,i|
      sheet = book.__send__("#{k}_sheet")
      sheet_items = sheet.__send__("#{k}_items")
      sheet_items.each do |sheet_item|
        item_id = sheet_item.item_id
        proc = lambda do |item|
          sheet_item.class::REGULAR_MEMBER.each do |m,|
            item.__send__("#{m}=", sheet_item.__send__(m))
          end
          item.display_id = sheet_item.display_id#追加
          item.model_name = sheet_item.model_name#追加
          item.save!
          iids << item.id
        end
        if !item_id || item_id == 0
          item_class = sheet_item.type.constantize
          display_id = sheet_item.display_id
          model_name = sheet_item.model_name
          #display_id = book.main_sheet.display_items[i].display_id
          po_message = PoMessageSingular.create!(:domain_id => domain_id,
                                                 :msgctxt   => "Application|",
                                                 :msgid     => "Item|#{sheet_item.code}")
          item = item_class.new(:domain_id  => domain_id,
                                :display_id => display_id,
                                :model_name => model_name,
                                :name_po    => po_message.id)
          proc.call(item)
        else
          item = Item.find(item_id)
          proc.call(item)
        end
      end
    end
    book.item_sheet.items.each(&:label_item)
    # delete the inherent ones
    eiids = Item.find(:all, :conditions => {:display_id => dids}).map(&:id)
    Item.delete(eiids - iids)
    return product.to_book.succeed
  end
end

# Design Web サービスで権限管理機能を扱う。
# 対象:
# - GrantOns
# - Permissions
# class GrantService < DesignSubService
#   web_service_api GrantApi
# end

# Design Web サービスでメール送信機能を扱う API を定義する。
# 対象:
# - メールテンプレート
class MailApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [MailFormatStruct::Book], :returns => [MailFormatStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [MailFormatStruct::Book], :returns => [MailFormatStruct::Book]
end

# Design Web サービスでメール送信機能を扱う API を実装する。
class MailService < DesignSubService
  web_service_api MailApi

  # mail_formats テーブルの内容をクライアントに送る。
  def import(book)
    return MailFormat.to_book.succeed
  end

  # <em>book</em> として与えられたデータで mail_formats テーブルの内容を置き換える。
  def export(book)
    ids = replace_with(book, MailFormat, :mail_conf, MailFormatStruct::MailConfItem::TABLE.invert)
    return MailFormat.to_book(ids).succeed
  end
end

{ Language   => ConfigLanguageStruct,
  MailFormat => MailFormatStruct,
}.each do |klass, struct_klass|
  klass.__send__ :include, struct_klass
  klass.class_eval <<-EOV, __FILE__, __LINE__+1
    def self.to_book(ids=nil)
      book = #{struct_klass}::Book.new
      book.main_sheet = #{struct_klass}::MainSheet.new
      if ids
        ids.each {|id| book.main_sheet.push(find(id))}
      else
        find(:all, :order => "id").each {|x| book.main_sheet.push(x)}
      end
      return book
    end
  EOV
end

# Design Web サービスでテーブル設定を扱う API を定義する。
class ApplicationTableApi < ActionWebService::API::Base
  # サーバ側からクライアント側へデータを送信する。
  api_method :import, :expects => [ApplicationTableStruct::Book], :returns => [ApplicationTableStruct::Book]
  # クライアント側からサーバ側へデータを送信する。
  api_method :export, :expects => [ApplicationTableStruct::Book], :returns => [ApplicationTableStruct::Book]
end

ApplicationTable.class_eval do
  def self.to_book
    book = ApplicationTableStruct::Book.new
    book.main_sheet = ApplicationTableStruct::MainSheet.new
    book.main_sheet.tables = []
    ApplicationTable.find(:all).each do |t|
      table = ApplicationTableStruct::Table.new
      table.name = t.name
      table.columns = []
      t.table_columns.each do |c|
        column = ApplicationTableStruct::Column.new
        column.name = c.name
        column.type = c.class.to_s
        column.length = c.length
        column.relation_table_name = c.relation_table_name
        column.relation_item_name = c.relation_item_name
        column.application_key = c.application_key
        column.polymorphic = c.polymorphic
        column.dependent = c.dependent
        table.columns << column
      end
      book.main_sheet.tables << table
    end
    return book
  end
end

# Design Web サービスでテーブル設定を扱う API を実装する。
class ApplicationTableService < DesignSubService
  web_service_api ApplicationTableApi

  # application_tables と table_columns の内容をクライアントへ送る。
  def import(book)
    return ApplicationTable.to_book.succeed
  end

  # <em>book</em> として与えられたデータでテーブルを置き換える。
  def export(book)
    # drop current tables
    ApplicationTable.destroy_all
    # generate new tables
    book.main_sheet.tables.each do |t|
      table = ApplicationTable.create!(:name => t.name)
      t.columns.each do |c|
        column_class = c.type.constantize
        column = column_class.new(:table_type => "ApplicationTable",
                                  :table_id => table.id)
        [:name, :length, :relation_table_name, :relation_item_name, :application_key, :polymorphic, :dependent].each {|k|
        unless c.__send__(k) == nil || c.__send__(k) == ""
        column.__send__("#{k}=", c.__send__(k))
        end
        }
        column.save!
      end
      table.create_table
    end
    return ApplicationTable.to_book.succeed
  end
end