# -*- coding: utf-8 -*-
require "rubygems"
require "active_record"
require "date"
require "iconv"
require "strscan"

unless /^U/ =~ $KCODE
  raise LoadError, "CustomValidations work with $KCODE=\"UTF8\" only."
end

# 検証部品を実装するモジュール。
module CustomValidations
  module SingletonValidateable
    CALLBACKS = %w[validate validate_on_create validate_on_update validate_on_destroy]
    def metaclass
      @metaclass ||= class << self; self; end
    end

    # analogue of activesupport-2.1.0/lib/active_support/callbacks.rb
    # if you want to learn more, see also activerecord-2.1.0/lib/active_record/validations.rb
    def run_callbacks(kind, options = {}, &block)
      if CALLBACKS.include?(kind.to_s)
        super
        metaclass.send("#{kind}_callback_chain").run(self, options, &block)
      else
        super
      end
    end

    def method_missing(sym, *args, &block)
      if /\Avalidates_/ =~ sym.to_s
        metaclass.__send__(sym, *args, &block)
      else
        super
      end
    end
  end

  module Validateable
    %w" save save! update_attribute new_record? destroy".each do |attr|
      define_method(attr) {}
    end

    def method_missing(symbol, *arguments)
      if /\A(.+)_before_type_cast\z/ =~ symbol.to_s
        __send__($1)
      else
        super
      end
    end

    module ClassMethods
      def human_attribute_name(attribute_name)
        ActiveRecord::Base.human_attribute_name(attribute_name)
      end
    end

    def self.included(base)
      base.extend(ClassMethods)
      base.__send__(:include, ActiveRecord::Validations)
    end
  end

  include GetText::Rails

  default_error_messages_of_custom = {
    :invalid_format_before_type_cast => ActiveRecord::Errors.default_error_messages[:invalid],
    :negative_integer => N_("rfw|message|error|%{fn} is not a positive integer"),
    :not_integer => N_("rfw|message|error|%{fn} is not an integer"),
    :negative_float => N_("rfw|message|error|%{fn} is not a positive decimal number"),
    :not_float => N_("rfw|message|error|%{fn} is not a decimal number"),
    :not_nonzero => N_("rfw|message|error|%{fn} must be nonzero number"),
    :not_upper_case => N_("rfw|message|error|%{fn} can contain upper case only"),
    :not_lower_case => N_("rfw|message|error|%{fn} con contain lower case only"),
    :not_alphabetic => N_("rfw|message|error|%{fn} can contain alphabetic characters only"),
    :not_alphanumeric => N_("rfw|message|error|%{fn} can contain alphabetic or numeric characters only"),
    :not_halfwidth_katakana => N_("rfw|message|error|%{fn} can contain halfwidth katakana only"),
    :not_fullwidth_katakana => N_("rfw|message|error|%{fn} can contain fullwidth katakana only"),
    :not_hour => N_("rfw|message|error|%{fn} is not hour"),
    :not_hour_minute => N_("rfw|message|error|%{fn} is not hour and minute"),
    :not_postal_code => N_("rfw|message|error|%{fn} is not postal code"),
    :not_phone_number => N_("rfw|message|error|%{fn} is not phone number"),
    :not_email => N_("rfw|message|error|%{fn} must be email"),
    :not_url => N_("rfw|message|error|%{fn} must be URL"),
    :not_fullwidth_chars => N_("rfw|message|error|%{fn} can contain fullwidth characters only"),
    :invalid_sequence => N_("rfw|message|error|%{fn} contains invalid sequence"),
    :not_inclusion_chars => N_("rfw|message|error|%{fn} can contain \"%{val}\" only"),
    :too_long_integer => N_("rfw|message|error|%{fn} is too long integer part (maximum is %d characters)"),
    :too_short_integer => N_("rfw|message|error|%{fn} is too short integer part (minimum is %d characters)"),
    :wrong_length_integer => N_("rfw|message|error|%{fn} is the wrong length of integer part (should be %d characters)"),
    :too_long_fractional => N_("rfw|message|error|%{fn} is too long fractional part (maximum is %d characters)"),
    :too_short_fractional => N_("rfw|message|error|%{fn} is too short fractional part (minimum is %d characters)"),
    :wrong_length_fractional => N_("rfw|message|error|%{fn} is the wrong length of fractional part (should be %d characters)"),
    :invalid_year => N_("rfw|message|error|%{fn} is invalid year"),
    :invalid_year_month => N_("rfw|message|error|%{fn} is invalid year or month"),
    :invalid_year_month_day => N_("rfw|message|error|%{fn} is invalid year, month, or day"),
    :too_large => N_("rfw|message|error|%{fn} is too large (maximum is %d)"),
    :too_small => N_("rfw|message|error|%{fn} is too small (minimum is %d)"),
    :wrong_number => N_("rfw|message|error|%{fn} is the wrong number (should be %d)"),
    :period => N_("rfw|message|error|%{fn} is not included in the period"),
    :later_date => N_("rfw|message|error|%{fn} must be later than %{val}"),
    :earlier_date => N_("rfw|message|error|%{fn} must be earlier than %{val}"),
    :row => N_("rfw|message|error|%{fn} is not included in %{column} of %{table}"),
    :not_isolated => N_("rfw|message|error|%{fn} is used in %{column} of %{table}"),
  }
  ActiveRecord::Errors.default_error_messages.update(default_error_messages_of_custom)
  default_error_messages_of_custom.each do |id, message|
    if /%d/ =~ message
      ActiveRecord::Errors.default_error_messages_d[
                                                    default_error_messages_of_custom[id]
                                                   ] = /#{Regexp.escape(default_error_messages_of_custom[id]).sub(/%d/, '(\d+)')}/
    elsif message['%{val}']
      ActiveRecord::Errors.default_error_messages_d[
                                                    default_error_messages_of_custom[id]
                                                   ] = /#{Regexp.escape(default_error_messages_of_custom[id]).sub('%\{val\}', '(.*)')}/
    end
  end

  def validates_format_before_type_cast_of(*attr_names) # :nodoc:
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:invalid_format_before_type_cast],
      :on => :save,
      :with => nil,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    raise(ArgumentError, "A regular expression must be supplied as the :with option of the configuration hash") unless configuration[:with].is_a?(Regexp)

    validates_each(attr_names, configuration) do |record, attr_name, value|
      record.errors.add(attr_name, configuration[:message]) unless record.send("#{attr_name}_before_type_cast").to_s =~ configuration[:with]
    end
  end

  def validates_non_negative_integer_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:negative_integer],
      :non_negative => true,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    attr_names.push configuration
    validates_integer_of(*attr_names)
  end

  def validates_integer_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_integer],
      :allow_nil => false,
      :non_negative => false,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    configuration[:with] = configuration[:non_negative] ? /\A\+?\d+\z/ : /\A[+-]?\d+\z/
    attr_names.push configuration
    validates_format_before_type_cast_of(*attr_names)
  end

  def validates_non_negative_float_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:negative_float],
      :non_negative => true,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    attr_names.push configuration
    validates_float_of(*attr_names)
  end

  def validates_float_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_float],
      :allow_nil => false,
      :non_negative => false,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    configuration[:with] = configuration[:non_negative] ? /\A\+?[0-9]+\.[0-9]+\z/ : /\A[+-]?[0-9]+\.[0-9]+\z/
    attr_names.push configuration
    validates_format_before_type_cast_of(*attr_names)
  end

  # deprecated method, use validates_nonzero_of instead.
  def validates_zero_of(*attr_names)
    if ENV["RAILS_ENV"] == "test"
      warn "deprecated method, use validates_nonzero_of instead."
    end
    validates_nonzero_of(*attr_names)
  end

  def validates_nonzero_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_nonzero],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless value.to_f.nonzero?
        record.errors.add(attr_name, configuration[:message])
      end
    end
  end

  def _validates_format_with(pattern, error_message_key, *attr_names) # :nodoc:
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[error_message_key],
      :with => pattern,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    attr_names.push configuration
    validates_format_before_type_cast_of(*attr_names)
  end

  halfwidth_ideographic_full_stop = [0xff61].pack('U')
  halfwidth_katakana_semi_voiced_sound_mark = [0xff9f].pack('U')
  katakana_hiragana_double_hyphen = [0x30a0].pack('U')
  katakana_letter_small_ro = [0x31ff].pack('U')
  [
   [:validates_upper_case_of, /\A[A-Z]+\z/],
   [:validates_lower_case_of, /\A[a-z]+\z/],
   [:validates_alphabetic_of, /\A[A-Za-z]+\z/],
   [:validates_alphanumeric_of, /\A[A-Za-z0-9]+\z/],
   [:validates_halfwidth_katakana_of, /\A[#{halfwidth_ideographic_full_stop}-#{halfwidth_katakana_semi_voiced_sound_mark}]+\z/],
   [:validates_fullwidth_katakana_of, /\A[#{katakana_hiragana_double_hyphen}-#{katakana_letter_small_ro}]+\z/],
   [:validates_hour_of, /\A(?:[01][0-9]|2[0-3])\z/],
   [:validates_hour_minute_of, /\A(?:[01][0-9]|2[0-3]):[0-5][0-9]\z/],
   [:validates_postal_code_of, /\A[0-9\-]+\z/],
   [:validates_phone_number_of, /\A[0-9\-()]+\z/],
   [:validates_email_of, /\A([^@\s]+)@((?:[\-A-Za-z0-9]+\.)+[A-Za-z]{2,})\z/],
   [:validates_url_of, /\Ahttps?:\/\/[!-~]+\z/],
  ].each do |method_name, pattern|
    key = method_name.to_s[/\Avalidates_(.+)_of/, 1]
    error_message_key = ("not_" + key).intern
    if ActiveRecord::Errors.default_error_messages[error_message_key].nil?
      raise "[bug] error message not found"
    end
    class_eval <<-"end_eval", __FILE__, __LINE__+1
      def #{method_name}(*attr_names)
        _validates_format_with(/#{pattern}/, #{error_message_key.inspect}, *attr_names)
      end
    end_eval
  end

  def validates_fullwidth_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_fullwidth_chars],
      :invalid_sequence_message => ActiveRecord::Errors.default_error_messages[:invalid_sequence],
      :blank_message => ActiveRecord::Errors.default_error_messages[:not_fullwidth_chars],
      :on => :save,
      :allow_nil => false,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    unless configuration[:allow_nil]
      # use validates_each instead of validates_presence_of because gettextize
      validates_each(attr_names, configuration) do |record, attr_name, value|
        if value.blank?
          record.errors.add attr_name, s_(configuration[:blank_message])
        end
      end
    end

    validates_each(attr_names, configuration) do |record, attr_name, value|
      begin
        cp932_string = Iconv.conv("cp932", "utf-8", value.to_s)
        scanner, char = StringScanner.new(cp932_string), /./s
        while c = scanner.scan(char)
          if c.length != 2
            record.errors.add attr_name, s_(configuration[:message])
            break
          end
        end
      rescue Iconv::IllegalSequence
        record.errors.add attr_name, s_(configuration[:invalid_sequence])
      end
    end
  end

  def validates_inclusion_chars_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_inclusion_chars],
      :on => :save,
      :allow_nil => false,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    chars = configuration[:chars]
    raise(ArgumentError, "Characters must be supplied as the :chars option of the configuration hash") if chars.blank?

    if chars.is_a?(Array)
      chars = chars.join("")
    end
    rchars = Regexp.escape(chars)
    configuration[:with] = /\A[#{rchars}]+\z/
    message = configuration[:message] % {:val => chars}

    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless record.send("#{attr_name}_before_type_cast").to_s =~ configuration[:with]
        record.errors.add(attr_name, message)
      end
    end
  end

  def validates_filtered_length_of(*attrs)
    options = {
      :too_long     => ActiveRecord::Errors.default_error_messages[:too_long],
      :too_short    => ActiveRecord::Errors.default_error_messages[:too_short],
      :wrong_length => ActiveRecord::Errors.default_error_messages[:wrong_length]
    }.merge(::ActiveRecord::Validations::ClassMethods::DEFAULT_VALIDATION_OPTIONS)
    options.update(attrs.pop.symbolize_keys) if attrs.last.is_a?(Hash)

    ## add filter option
    filter = options[:filter]
    unless filter.respond_to?(:call)
      raise ArgumentError, "Filter unspecified. Specify the :filter option."
    end

    # Ensure that one and only one range option is specified.
    range_options = ::ActiveRecord::Validations::ClassMethods::ALL_RANGE_OPTIONS & options.keys
    case range_options.size
    when 0
      raise ArgumentError, 'Range unspecified.  Specify the :within, :maximum, :minimum, or :is option.'
    when 1
      # Valid number of options; do nothing.
    else
      raise ArgumentError, 'Too many range options specified.  Choose only one.'
    end

    # Get range option and value.
    option = range_options.first
    option_value = options[range_options.first]

    case option
    when :within, :in
      raise ArgumentError, ":#{option} must be a Range" unless option_value.is_a?(Range)

      too_short = options[:too_short] % option_value.begin
      too_long  = options[:too_long]  % option_value.end

      validates_each(attrs, options) do |record, attr, value|
        ## change from "value.split(//).size" in validates_length_of
        ## to "filter.call(value)".
        if value.nil? or filter.call(value) < option_value.begin
          record.errors.add(attr, too_short)
        elsif filter.call(value) > option_value.end
          record.errors.add(attr, too_long)
        end
      end
    when :is, :minimum, :maximum
      raise ArgumentError, ":#{option} must be a nonnegative Integer" unless option_value.is_a?(Integer) and option_value >= 0

      # Declare different validations per option.
      validity_checks = { :is => "==", :minimum => ">=", :maximum => "<=" }
      message_options = { :is => :wrong_length, :minimum => :too_short, :maximum => :too_long }

      message = (options[:message] || options[message_options[option]]) % option_value

      validates_each(attrs, options) do |record, attr, value|
        ## change from "value.split(//).size" or "value.size" in validates_length_of
        ## to "filter.call(value)".
        begin
          unless !value.nil? and filter.call(Integer(value)).method(validity_checks[option])[option_value]
            record.errors.add(attr, message)
          end
        rescue ArgumentError
          record.errors.add(attr, message)
        end
      end
    end
  end

  def validates_integral_length_of(*attr_names)
    configuration = {
      :too_long     => ActiveRecord::Errors.default_error_messages[:too_long_integer],
      :too_short    => ActiveRecord::Errors.default_error_messages[:too_short_integer],
      :wrong_length => ActiveRecord::Errors.default_error_messages[:wrong_length_integer],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    configuration[:filter] = proc do |value|
      if /\A[+-]?([0-9]*)(?![0-9])/ =~ value.to_s
        $1.length
      else
        0
      end
    end
    attr_names.push configuration
    validates_filtered_length_of(*attr_names)
  end

  def validates_fractional_length_of(*attr_names)
    configuration = {
      :too_long     => ActiveRecord::Errors.default_error_messages[:too_long_fractional],
      :too_short    => ActiveRecord::Errors.default_error_messages[:too_short_fractional],
      :wrong_length => ActiveRecord::Errors.default_error_messages[:wrong_length_fractional],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    configuration[:filter] = proc do |value|
      if /\.([0-9]*)(?![0-9])/ =~ value.to_s
        $1.length
      else
        0
      end
    end
    attr_names.push configuration
    validates_filtered_length_of(*attr_names)
  end

  # 1〜9999年の範囲内かどうかをチェックする。
  def validates_year_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:invalid_year],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless value.blank?
        case value
        when Fixnum
          unless (1..9999).include?(value.to_i)
            record.errors.add(attr_name, configuration[:message])
          end
        when String
          unless /\A\d{4}\z/ =~ value && (1..9999).include?(value.to_i)
            record.errors.add(attr_name, configuration[:message])
          end
        else
        end
      end
    end
  end

  # 1〜9999年と1〜12月の範囲内かどうかをチェックする。
  def validates_year_month_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:invalid_year_month],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless value.blank?
        date = Rfw::YearMonth::Format.valid_year_month(value)
        unless date
          record.errors.add attr_name, configuration[:message]
        end
      end
    end
  end

  # 1〜9999年の範囲内の日付かどうかをチェックする。
  def validates_year_month_day_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:invalid_year_month_day],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless value.blank?
        date = Rfw::Date::Format.valid_date(value)
        unless date
          record.errors.add attr_name, configuration[:message]
        end
      end
    end
  end

  def validates_minimum_number_of(*attr_names)
    configuration = {
      :too_long     => ActiveRecord::Errors.default_error_messages[:too_large],
      :too_short    => ActiveRecord::Errors.default_error_messages[:too_small],
      :wrong_length => ActiveRecord::Errors.default_error_messages[:wrong_number],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    configuration[:filter] = proc {|value| value }
    attr_names.push configuration
    validates_filtered_length_of(*attr_names)
  end

  # NOTE: 検証する対象の値が日付を表す書式に合っていなかったら無視する。
  def validates_period_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:period],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    if e = configuration.delete(:in)
      proc = lambda do |record, value|
        return true unless Rfw::Date::Format.valid_date(value)
        vdate = value.to_display_date(true)
        first = (e.first =~ /\A\d/) ? e.first : record.instance_eval(e.first).to_s
        last = (e.last =~ /\A\d/) ? e.last : record.instance_eval(e.last).to_s
        ( (!Rfw::Date::Format.valid_date(first) || first.to_display_date <= vdate) &&
          (!Rfw::Date::Format.valid_date(last) || vdate <= last.to_display_date) )
      end
    elsif min = configuration.delete(:minimum)
      proc = lambda do |record, value|
        min = (min =~ /\A\d/) ? min : record.instance_eval(min).to_s
        return true unless Rfw::Date::Format.valid_date(min)
        return true unless Rfw::Date::Format.valid_date(value)
        min.to_display_date(true) <= value.to_display_date(true)
      end
    elsif max = configuration.delete(:maximum)
      proc = lambda do |record, value|
        max = (max =~ /\A\d/) ? max : record.instance_eval(max).to_s
        return true unless Rfw::Date::Format.valid_date(max)
        return true unless Rfw::Date::Format.valid_date(value)
        value.to_display_date(true) <= max.to_display_date(true)
      end
    else
      raise ArgumentError, "no option of the configuration hash"
    end
    validates_each(attr_names, configuration) do |record, attr_name, value|
      record.errors.add(attr_name, configuration[:message]) unless proc.call(record, value)
    end
  end

  # NOTE: 検証する対象の値が日付を表す書式に合っていなかったら無視する。
  def validates_later_date_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:later_date],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    date = configuration.delete(:than)
    case date
    when String
      display_date = date.to_display_date
      enum = "" .. display_date
      proc = lambda {|value| Rfw::Date::Format.valid_date(value) && enum.include?(value.to_display_date(true))}
    when Time
      display_date = date.strftime("%Y/%m/%d")
      enum = Time.at(0) .. date
      proc = lambda {|value| enum.include?(value)}
    else
      raise ArgumentError, "Date unspecified. Specify the :than option."
    end
    message = configuration[:message] % {:val => display_date}
    # logic from validates_exclusion_of
    validates_each(attr_names, configuration) do |record, attr_name, value|
      if proc.call(value)
        record.errors.add(attr_name, message)
      end
    end
  end

  # NOTE: 検証する対象の値が日付を表す書式に合っていなかったら無視する。
  def validates_earlier_date_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:earlier_date],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    date = configuration.delete(:than)
    case date
    when String
      display_date = date.to_display_date
      enum = "" ... display_date
      proc = lambda {|value| !Rfw::Date::Format.valid_date(value) || enum.include?(value.to_display_date(true))}
    when Time
      display_date = date.strftime("%Y/%m/%d")
      enum = Time.at(0) ... date
      proc = lambda {|value| enum.include?(value)}
    else
      raise ArgumentError, "Date unspecified. Specify the :than option."
    end
    message = configuration[:message] % {:val => display_date}
    # logic from validates_inclusion_of
    validates_each(attr_names, configuration) do |record, attr_name, value|
      unless proc.call(value)
        record.errors.add(attr_name, message)
      end
    end
  end

  def validates_row_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:row],
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)

    model_class_name = configuration[:table].classify
    model_class = model_class_name.constantize
    column_name = configuration[:column]

    validates_each(attr_names, configuration) do |record, attr_name, value|
      if model_class.find(:first, :conditions => {column_name => value}, :select => "id").nil?
        message = s_(configuration[:message]) % {
          :column => model_class.human_attribute_name(column_name),
          :table => s_(model_class_name), # FIXME
        }
        record.errors.add(attr_name, message)
      end
    end
  end

  def validates_isolated_of(*attr_names)
    configuration = {
      :message => ActiveRecord::Errors.default_error_messages[:not_isolated],
      :on      => :destroy,
    }
    configuration.update(attr_names.pop) if attr_names.last.is_a?(Hash)
    isolated_from = configuration[:from].split(',').map{|v| v.split('#') }

    validates_each(attr_names, configuration) do |record, attr_name, value|
      isolated_from.each do |model_class_name, column_name|
        if model_class_name.constantize.__send__(:exists?, column_name => value)
          message = s_(configuration[:message]) % {
            :column => column_name,
            :table  => model_class_name,
          }
          record.errors.add(attr_name, message)
        end
      end
    end
  end

  # HACK
  # 通常の validates_uniqueness_of だとコンテキストが特異クラスになるので
  # コンテキストをすり替えたり、内部で呼び出しているメソッドのレシーバを
  # すり替えたりしている。see also super
  def validates_uniqueness_of(*attr_names)
    configuration = { :message => ActiveRecord::Errors.default_error_messages[:taken], :case_sensitive => true }
    configuration.update(attr_names.extract_options!)

    validates_each(attr_names,configuration) do |record, attr_name, value|
      # The check for an existing value should be run from a class that
      # isn't abstract. This means working down from the current class
      # (self), to the first non-abstract class. Since classes don't know
      # their subclasses, we have to build the hierarchy between self and
      # the record's class.
      class_hierarchy = [record.class]
      while class_hierarchy.first != self
        break if class_hierarchy.first.superclass == ActiveRecord::Base
        class_hierarchy.insert(0, class_hierarchy.first.superclass)
      end

      # Now we can work our way down the tree to the first non-abstract
      # class (which has a database table to query from).
      finder_class = class_hierarchy.detect { |klass| !klass.abstract_class? }

      is_text_column = finder_class.columns_hash[attr_name.to_s].text?

      if !value.nil? && is_text_column
        value = value.to_s
      end

      if value.nil? || (configuration[:case_sensitive] || !is_text_column)
        condition_sql = "#{record.class.quoted_table_name}.#{attr_name} #{attribute_condition(value)}"
        condition_params = [value]
      else
        # sqlite has case sensitive SELECT query, while MySQL/Postgresql don't.
        # Hence, this is needed only for sqlite.
        condition_sql = "LOWER(#{record.class.quoted_table_name}.#{attr_name}) #{attribute_condition(value)}"
        condition_params = [value.chars.downcase.to_s]
      end

      if scope = configuration[:scope]
        Array(scope).map do |scope_item|
          scope_value = record.send(scope_item)
          condition_sql << " AND #{record.class.quoted_table_name}.#{scope_item} #{attribute_condition(scope_value)}"
          condition_params << scope_value
        end
      end

      unless record.new_record?
        condition_sql << " AND #{record.class.quoted_table_name}.#{record.class.primary_key} <> ?"
        condition_params << record.send(:id)
      end

      results = finder_class.with_exclusive_scope do
        finder_class.connection.select_all(
          finder_class.__send__(:construct_finder_sql,
                :select     => "#{finder_class.connection.quote_column_name(attr_name)}",
                :from       => "#{finder_class.quoted_table_name}",
                :conditions => [condition_sql, *condition_params]
                               )
                              )
      end

      unless results.length.zero?
        found = true

        # As MySQL/Postgres don't have case sensitive SELECT queries, we try to find duplicate
        # column in ruby when case sensitive option
        if configuration[:case_sensitive] && finder_class.columns_hash[attr_name.to_s].text?
          found = results.any? { |a| a[attr_name.to_s] == value }
        end

        record.errors.add(attr_name, configuration[:message]) if found
      end
    end
  end
end

ActiveRecord::Base.__send__(:extend, CustomValidations)

module CustomValidations
  module OnDestroy
    VALIDATIONS_ON_DESTROY = %w[validate_on_destroy]

    def self.included(base) # :nodoc:
      base.extend ClassMethods
      base.class_eval do
        alias_method_chain :destroy, :validation
      end

      base.__send__ :include, ActiveSupport::Callbacks
      base.define_callbacks *VALIDATIONS_ON_DESTROY
    end

    module ClassMethods
      private
      def validation_method(on)
        case on
        when :save   then :validate
        when :create then :validate_on_create
        when :update then :validate_on_update
        when :destroy then :validate_on_destroy
        end
      end
    end

    def destroy_with_validation(perform_validation = true)
      if perform_validation && valid_on_destroy? || !perform_validation
        destroy_without_validation
      else
        false
      end
    end

    def valid_on_destroy?
      errors.clear

      run_callbacks(:validate_on_destroy)
      validate_on_destroy

      errors.empty?
    end

    protected

    # Overwrite this method for validation checks used only on destroy.
    def validate_on_destroy # :doc:
    end
  end
end

ActiveRecord::Base.__send__(:include, CustomValidations::OnDestroy)
