# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: products
#
#  id                      :integer       not null, primary key
#  domain_id               :integer       not null
#  code                    :string(255)   not null
#  name_po                 :integer       not null
#  motion                  :string(255)
#  parent_id               :integer
#  lft                     :integer
#  rgt                     :integer
#  type                    :string(255)   not null
#  model_name              :string(255)
#  table_name              :string(255)
#  detail_model_name       :string(255)
#  detail_table_name       :string(255)
#  parent_succeeded        :boolean
#  pivot                   :string(255)
#  workflow_enabled        :boolean
#  workflow                :string(255)
#  workflow_body_method    :string(255)
#  mail                    :boolean
#  mail_skip_auth          :boolean
#  document                :boolean
#  document_name_method    :string(255)   default("name")
#  document_number_method  :string(255)   default("id")
#  document_content_method :string(255)   default("name")
#  attachment              :boolean
#  search                  :boolean
#  csv                     :boolean
#  initial_roleable_type   :string(255)
#  scope_roleable_type     :string(255)
#  permission_enabled      :boolean
#  created_at              :string(14)
#  updated_at              :string(14)
#  created_by              :integer
#  updated_by              :integer
#  created_in              :integer
#  updated_in              :integer
#  lock_version            :integer       default(0), not null
#

# プロダクト(機能)のモデル。
class Product < ActiveRecord::Base
  LABEL_TYPE = [
    [N_("rfw|Product|type|Type 1"), "ProductSingle"],
    [N_("rfw|Product|type|Type 3"), "ProductDetailed"],
  ]

  untranslate_all
  timestamps_as_string
  user_monitor
  acts_as_nested_set :scope => :domain_id
  acts_as_translated :name
  has_many :grant_ons, :as => :grant_targettable, :dependent => :destroy
  has_many :grant_on_without_periodic, :source => :grant_ons,:as => :grant_targettable, :order => "grant_ons.id"
  has_many :permissions, :as => :grant_targettable
  has_many :displays, :dependent => :destroy, :order => "id"

  after_create :move_to_child_of_root
  after_create :initialize_name

  alias_method :nested_set_root, :root
  alias_method :nested_set_children, :children
  alias_method :title, :name

  validates_presence_of :code
  validates_presence_of :document_name_method, :document_number_method, :document_content_method, :if => :document
  validates_presence_of :workflow, :workflow_body_method, :if => :workflow_enabled

  #validates_each :model_name do |record, attr_name, value|
  #  unless value.blank?
  #    begin
  #      next if value.constantize < ActiveRecord::Base
  #    rescue NameError
  #    end
  #    record.errors.add attr_name, s_("rfw|message|error|%{fn} is invalid.")
  #  end
  #end

  #delegate :table_name, :to => :model_class

  # エラーメッセージのために <em>attributes</em> を属性に持つインスタンスを返す。
  def self.for_error_message(attributes)
    product = self.new(attributes)
    product.name_po ||= 0
    if product.valid?
      raise "[bug] product must be with errors at here"
    end
    return product
  end

  # メニューレイアウトでの根を返す。
  def self.root
    find(:first, :conditions => "products.parent_id IS NULL", :order => "products.id")
  end

  # <em>key</em> についての情報を返す。
  def self.current(key)
    ccache = CacheEachRequest.current
    ccache[:product] ||= {}
    ccache[:product][:root] ||= root
    ccache[:product][key] ||= yield root
  end

  # 権限によって変更が可能と指定されている全てのプロダクトを返す。
  def self.modifiable
    Product.current(:modifiable) do |r|
      with_scope(:find => {:conditions => Permission.modifiable_conditions(:product)}) { r.all_children }
    end
  end

  # 権限によって閲覧が可能と指定されている全てのプロダクトを返す。
  def self.permissible
    Product.current(:permissible) do |r|
      with_scope(:find => {:conditions => Permission.permissible_conditions(:product)}) { r.all_children }
    end
  end

  # 権限によって閲覧が不可能と指定されている全てのプロダクトを返す。
  def self.invisible
    Product.current(:invisible) do |r|
      with_scope(:find => {:conditions => Permission.invisible_conditions(:product)}) { r.all_children }
    end
  end

  # 閲覧が可能、または閲覧が可能な子孫を持つプロダクトを返す。
  def self.necessary
    Product.current(:necessary) do |r|
      top = Product.permissible.inject([]) {|seed,m| (seed.include? m) ? seed : (seed | m.self_and_ancestors)} - [r]
      bot = Product.invisible.inject([])   {|seed,m| (seed.include? m) ? seed : (seed | m.full_set)}
      mid = (r.all_children - top - bot).reject {|m| (top & m.ancestors).empty?}
      top | mid
    end
  end

  # <tt>necessary</tt> な子プロダクト全体を返す。
  def children
    @children ||= nested_set_children & Product.necessary
  end

  # 階層化された権限にしたがって変更可能かどうかを判定する。
  def modifiable?
    return @modifiable if defined?(@modifiable)
    return @modifiable = true if Product.modifiable.include?(self)
    return @modifiable = false if Product.permissible.include?(self)
    return @modifiable = false if Product.invisible.include?(self)
    return @modifiable = parent.modifiable? if parent
    return @modifiable = false
  end

  # 階層化された権限にしたがって利用可能かどうかを判定する。
  def visible?
    return @visible if defined?(@visible)
    return @visible = true if Product.permissible.include?(self)
    return @visible = false if Product.invisible.include?(self)
    return @visible = parent.visible? if parent
    return @visible = false
  end

  # URL を構成するためのオプションを返す。
  def url_options
    return @url_options if defined?(@url_options)
    return {} unless self.motion
    @url_options = {}
    self.motion.scan(/(\w+)=(\w+)/) do |key, value|
      @url_options[key.to_sym] = value
    end
    return @url_options
  end

  def with_ajax?
    if /\A\// =~ motion
      return false
    else
      return true
    end
  end

  # <tt>model_name</tt> からモデル名を返す。
  def model_class
    @model_class ||= model_name.constantize
  end
  
  #追加
  # <tt>detail_model_name</tt> からモデル名を返す。
  def detail_model_class
    @detail_model_class ||= detail_model_name.constantize
  end
  
  # アプリケーションの管理者によって用意される画面を返す。
  def master_displays
    displays.find(:all, :conditions => {:type => %w|DisplayToList DisplayToShow DisplayToEdit DisplayToNew|})
  end

  # <em>x</em> に関連付けられている文書を返す。
  # (削除されているノードを参照している文書は削除する。)
  def documents_for(x)
    rels, removed = Document.find(:all, :conditions => {:relatable_product_id => id, :relatable_id => x.id, :relatable_type => x.class.to_s}).select {|d| d.target_product.visible?}.partition(&:target)
    removed.each(&:destroy)
    alts, removed = Document.find(:all, :conditions => {:target_product_id => id, :target_id => x.id, :target_type => x.class.to_s}).select {|d| d.relatable_product.visible?}.partition(&:relatable)
    removed.each(&:destroy)
    return rels | alts
  end

  # <em>x</em> に関連する文書のノード全体を返す。
  # (削除されているノードを参照している辺やパスは削除する。)
  def document_nodes_for(x)
    existing, removed = DocumentEdge.find(:all, :conditions => {:from_product_id => id, :from_id => x.id, :from_type => x.class.to_s}).select {|e| e.to_product.visible?}.partition(&:to)
    removed.each do |e|
      e.destroy
      e.document_paths.each(&:destroy)
    end
    return existing.map do |e|
      to = e.to
      to.product_as_document_node = e.to_product
      to
    end
  end

  # このプロダクトに関連する GrantOn のリストを返す
  def details
    grant_on_without_periodic
  end

  private

  def move_to_child_of_root
    self[:parent_id] || move_to_child_of(Product.root)
  end

  def initialize_name
    unless self.name_po_message
      self.name_po_message = PoMessageSingular.create!(:msgctxt => "", :msgid => "Product|#{id}|name|")
      Language.all.each do |lang|
        self.__send__ "name_#{lang.code}=", self.code
      end
      self.save!
    end
  end

  def initialize_menu_and_displays
    self.motion = "controller=product/product_id=#{id}/action=list"
    self.save!
    %w[Show New Edit List].each do |x|
      display_class = "DisplayTo#{x}".constantize
      d = display_class.create!(:product_id => id,
                                :name_po => 0,
                                :code => x.upcase,
                                :enabled => true)
      d.save!
    end
    return true
  end
end
