# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: po_translations
#
#  id            :integer       not null, primary key
#  domain_id     :integer       not null
#  type          :string(255)   not null
#  po_message_id :integer       not null
#  msgstr        :text
#  msgstr_0      :text
#  msgstr_1      :text
#  msgstr_2      :text
#  msgstr_3      :text
#  msgstr_4      :text
#  msgstr_5      :text
#  created_at    :string(14)
#  updated_at    :string(14)
#  created_by    :integer
#  updated_by    :integer
#  created_in    :integer
#  updated_in    :integer
#  lock_version  :integer       default(0), not null
#

require "gettext/rmsgfmt"

# gettext による翻訳のモデル。
class PoTranslation < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor
  belongs_to :po_message

  # .po ファイルの翻訳エントリに変換する。
  def to_po_entry(global=nil)
    str = "\n\#: #{self.class}:#{id} #{po_message.class}:#{po_message.id}\n"
    unless po_message
      logger.warn("missing po message of #{self.to_s}")
      return str
    end
    if global
      str << po_message.to_po_string_global
      str << self.to_po_string
      str << "\n"
    end
    str << po_message.to_po_string
    str << self.to_po_string
    if po_message.respond_to?(:to_po_string_fast)
      str << "\n"
      str << po_message.to_po_string_fast
      str << self.to_po_string
    end
    return str
  end

  # .po ファイルでの文字列に変換する。
  def to_po_string
    if po_message.plural?
      str = ""
      self.class.nplurals.times do |i|
        m = __send__("msgstr_#{i}")
        str << "msgstr[#{i}] #{m.to_s.po_mangle}\n"
      end
    else
      str = "msgstr #{msgstr.to_s.po_mangle}\n"
    end
    return str
  end

  # .po ファイルとしてエクスポートする。
  def self.export_to_po(out)
    messages = find_all_by_type(self.to_s, :order => "domain_id, id")
    require "gettext/rgettext"
    out << GetText::RGetText.generate_pot_header.sub(/Plural-Forms: [^\\]+/) do
      "Plural-Forms: nplurals=#{nplurals}; plural=#{plural};"
    end.sub(/Project-Id-Version: PACKAGE VERSION/) do
      "Project-Id-Version: #{App::PACKAGE} #{App::VERSION}"
    end.sub(/Last-Translator: [^\\]+/) do
      "Last-Translator: #{App::DB_LAST_TRANSLATOR}"
    end.sub(/Language-Team: [^\\]+/) do
      "Language-Team: #{App::DB_LANGUAGE_TEAM}"
    end.sub(/^\#, fuzzy\n/, "")
    out << "\n"
    hashes = []
    messages.each do |msg|
      po_msg = msg.po_message
      unless po_msg
        logger.warn("missing po message of #{self.to_s}")
        next
      end
      x = "#{po_msg.msgctxt}#{po_msg.msgid}".hash
      if hashes.include?(x)
        out << msg.to_po_entry
      else
        out << msg.to_po_entry(true)
        hashes << x
      end
    end
  end

  # 言語を2文字からなる文字列で返す。
  def self.lang
    to_s[/[A-Z][a-z]\z/].downcase
  end

  # .mo ファイルおよび .po ファイルのパスを返す。
  def self.get_mo_path_and_po_path
    return @po_path, @mo_path if @po_path && @mo_path
    mo_outdir = File.join(RAILS_ROOT, "locale", lang, "LC_MESSAGES")
    FileUtils.mkdir_p(mo_outdir) unless File.directory?(mo_outdir)
    po_outdir = File.dirname(mo_outdir)
    @po_path = File.join(po_outdir, "database.po")
    @mo_path = File.join(mo_outdir, "database.mo")
    return @po_path, @mo_path
  end

  # .po ファイルを削除する。
  def self.remove_po
    po_path, = get_mo_path_and_po_path
    begin
      if File.exist?(po_path)
        File.unlink(po_path)
        logger.info "[INFO] Remove #{po_path}"
      end
    rescue SystemCallError => e
      logger.error(e)
      # ignore
    end
  end

  # データベース由来の .po ファイルおよび .mo ファイルを更新する。
  def self.database_only_update_po_and_make_mo(skip_if_exist_po=true)
    po_path, mo_path = get_mo_path_and_po_path
    return if skip_if_exist_po && File.exist?(po_path)
    File.open(po_path, "wb") do |f|
      export_to_po(f)
    end
    GetText.rmsgfmt(po_path, mo_path)
    GetText.clear_cache
    logger.info("%%% #{self}.database_only_update_po_and_make_mo updated")
  end


  # 全言語について <em>database_only_update_po_and_make_mo</em> を行う。
  def self.database_only_update_po_and_make_mo_all(skip_if_exist_po=true)
    @sub_classes ||= Language.all.collect do |lang|
      "PoTranslation#{lang.code.capitalize}".constantize
    end
    @sub_classes.each do |po_translation|
      po_translation.database_only_update_po_and_make_mo(skip_if_exist_po)
    end
  end

  # 個人利用のためのコピーを返す。
  def private_copy(po_message_id)
    translation = self.class.new
    translation.attributes = attributes
    translation.po_message_id = po_message_id
    translation.save!
    return translation
  end
end
