# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: permissions
#
#  id                     :integer       not null, primary key
#  domain_id              :integer       not null
#  grant_on_id            :integer       not null
#  user_id                :integer       not null
#  priority               :integer       not null
#  value                  :string(255)   not null
#  inception              :string(8)     not null
#  expiry                 :string(8)     not null
#  created_at             :string(14)
#  updated_at             :string(14)
#  created_by             :integer
#  updated_by             :integer
#  created_in             :integer
#  updated_in             :integer
#  lock_version           :integer       default(0), not null
#  grant_targettable_id   :integer
#  grant_targettable_type :string(255)
#

# 権限の参照のモデル。
class Permission < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor
  acts_as_periodic
  belongs_to :user
  belongs_to :grant_on
  belongs_to :grant_targettable, :polymorphic => true

  validates_presence_of :grant_on_id, :grant_targettable_id, :grant_targettable_type

  # 権限の対象になっており型 <em>type</em> を持つレコードを配列で返す。
  def self.specified(type)
    condition = "grant_targettable_type = '#{type.to_s.classify}'"
    find(:all, :conditions => (User.current ? ["user_id = :user_id AND #{condition}", {:user_id => User.current.id}] : condition))
  end

  def self.positive(type, *values)
    result, alt = specified(type).partition {|perm| values.include?(perm.value)}
    result.inject [] do|seed, perm|
      temp = alt.select {|x| x.grant_targettable_id == perm.grant_targettable_id}
      if (temp.empty? || perm.priority <= temp.map {|x| x.priority}.min)
        seed << perm
      else
        seed
      end
    end
  end

  # 全権限の対象になっており型 <em>type</em> を持つレコードを配列で返す。
  def self.modifiable(type)
    positive(type, "full")
  end

  # 読み取りもしくは全権限の対象になっており型 <em>type</em> を持つレコードを配列で返す。
  def self.permissible(type)
    positive(type, "full", "visible")
  end

  # 不可視権限の対象になっており型 <em>type</em> を持つレコードを配列で返す。
  def self.invisible(type)
    result, alt = specified(type).partition {|perm| perm.value == "invisible"}
    result.inject [] do |seed, perm|
      temp = alt.select {|x| x.grant_targettable_id == perm.grant_targettable_id}
      if (temp.empty? || perm.priority < temp.map {|x| x.priority}.min)
        seed << perm
      else
        seed
      end
    end
  end

  # 全権限の対象で型 <em>type</em> を持つ条件節を返す。
  def self.modifiable_conditions(type)
    (User.current.nil? || User.admin?) ? "1 = 1" : {:id => modifiable(type).map(&:grant_targettable_id)}
  end

  # 読み取りもしくは全権限の対象で型 <em>type</em> を持つ条件節を返す。
  def self.permissible_conditions(type)
    (User.current.nil? || User.admin?) ? "1 = 1" : {:id => permissible(type).map(&:grant_targettable_id)}
  end

  # 不可視権限の対象で型 <em>type</em> を持つ条件節を返す。
  def self.invisible_conditions(type)
    (User.current.nil? || User.admin?) ? "0 = 1" : {:id => invisible(type).map(&:grant_targettable_id)}
  end

  def self.create_with_grant_on_and_user_id_and_priority(grant_on, user_id, priority, inception=nil, expiry=nil)
    create(:grant_on_id => grant_on.id,
           :domain_id   => grant_on.domain_id,
           :user_id     => user_id,
           :priority    => priority,
           :value       => grant_on.value,
           :inception   => inception.blank? ? grant_on.inception : [grant_on.inception, inception].max,
           :expiry      => expiry.blank? ? grant_on.expiry : [grant_on.expiry, expiry].min,
           :grant_targettable_id   => grant_on.grant_targettable_id,
           :grant_targettable_type => grant_on.grant_targettable_type)
  end

end
