# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: items
#
#  id                             :integer       not null, primary key
#  domain_id                      :integer       not null
#  display_id                     :integer       not null
#  code                           :string(255)   not null
#  name_po                        :integer       not null
#  type                           :string(255)   not null
#  model_name                     :string(255)
#  column_name                    :string(255)
#  method_chain                   :text
#  adapter_name                   :string(255)
#  layout                         :float
#  width                          :string(255)
#  align                          :string(255)   default("left"), not null
#  decorator                      :string(255)   default("none"), not null
#  decorator_parameter            :string(255)
#  linked                         :boolean       not null
#  link_parameter                 :string(255)
#  link_target                    :string(255)
#  position                       :integer
#  direction                      :string(255)
#  condition_value                :string(255)
#  condition_pattern              :string(255)   default("none"), not null
#  selected                       :boolean
#  ordered                        :boolean
#  search                         :boolean       default(TRUE), not null
#  control                        :string(255)   default("free"), not null
#  input_type                     :string(255)
#  field_size                     :string(255)
#  input_parameter                :string(255)
#  input_initializer              :string(255)
#  input_initial_value            :string(255)
#  write_protected                :boolean
#  validates_presence             :boolean
#  validates_non_negative_integer :boolean
#  validates_integer              :boolean
#  validates_non_negative_float   :boolean
#  validates_float                :boolean
#  validates_zero                 :boolean
#  validates_upper_case           :boolean
#  validates_lower_case           :boolean
#  validates_alphabetic           :boolean
#  validates_alphanumeric         :boolean
#  validates_halfwidth_katakana   :boolean
#  validates_fullwidth            :boolean
#  validates_fullwidth_katakana   :boolean
#  validates_inclusion_chars      :boolean
#  validates_minimum_length       :boolean
#  validates_maximum_length       :boolean
#  validates_integral_length      :boolean
#  validates_fractional_length    :boolean
#  validates_year                 :boolean
#  validates_year_month           :boolean
#  validates_year_month_day       :boolean
#  validates_hour                 :boolean
#  validates_hour_minute          :boolean
#  validates_postal_code          :boolean
#  validates_phone_number         :boolean
#  validates_email                :boolean
#  validates_url                  :boolean
#  validates_minimum_number       :boolean
#  validates_period               :boolean
#  validates_future_date          :boolean
#  validates_past_date            :boolean
#  validates_uniqueness           :boolean
#  validates_row                  :boolean
#  validates_isolated             :boolean
#  validates_inclusion_chars_1    :string(255)
#  validates_minimum_length_1     :string(255)
#  validates_maximum_length_1     :string(255)
#  validates_integral_length_1    :string(255)
#  validates_integral_length_2    :string(255)
#  validates_fractional_length_1  :string(255)
#  validates_fractional_length_2  :string(255)
#  validates_minimum_number_1     :string(255)
#  validates_period_1             :string(255)
#  validates_period_2             :string(255)
#  validates_row_1                :string(255)
#  validates_row_2                :string(255)
#  validates_isolated_1           :string(255)
#  created_at                     :string(14)
#  updated_at                     :string(14)
#  created_by                     :integer
#  updated_by                     :integer
#  created_in                     :integer
#  updated_in                     :integer
#  lock_version                   :integer       default(0), not null
#

# 画面の項目のモデル。
class Item < ActiveRecord::Base
  class Error < StandardError; end

  LABEL_ALIGN = [
    [N_("rfw|Item|align|Left"), "left"],
    [N_("rfw|Item|align|Right"), "right"],
    [N_("rfw|Item|align|Center"), "center"],
  ]
  LABEL_DECORATOR = [
    [N_("rfw|Item|decorator|None"), "none"],
    [N_("rfw|Item|decorator|Front"), "front"],
    [N_("rfw|Item|decorator|Back"), "back"],
    [N_("rfw|Item|decorator|Currency"), "currency"],
    [N_("rfw|Item|decorator|Convert line break"), "nl2br"],
  ]
  LABEL_CONDITION_PATTERN = [
    [N_("rfw|Item|condition_pattern|None"), "none"],
    [N_("rfw|Item|condition_pattern|Equal"), "eq"],
    [N_("rfw|Item|condition_pattern|Not Equal"), "neq"],
    [N_("rfw|Item|condition_pattern|Matched with Head"), "head"],
    [N_("rfw|Item|condition_pattern|Matched with Tail"), "tail"],
    [N_("rfw|Item|condition_pattern|Include"), "include"],
    [N_("rfw|Item|condition_pattern|Exclude"), "exclude"],
    [N_("rfw|Item|condition_pattern|Null"), "null"],
    [N_("rfw|Item|condition_pattern|Not Null"), "not-null"],
    [N_("rfw|Item|condition_pattern|Greater than or Equal"), "ge"],
    [N_("rfw|Item|condition_pattern|Less than or Equal"), "le"],
    [N_("rfw|Item|condition_pattern|Greater than"), "gt"],
    [N_("rfw|Item|condition_pattern|Less than"), "lt"],
  ]
  LABEL_DIRECTION = [
    [N_("rfw|Item|direction|NONE"), "NONE"],
    [N_("rfw|Item|direction|ASC"),  "ASC"],
    [N_("rfw|Item|direction|DESC"), "DESC"],
  ]
  LABEL_CONTROL = [
    [N_("rfw|Item|control|Available Freely"), "free"],
    [N_("rfw|Item|control|According to Permission"), "permission"],
    [N_("rfw|Item|control|Administrator Only"), "admin"],
  ]
  LABEL_INPUT_TYPE = [
    [N_("rfw|Item|input_type|Text Field"), "text"],
    [N_("rfw|Item|input_type|Text Area"), "textarea"],
    [N_("rfw|Item|input_type|Radio Button"), "radio"],
    [N_("rfw|Item|input_type|Check Box"), "checkbox"],
    [N_("rfw|Item|input_type|Combo Box"), "select"],
    [N_("rfw|Item|input_type|Picker"), "picker"],
  ]
  LABEL_INPUT_INITIALIZER = [
    [N_("rfw|Item|input_initializer|None"), ""],
    [N_("rfw|Item|input_initializer|Person Name"), "person"],
    [N_("rfw|Item|input_initializer|Company Name"), "company"],
    [N_("rfw|Item|input_initializer|Organization Name"), "organization"],
    [N_("rfw|Item|input_initializer|Today's Date"), "today"],
    [N_("rfw|Item|input_initializer|Counter"), "counter"],
    [N_("rfw|Item|input_initializer|Use the specified value"), "custom"]
  ]
  LABEL_SEARCH = [
    [N_("rfw|Item|search|On"), true],
    [N_("rfw|Item|search|Off"), false],
  ]
  LABEL_LINKED = [
    [N_("rfw|Item|linked|Linked"), true],
    [N_("rfw|Item|linked|Unlinked"), false],
  ]
  LABEL_ORDERED = [
    [N_("rfw|Item|ordered|On"), true],
    [N_("rfw|Item|ordered|Off"), false],
  ]
  LABEL_SELECTED = [
    [N_("rfw|Item|selected|On"), true],
    [N_("rfw|Item|selected|Off"), false],
  ]

  VALIDATES_COLUMN_NAME_ON_SAVE = %w[
    presence
    integer
    float
    non_negative_integer
    non_negative_float
    zero
    upper_case
    lower_case
    alphabetic
    alphanumeric
    halfwidth_katakana
    fullwidth
    fullwidth_katakana
    inclusion_chars
    minimum_length
    maximum_length
    integral_length
    fractional_length
    year
    year_month
    year_month_day
    hour
    hour_minute
    postal_code
    phone_number
    email
    url
    minimum_number
    period
    future_date
    past_date
    uniqueness
    row
  ].map {|x| "validates_#{x}"}

  VALIDATES_COLUMN_NAME_ON_DESTROY = %w[
    isolated
  ].map{|x| "validates_#{x}" }

  VALIDATES_COLUMN_NAME = VALIDATES_COLUMN_NAME_ON_SAVE + VALIDATES_COLUMN_NAME_ON_DESTROY

  VALIDATES_COLUMN_PARAMETER_COUNT = {
    :validates_inclusion_chars => 1,
    :validates_minimum_length => 1,
    :validates_maximum_length => 1,
    :validates_integral_length => 2,
    :validates_fractional_length => 2,
    :validates_minimum_number => 1,
    :validates_period => 2,
    :validates_row => 2,
    :validates_isolated => 1,
  }
  VALIDATES_COLUMN_PARAMETER_COUNT.default = 0

  untranslate_all
  timestamps_as_string
  user_monitor
  acts_as_translated :name
  belongs_to :display
  acts_as_list :scope => :display
  has_many :grant_ons, :as => :grant_targettable, :dependent => :destroy
  has_many :permissions, :as => :grant_targettable
  has_many :grant_on_without_periodic, :source => :grant_ons,:as => :grant_targettable, :order => "grant_ons.id"

  validates_presence_of :code

  #validates_each :model_name do |record, attr_name, value|
  #  unless value.blank?
  #    begin
  #      next if value.constantize < ActiveRecord::Base
  #    rescue NameError
  #    end
  #    record.errors.add attr_name, s_("rfw|message|error|%{fn} is invalid.")
  #  end
  #end

  delegate :quoted_table_name, :to => :model_class

  # モデルのクラスを返す。
  def model_class
    @model_class ||= model_name.constantize
  end

  def detail?
    model_class != display.model_class
  end

  # 並び替えを指定する文字列または <tt>nil</tt> を返す。
  def order
    nil
  end

  # 検証項目の一覧を返す。
  def validations(action)
    case action
    when :save
      validations_on_save
    when :destroy
      validations_on_destroy
    else
      raise 'must not happen!'
    end
  end

  def validations_on_save
    @validations_on_save ||= Item::VALIDATES_COLUMN_NAME_ON_SAVE.map do |validation|
      if __send__("#{validation}?")
        case validation
        when "validates_inclusion_chars"
          [:validates_inclusion_chars_of, column_name, {:chars => validates_inclusion_chars_1, :if => proc_filled?}]
        when "validates_minimum_length"
          [:validates_length_of, column_name, {:minimum => validates_minimum_length_1.to_i, :if => proc_filled?}]
        when "validates_maximum_length"
          [:validates_length_of, column_name, {:maximum => validates_maximum_length_1.to_i, :if => proc_filled?}]
        when "validates_integral_length"
          [:validates_integral_length_of, column_name, option_for_range(validates_integral_length_1.to_i, validates_integral_length_2.to_i, 0)]
        when "validates_fractional_length"
          [:validates_fractional_length_of, column_name, option_for_range(validates_fractional_length_1.to_i, validates_fractional_length_2.to_i, 0)]
        when "validates_minimum_number"
          [:validates_minimum_number_of, column_name, {:minimum => validates_minimum_number_1.to_i, :if => proc_filled?}]
        when "validates_period"
          [:validates_period_of, column_name, option_for_range(validates_period_1, validates_period_2, "")]
        when "validates_future_date"
          [:validates_later_date_of, column_name, {:than => Clock.now.strftime("%Y%m%d"), :if => proc_filled?}]
        when "validates_past_date"
          [:validates_earlier_date_of, column_name, {:than => Clock.now.strftime("%Y%m%d"), :if => proc_filled?}]
        when "validates_row"
          [:validates_row_of, column_name, {:table => validates_row_1, :column => validates_row_2}]
        when "validates_presence"
          chash = model_class.columns_hash
          c = chash[column_name]
          unless c || (chash["#{column_name}_id"] && chash["#{column_name}_type"])
            logger.warn("column '#{column_name}' not found, so ignored")
            next nil
          end
          # http://groups.google.com/group/rubyonrails-talk/browse_thread/thread/2f70159dc2e525fc
          if c.type == :boolean
            [:validates_inclusion_of, column_name, {:in => [true, false]}]
          else
            [:validates_presence_of, column_name]
          end
        else
          ["#{validation}_of", column_name, {:if => proc_filled?}]
        end
      end
    end.compact
    return @validations_on_save
  end

  def validations_on_destroy
    @validations_on_destroy ||= Item::VALIDATES_COLUMN_NAME_ON_DESTROY.map do |validation|
      if __send__("#{validation}?")
        case validation
        when "validates_isolated"
          [:validates_isolated_of, column_name, { :from => validates_isolated_1 } ]
        else
          # nop
        end
      end
    end.compact
    return @validations_on_destroy
  end

  # 
  def apply_validations(it, action)
    validations(action).reverse_each do |args|
      it.__send__(*args)
    end
  end

  # 読み取り権限があるかどうかを判定する。
  def readable?
    @readable = controlled? "visible", "full" unless defined? @readable
    @readable
  end

  # 書き込み権限があるかどうかを判定する。
  def writable?
    @writable = controlled? "full" unless defined? @writable
    @writable
  end

  # <em>it</em> の該当カラムの値を初期化する。
  # 初期化を行った場合は true、さもなくば false を返す。
  # 複写時にも呼び出される点で initialize_column_of と異なる。
  def fill_column_of(it)
    return false unless writable?
    return false if input_initializer.blank?
    case input_initializer
    when "counter"
      args = input_initial_value.split(',')
      it[column_name] = Counter.format(*args)
      return true
    end
    return false
  end

  # <em>it</em> の該当カラムの値を初期化する。
  def initialize_column_of(it)
    return unless writable?
    return if input_initializer.blank?
    return if fill_column_of(it)
    case input_initializer
    when "person"
      it[column_name] = User.current.person_id
    when "company"
      record = User.current.person.company_members.find(:first, :select => "DISTINCT company_id")
      if record
        it[column_name] = record.company_id
      end
    when "organization"
      record = User.current.person.organization_members.find(:first, :select => "DISTINCT organization_id")
      if record
        it[column_name] = record.organization_id
      end
    when "today"
      it[column_name] = Time.now.strftime("%Y/%m/%d")
    when "custom"
      it[column_name] = input_initial_value
    else
      logger.warn("WARN: invalid input_initializer: #{input_initializer}")
    end
  end

  # レイアウトの名前を返す。
  def coordinates
    return @coordinates if defined? @coordinates
    return @coordinates = nil if layout <= 0
    i = layout.to_i
    if display.is_a?(DisplayToList)
      @coordinates = "#{(i+64).chr}#{i}"
    else
      @coordinates = "#{(i%2==1)?'A':'B'}#{(i+1)/2}"
    end
    return @coordinates
  end

  # レイアウトのアルファベット部分を返す。
  def coordinate_alphabet
    c = coordinates
    return c && c[/\A[A-Z]/]
  end

  # レイアウトの番号部分を返す。
  def coordinate_number
    c = coordinates
    return c && c[/\d+/].to_i
  end

  # style 属性の値を返す。
  def style
#    data_length = human_name.split(//u).length
#    width_from__data = data_length > 0 ? "width: #{data_length*1.2}em;" : nil # Boldを考慮して1.2倍しています
#    width.blank? ? width_from__data  : "width: #{width}px;"
    width.blank? ? nil  : "width: #{width}px;"
  end

  # 入力フィールドの style 属性の値を返す。
  def style_for_input_field
    align.blank? ? nil : "text-align: #{align};"
  end

  # 入力フィールドの size 属性の値を返す。
  def size_for_input_field
    case input_type
    when "textarea"
      return field_size.blank? ? nil : field_size
    else
      return field_size.blank? ? nil : field_size.to_i
    end
  end

  # リンクが有効ならばその内容を表す配列、無効ならば false を返す。
  def link(it)
    return false unless linked?
    case link_parameter
    when /\Amailto:?(.*)\z/
      rest = $1
      if rest.blank?
        return [:mail_to, to_data(it)]
      else
        parameter, link_label = rest.split('>')
        addr = parameter % {:id => it.id, :class => it.class, :data => to_data(it)}
        return [:mail_to, addr, link_label && _(link_label)]
      end
    else
      result = link_url(it)
      result.unshift(:link_to)
      return result
    end
  end

  # リンクが有効ならばその URL、無効ならば false を返す。
  def link_url(it)
    return false unless linked?
    parameter, link_label = link_parameter.split('>')
    url = parameter % {:id => it.id, :class => it.class, :data => to_data(it)}
    [url, link_label && _(link_label)]
  end

  # リンクが有効ならばそのオプション、無効ならば false を返す。
  def link_options
    return false unless linked?
    link_target.blank? ? {} : {:target => link_target}
  end

  # 個人利用のためのコピーを返す。
  def private_copy(display_id, &block)
    item = self.class.new
    item.attributes = attributes
    message_source = PoMessageSingular.find(name_po)
    message_copy = message_source.private_copy
    item.name_po = message_copy.id
    item.display_id = display_id
    block.call(item) if block_given?
    item.save!
    return item
  end

  # condition_parameter = 'custom' の時に使用する
  def condition_hash=(hash)
    CacheEachRequest.current[:condition_hash] ||= { }
    CacheEachRequest.current[:condition_hash]["#{self.display_id}_#{self.id}"] ||= hash
  end

  def condition_hash
    CacheEachRequest.current[:condition_hash]["#{self.display_id}_#{self.id}"]
  end

  # この項目に関する GrantOn のリストを返す
  def details
    grant_on_without_periodic
  end

  private

  def controlled?(*values)
    return true unless User.current
    case control
    when "free"
      return true
    when "admin"
      return User.admin?
    end
    perms = permissions.select {|perm| perm.user_id == User.current_id}
    return true if perms.empty?
    perms, alt = perms.partition {|perm| values.include? perm.value}
    return false if perms.empty?
    return true if alt.empty?
    return (perms.map {|x| x.priority}.min <= alt.map {|x| x.priority}.min)
  end

  def proc_filled?
    Proc.new {|record| !record.attributes[column_name].blank?}
  end

  def option_for_range(x1, x2, invalid)
    option = {:if => proc_filled?}
    if x1.nil? || (x1 == invalid)
      option[:maximum] = x2
    elsif x2.nil? || (x2 == invalid)
      option[:minimum] = x1
    else
      option[:in] = (x1..x2)
    end
    return option
  end

  def resolve_method_chain(value, polymorphic=false)
    ItemMethodChain.resolve_method_chain(value, method_chain, polymorphic)
  rescue ActiveRecord::RecordNotFound
    ''
  end
end
