# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: config_outputs
#
#  id                  :integer       not null, primary key
#  domain_id           :integer       not null
#  user_id             :integer       not null
#  display_to_list_id  :integer       not null
#  file_format         :string(255)   default("csv"), not null
#  name                :string(255)
#  eol_pattern         :string(255)   default("none"), not null
#  eol_parameter       :string(255)
#  separator_pattern   :string(255)   default("tab"), not null
#  separator_parameter :string(255)
#  quotation_pattern   :string(255)   default("none"), not null
#  quotation_parameter :string(255)
#  encoding            :string(255)   default("utf-8"), not null
#  created_at          :string(14)
#  updated_at          :string(14)
#  created_by          :integer
#  updated_by          :integer
#  created_in          :integer
#  updated_in          :integer
#  lock_version        :integer       default(0), not null
#

# 出力ファイルの設定のモデル
class ConfigOutput < ActiveRecord::Base
  LABEL_FILE_FORMAT = [
    [N_("rfw|ConfigOutput|file_format|CSV"), "csv"],
    [N_("rfw|ConfigOutput|file_format|PDF"), "pdf"],
    [N_("rfw|ConfigOutput|file_format|XML"), "xml"],
  ]

  LABEL_EOL_PATTERN = [
    [N_("rfw|ConfigOutput|eol_pattern|none"), "none"],
    [N_("rfw|ConfigOutput|eol_pattern|disabled"), "disabled"],
    [N_("rfw|ConfigOutput|eol_pattern|specified"), "specified"],
  ]

  LABEL_SEPARATOR_PATTERN = [
    [N_("rfw|ConfigOutput|separator_pattern|tab"), "tab"],
    [N_("rfw|ConfigOutput|separator_pattern|comma"), "comma"],
    [N_("rfw|ConfigOutput|separator_pattern|semicolon"), "semicolon"],
    [N_("rfw|ConfigOutput|separator_pattern|specified"), "specified"],
  ]

  LABEL_QUOTATION_PATTERN = [
    [N_("rfw|ConfigOutput|quotation|none"), "none"],
    [N_("rfw|ConfigOutput|quotation|single"), "single"],
    [N_("rfw|ConfigOutput|quotation|double"), "double"],
    [N_("rfw|ConfigOutput|quotation|specified"), "specified"],
  ]

  LABEL_ENCODING = [
    [N_("rfw|ConfigOutput|encoding|UTF-8"), "utf-8"],
    [N_("rfw|ConfigOutput|encoding|Shift_JIS"), "CP932"],
  ]

  untranslate_all
  timestamps_as_string
  user_monitor

  belongs_to :domain
  belongs_to :user
  belongs_to :display_to_list
  has_many :config_output_items, :order => "config_output_items.position", :dependent => :destroy

  validates_presence_of :eol_parameter,       :if => Proc.new {|record| record.eol_pattern == "specified"}
  validates_presence_of :separator_parameter, :if => Proc.new {|record| record.separator_pattern == "specified"}
  validates_presence_of :quotation_parameter, :if => Proc.new {|record| record.quotation_pattern == "specified"}

  # 行末文字を返す。
  def process_eol(x)
    x = x.to_s
    case eol_pattern
    when "none"
      x
    when "disabled"
      x.gsub(/\r?\n/, '')
    when "specified"
      x.gsub(/\r?\n/, eol_parameter)
    else
      raise ArgumentError, "invalid eol_pattern; #{eol_pattern}"
    end
  end

  # (CSV 形式で)値を区切る文字を返す。
  def separator
    case separator_pattern
    when "tab"
      "\t"
    when "comma"
      ","
    when "semicolon"
      ";"
    when "specified"
      separator_parameter
    else
      raise ArgumentError, "invalid separator_pattern: #{separator_pattern}"
    end
  end

  # (CSV 形式で)値を囲む引用符を返す。
  def quotation
    case quotation_pattern
    when "none"
      ""
    when "single"
      "'"
    when "double"
      '"'
    when "specified"
      quotation_parameter
    else
      raise ArgumentError, "invalid quotation_pattern: #{quotation_pattern}"
    end
  end

  # (CSV 形式で)引用符で囲まれた値を返す。
  def quote(x)
    q = quotation
    y = process_eol(x)
    return y if q.blank?
    z = y.gsub(q, "#{q}#{q}")
    return "#{q}#{z}#{q}"
  end

  # 設定を初期化する。
  def clear
    self.attributes = {
      :name => display_to_list.name,
      :file_format => "csv",
      :eol_pattern => "none",
      :eol_parameter => "",
      :separator_pattern => "tab",
      :separator_parameter => "",
      :quotation_pattern => "none",
      :quotation_parameter => "",
      :encoding => "utf-8",
    }
  end

  # <em>x</em> のエンコーディングを設定に合わせて変換する。
  def iconv(x)
    return x if file_format == "pdf"
    return x if encoding == "utf-8"
    return x.map {|e| iconv(e)} if x.is_a?(Array)
    if x.is_a?(String)
      @iconverter ||= Iconv.new(encoding, "UTF-8")
      x = @iconverter.iconv(x)
      x << @iconverter.iconv(nil)
    end
    return x
  end

  # 表示項目に対応するヘッダを返す。
  def headers
    result = []
    each_item {|config_output_item| result << config_output_item.name}
    result
  end

  # 出力対象の行の一覧を返す。
  def rows(options, session_readonly, data_options={})
    unless display_to_list.output_module_name.blank?
      self.extend("ConfigOutput::#{display_to_list.output_module_name}".constantize)
      return self.output_rows(display_to_list_id, options, session_readonly)
    end
    result = []
    display_to_list.model_class.find(:all, options).each do |it|
      row = []
      each_item do |config_output_item|
        begin
          if data_options[:link]
            row << {:data => config_output_item.to_data(it), :link => config_output_item.link_url(it)}
          else
            row << config_output_item.to_data(it)
          end
        rescue NoMethodError
          # ignored
        end
      end
      result << row
    end
    result
  end

  # 出力を生成する。
  def generate(options, session_readonly)
    case file_format
    when "csv"
      require "csv"
      result = ''
      csv_options = {
        :col_sep => separator,
        :row_sep => "\r\n",
      }
      unless quotation.blank?
        csv_options.update(:quote_char => quotation,
                           :force_quotes => true)
      end
      FasterCSV.generate(result, csv_options) do |csv|
        csv << iconv(headers)
        rows(options, session_readonly).each do |row|
          csv << iconv(row.map{|col| process_eol(col) })
        end
      end
      return result
    when "pdf"
      header = SimplePDF::TextBox.new("%t", 8,
                                      :align => :left,
                                      :top_margin => 5.0,
                                      :left_margin => 10.0,
                                      :right_margin => 10.0)
      footer = SimplePDF::TextBox.new("%p/%P", 8,
                                      :align => :center,
                                      :top_margin => -20.0,
                                      :left_margin => 10.0,
                                      :right_margin => 10.0)

      table_header_style = SimpleCellStyle.new(:border_color => [0, 0, 0],
                                               :text_color => [0, 0, 0],
                                               :fill_color => [200, 200, 200],
                                               :top_margin => 2.0,
                                               :bottom_margin => 2.0,
                                               :left_margin => 2.0,
                                               :right_margin => 2.0,
                                               :align => :center,
                                               :font_size => 20.0,
                                               :line_height => 8.0)
      table_cell_style =  SimpleCellStyle.new(:border_color => [0, 0, 0],
                                              :text_color => [0, 0, 0],
                                              :fill_color => nil,
                                              :top_margin => 2.0,
                                              :bottom_margin => 2.0,
                                              :left_margin => 2.0,
                                              :right_margin => 2.0,
                                              :align => :left,
                                              :font_size => 20.0,
                                              :line_height => 10.0)

      contents = [headers.map {|h| {:data => h, :style => table_header_style}}]
      rows(options, session_readonly, :link => true).each do |row|
        contents << row.map {|x| x.update(:style => table_cell_style)}
      end
      table = SimpleTable.new(contents.size, headers.size)
      table.set_contents do |row, col|
        SimpleCell.new(contents[row][col][:data], contents[row][col][:style], contents[row][col][:link])
      end
      
      pdf = SimpleTablePDF.new(:orientation => "Portrait",
                               :format => "A4",
                               :language => user.person.last_language,
                               :title => name,
                               :author => "PjC",
                               :creator => "PjC",
                               :keywords => ["PjC", "テスト"], # FIXME
                               :subject => "サブタイトル", # FIXME
                               :zoom => "fullpage",
                               :header => header,
                               :footer => footer,
                               :table => table,
                               :top_margin => 25.0, # mm
                               :left_margin => 20.0, # mm
                               :right_margin => 20.0, # mm
                               :bottom_margin => 30.0, # mm
                               :page_order => 0)

      return pdf.output
    when "xml"
      require "rexml/document"

      doc = REXML::Document.new "<table></table>"
      root = doc.root

      thead = REXML::Element.new "thead"
      tr = REXML::Element.new "tr"
      headers.each do |header|
        th = REXML::Element.new "th"
        th.text = header
        tr.add_element th
      end
      thead.add_element tr
      root.add_element thead

      tbody = REXML::Element.new "tbody"
      rows(options, session_readonly).each do |row|
        tr = REXML::Element.new "tr"
        row.each do |d|
          td = REXML::Element.new "td"
          td.text = process_eol(d)
          tr.add_element td
        end
        tbody.add_element tr
      end
      root.add_element tbody

      doc.xml_decl.encoding = "utf-8"
      output = ''
      doc.write output

      output.sub!(/encoding='UTF-8'/, "encoding='#{encoding == 'CP932' ? 'Shift_JIS' : encoding.upcase}'")
      return iconv(output)
    else
      raise ArgumentError, "invaild file_format: #{file_format}"
    end
  end

  # 出力形式に合わせた MIME type を返す。
  def mime_type
    case file_format
    when "csv"
      "text/csv; charset=#{encoding}; header=present"
    when "pdf"
      "application/pdf"
    when "xml"
      "text/xml; charset=#{encoding}"
    else
      raise ArgumentError, "invalid file_format: #{file_format}"
    end
  end

  # ファイル名を返す。
  def file_name(user_agent)
    if user_agent =~ /MSIE/
      conv = Iconv.new('CP932', 'UTF-8') # FIXME: works only in japanese
      ret = conv.iconv(name)
      ret << conv.iconv(nil)
    else
      ret = iconv(name)
    end
    return "#{ret}.#{file_format}"
  end

  private

  def each_item(&block)
    config_output_items.each do |config_output_item|
      if config_output_item.enabled? && config_output_item.readable? && config_output_item.selected?
        yield config_output_item
      end
    end
  end
end
