# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: config_attachments
#
#  id                :integer       not null, primary key
#  domain_id         :integer       not null
#  default_file_type :string(255)   not null
#  maximum_length    :integer       not null
#  created_at        :string(14)
#  updated_at        :string(14)
#  created_by        :integer
#  updated_by        :integer
#  created_in        :integer
#  updated_in        :integer
#  lock_version      :integer       default(0), not null
#

# 添付ファイルの設定のモデル
class ConfigAttachment < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor

  FILENAME_MAXBYTESIZE = 64
  FILENAME_PROHIBITED_CHARS = ["〜"]

  # ファイル名のバイトサイズに関するエラー
  class FilenameMaximumBytesizeError < ArgumentError; end
  # ファイル名に含まれる文字に関するエラー
  class FilenameProhibitedCharacterError < ArgumentError; end
  # ファイルのサイズの上限を越えた場合のエラー
  class MaximumLengthError < ArgumentError; end

  # 各ドメインの現在の設定を返す。
  def self.current
    find(:first)
  end

  # 設定されているファイルの上限サイズを <em>length</em> が越えないかどうか判定する。
  def validate_length(length)
    if length > maximum_length
      raise MaximumLengthError, "exceeded maximum file length (= #{maximum_length}): #{length}"
    end
    return true
  end

  # 設定されているファイル名の制限に沿っているかどうかを判定する。
  def validate_filename(name)
    bs = name.respond_to?(:bytesize) ? name.bytesize : name.size
    if bs > FILENAME_MAXBYTESIZE
      raise FilenameMaximumBytesizeError, "exceeded maximum byte size (= #{FILENAME_MAXBYTESIZE}) of file name: #{bs}"
    end
    FILENAME_PROHIBITED_CHARS.each do |c|
      if name[c]
        raise FilenameProhibitedCharacterError, "prohibited character found in file name: #{c}"
      end
    end
    return true
  end

end
