# -*- coding: utf-8 -*-
# 関連文書機能のヘルパーを与える。
module DocumentHelper
  # 関連付け対象を埋め込むタグを返す。
  def hidden_relatable(name, x)
    hidden_field_tag name, params[name], {:id => "relatable_#{name}_#{x}"}
  end

  # ポップアップを閉じるリンクを返す。
  def link_to_close_popup
    link_with_effect s_("rfw|button|Close"), "summary"
  end

  # 取り消しのリンクを返す。
  def link_to_cancel
    link_with_effect s_("rfw|button|Cancel"), "cancel"
  end

  # 関連付ける機能を選択するための一覧を返す。
  def document_product_form_content
    content = []
    content << hidden_relatable(:type, :top)
    content << hidden_relatable(:id, :top)
    content << hidden_relatable(:product_id, :top)
    content << product_selector
    unless params[:target_product_id].blank?
      if product = Product.find_by_id(params[:target_product_id].to_i)
        if product.visible? && product.document?
          content << submit_to_select
          content << submit_to_create if product.modifiable?
        end
      end
    end
    return content.join("")
  end

  # 関連付ける機能を選択するための select box を返す。
  def product_selector
    id = "document_product_id"
    options = Product.find(:all).select {|p| p.visible? && p.document?}.map {|p| [p.name, p.id]}
    options.unshift([h(s_("rfw|Document|select product for new document")), nil])
    result = select_tag(:target_product_id,
                        options_for_select(options, params[:target_product_id].to_i),
                        :id => id,
                        :onchange => "['document_product_select', 'document_product_create'].each(function(x) {(e=$(x))&&e.disable();});$('view_dp').hide();")
    result += observe_field(id,
                            :with => "Form.serialize('document_product')",
                            :update => "document_product",
                            :complete => visual_effect(:highlight, id, :queue => {
                                                         :scope => "document",
                                                         :limit => 2,
                                                         :position => "end"
                                                       }),
                            :url => {:action => "product"})
    return result
  end

  # 関連文書選択画面を表示するための submit ボタンを返す。
  def submit_to_select
    submit_to_remote("select",
                     h(s_("rfw|button|Select")),
                     :update => "view_dp",
                     :complete => "$('view_dp').show();",
                     :url => {:controller => "product", :action => "select_document", :product_id => params[:target_product_id], :relatable_product_id => params[:product_id]},
                     :html => {:class => "button", :id => "document_product_select"})
  end

  # 新規文書作成画面を表示するための submit ボタンを返す。
  def submit_to_create
    submit_to_remote("create",
                     h(s_("rfw|button|New")),
                     :update => "view_dp",
                     :complete => "$('view_dp').show();",
                     :url => {:controller => "product", :action => "create_document", :product_id => params[:target_product_id], :relatable_product_id => params[:product_id]},
                     :html => {:class => "button", :id => "document_product_create"})
  end

  # 削除を実行するための submit ボタンを返す。
  def submit_to_delete
    submit_to_remote("delete",
                     h(s_("rfw|Document|Delete")),
                     :update => "document_table_base",
                     :complete => visual_effect(:highlight, "document_table", :queue => {:scope => "document", :limit => 2, :position => "end"}),
                     :url => {:action => "delete", :product_id => @relatable_product.id},
                     :html => {:class => "button"})
  end

  # 関連文書のリンクのための URL を返す。
  def document_url(x)
    if x.is_a?(Document)
      options = x.fragment_options_for(@relatable)
    else
      options = x.fragment_options_as_document_node
    end
    url_for(:action => "view", :fragment => URI.encode(fragment_for(options)))
  end

  # 関連文書のリンクのための URL を返す。
  def document_inner_url(x)
    if x.is_a?(Document)
      options = x.url_options_for(@relatable)
    else
      options = x.url_options_as_document_node
    end
    url_for(options.merge(:controller => "product", :action => "show_only"))
  end

  # 削除するための check box を返す。
  def check_box_to_delete(document)
    name = document.new_record? ? "deleted[#{document.target_type}_#{document.target_id}]" : "deleted[#{document.id}]"
    check_box_tag(name, 1, false)
  end

  # 相互参照のための check box を返す。
  def td_check_box_for_mutual(document)
    id = document.new_record? ? "document_mutual_#{document.target_type}_#{document.target_id}" : "document_mutual_#{document.id}"
    name = document.new_record? ? "mutual[#{document.target_type}_#{document.target_id}]" : "mutual[#{document.id}]"
    result = check_box_tag(name, 1, document.mutual?, :id => id)
    result += observe_field(id,
                            :with => "Form.serialize('document_table_form')",
                            :update => "document_table_base",
                            :complete => visual_effect(:highlight, "td_#{id}", :queue => {:scope => "document", :limit => 2, :position => "end"}),
                            :url => {:action => "toggle_mutual", :product_id => @relatable_product.id})
    return content_tag(:td, result, :id => "td_#{id}")
  end

  # 関連文書のサマリを返す。
  def summary
    result = []
    if (count = @documents.size) > 0
      result << ns_("rfw|Document|%{count} document related", "%{count} documents related", count) % {:count => count}
    else
      result << s_("rfw|Document|no document related")
    end
    if session[:added_documents]
      if @relatable
        documents = session[:added_documents].select {|a| a.relatable == @relatable}
      else
        documents = session[:added_documents].select {|a| a.relatable_type == params[:type] && !a.relatable_id}
      end
      ids = Document.all.map(&:id)
      documents = documents.select {|doc| ids.include? doc.target_id }
      if (count = documents.size) > 0
        result << ns_("rfw|Document|%{count} document will be added", "%{count} documents will be added", count) % {:count => count}
      end
    end
    if @relatable
      if session[:deleted_documents] && (count = session[:deleted_documents].select {|a| a.relatable == @relatable || a.target == @relatable}.size) > 0
        result << ns_("rfw|Document|%{count} document will be deleted", "%{count} documents will be deleted", count) % {:count => count}
      end
    end
    s_("rfw|Document|Documents: ") + result.join(" / ")
  end

  # タブ選択のための JavaScript を文字列として返す
  # TODO: JavaScript 内で引数チェックをするべき？
  def select_tab(uuid, index)
    <<-JS
(function(){
  var tabId = jQuery("##{uuid}").parent().parent().get(0).id;
  jQuery("#" + tabId + " > ul").tabs('select', #{index});
})();
    JS
  end

  def tab_id(product_id, document_id)
    "tabs_document#{document_id}of#{product_id}_#{rand.to_s.gsub('0.', '')}"
  end

  private

  def link_with_effect(name, action)
    link_to_remote(h(name), {
                     :update => "summary_documents",
                     :complete => "$('view_de').hide();" + visual_effect(:highlight, "summary_documents", :queue => {:scope => "document", :limit => 2, :position => "end"}),
                     :url => {:action => action, :id => params[:id], :type => params[:type], :product_id => params[:product_id]},
                   }, {:class => "button"})
  end
end
