# ポータルを制御する。
class PortalController < ApplicationController
  # ユーザーが不明な場合のエラー
  class UnknownUser < StandardError; end
  # 重複するポートレットが存在する場合のエラー
  class DuplicatePortlet < StandardError; end

  around_filter :catch_unknown_user

  # ポータルレイアウトで表示する。
  def index
    setup_portal
    @screen = @portal.screen
    @panes = @screen.panes
    @current_view = "view_m"
  end

  # ポータルの設定画面を表示する。
  def config
    setup_portal
    @screen = @portal.screen
    @current_panes = @screen.panes
    @groups = PortlettableGroup.find(:all)
    @current_view = "view_m"
  end

  # グループごとのポートレッタブルの一覧を表示する。
  def portlettable_members
    setup_portal
    unless params[:group_id]
      render :nothing => true
      return
    end
    @group = PortlettableGroup.find(params[:group_id])
    @members = @group.portlettable_members.reject {|m| @portal.portlets.map(&:portlettable).include?(m.portlettable)}
    render :partial => "portlettable_members"
  end

  # ポータルのレイアウトのパターンを設定する。
  def update_pattern
    if params[:pattern]
      setup_portal
      @portal.update_pattern params[:pattern]
    end
    render :nothing => true
  end

  # ポータルのレイアウトのペインを表示する。
  def pane
    unless params[:pane]
      render :nothing => true
      return
    end
    pane = params[:pane].upcase.to_sym
    setup_portal
    @screen = @portal.screen
    @current_panes = @screen.panes
    @groups = PortlettableGroup.find(:all)
    render :partial => "pane", :locals => {:pane => pane}
  end

  # ペイン内のポートレットを配置する。
  def update_portlets
    return render(:nothing => true) unless params[:pane] =~ /\A[aAbBcC]\z/

    kept, newbie = [], []
    key = "portlets_#{params[:pane].downcase}".to_sym
    if params[key]
      params[key].each_with_index do |x, i|
        case x
        when /\Aportlet_([0-9]+)\z/
          kept[$+.to_i] = i + 1
        when /\Aportlettable_member_([0-9]+)\z/
          newbie << [$+.to_i, i + 1]
        else
          # nothing to do
        end
      end
    end

    setup_portal
    Portal.transaction do
      pane = @portal.pane(params[:pane])
      # arrange or remove current portlets
      @portal.portlets.each do |portlet|
        if position = kept[portlet.id]
          portlet.position = position
          portlet.save
        elsif portlet.pane == pane
          portlet.destroy
        end
      end
      # add new portlets
      newbie.each do |member_id, position|
        member = PortlettableMember.find(member_id)
        begin
          Portlet.create!(:portal_id => @portal.id,
                          :pane_id => pane.id,
                          :position => position,
                          :portlettable_id => member.portlettable_id,
                          :portlettable_type => member.portlettable_type)
        rescue ActiveRecord::ActiveRecordError
          flash[:warning] = s_("rfw|flash|notice|It is impossible to add a duplicate portlet in your portal.")
          raise DuplicatePortlet, ""
        end
      end
    end
    render :nothing => true
  end

  private

  def catch_unknown_user
    yield
  rescue UnknownUser => e
    render :text => s_("rfw|Portal|(Portal)")
  rescue NotFoundException => e
    render :nothing => true
  end

  def setup_portal
    user = User.current
    raise UnknownUser, "" unless user
    @portal = user.portal
    raise NotFoundException, "no portal" unless @portal
  end
end
