# -*- coding: utf-8 -*-
# 選択部品を制御する:
# - 個人選択
# - グループ選択
# - 役職選択
# - 組織選択
# - 企業選択
# - 個人一括選択
# - 一括選択
# - 指定マスタ選択
class PickerController < ApplicationController
  before_filter :view_in_picker

  # allow registered return_to only; do not redirect to an arbitrary url
  PICKER_RETURN_TO = {
    :list_share => {:controller => "list", :action => "edit"},
    :picker_test => {:controller => "picker", :action => "test"},
  }.with_indifferent_access
  PICKER_ERROR_RETURN_TO = "/"

  # <tt>test</tt> へリダイレクトする。
  def index
    redirect_to :action => :test
  end

  # (試験用)
  def test
    @current_view = "view_main"
    @sub_view = "view_picker"

    if flash[:pick]
      case flash[:pick][:field]
      when "group"
        @group = Group.find(flash[:pick][:group])
      when "company"
        @company = Company.find(flash[:pick][:company])
      when "organization"
        @organization = Organization.find(flash[:pick][:organization])
      when "organization,company"
        @organization = Organization.find(flash[:pick][:organization])
        @company = Company.find(flash[:pick][:company])
      when "person"
        @person = Person.find(flash[:pick][:person])
      when "person,organization"
        @person = Person.find(flash[:pick][:person])
        @organization = Organization.find(flash[:pick][:organization])
      when "person,organization,company"
        @person = Person.find(flash[:pick][:person])
        @organization = Organization.find(flash[:pick][:organization])
        @company = Company.find(flash[:pick][:company])
      when "post"
        @post = Post.find(flash[:pick][:post])
      when "lump"
        case flash[:pick][:lump_pick]
        when "company"
          lump = @company = Company.find(flash[:pick][:company])
        when "organization"
          lump = @organization = Organization.find(flash[:pick][:organization])
        when "group"
          lump = @group = Group.find(flash[:pick][:group])
        when "personal_group"
          lump = @personal_group = PersonalGroup.find(flash[:pick][:personal_group])
        when "person"
          lump = @person = Person.find(flash[:pick][:person])
        end
        @people_ids = lump.id
        @people_names = lump.name
      when "lump_people"
        case flash[:pick][:lump_pick]
        when "company"
          people = Company.find(flash[:pick][:company]).people
        when "organization"
          people = Organization.find(flash[:pick][:organization]).people
        when "group"
          people = Group.find(flash[:pick][:group]).people
        when "personal_group"
          people = PersonalGroup.find(flash[:pick][:personal_group]).people
        when "person"
          people = [Person.find(flash[:pick][:person])]
        end
        @people_ids = people.map(&:id).join(", ")
        @people_names = people.map(&:name).join(", ")
      when "mail"
        if flash[:pick][:mail]
          @mail = flash[:pick][:mail]
          if @mail[:recipients]
            @recipients = Person.joints(@mail[:recipients]).select(&:can_be_recipient?)
          end
        end
      when "record"
        @record = flash[:pick][:record] || {}
      end
    end
  end

  # 選択を行う。
  def pick
    flash[:pick] = {
      :field => params[:return_field],
      :group => params[:group_id],
      :personal_group => params[:personal_group_id],
      :company => params[:company_id],
      :organization => params[:organization_id],
      :person => params[:person_id],
      :post => params[:post_id],
      :lump_pick => params[:lump_pick],
    }
    return_to_action = [:picker, :return_to, :action].trav(session) do |key|
      logger.warn("PickerController#pick: #{key} does not set")
    end
#    if %w[ new edit ].include? return_to_action
    if /\A(?:list|new|edit)/ =~ return_to_action
      x_close_or_redirect_to return_to_url
    else
      redirect_to return_to_url
    end
  end

  # グループを検索する。
  def group
    with_picker_scope(Group) do
      search :groups
      pass_params
    end
  end

  # 企業を検索する。
  def company
    with_picker_scope(Company) do
      search :companies
      pass_params
    end
  end

  # 組織を検索する。
  def organization
    with_picker_scope(Company, Organization) do
      if company = search_by_id(:company)
        search_under company, :organizations
      end
      pass_params :company
    end
  end

  # 個人を検索する。
  def person
    with_picker_scope(Company, Organization, Person) do
      if company = search_by_id(:company)
        if organization = search_by_id(:organization, company)
          search_under organization, :people
        end
      end
      pass_params :company, :organization
    end
  end

  # 役職を検索する。
  def post
    with_picker_scope(Company, Post) do
      company = search_by_id :company
      search_under company, :posts
      pass_params :company
    end
  end

  # 一括で検索する。
  def lump
    set_mode_options
    case params[:mode]
    when "group"
      lump_group
    when "personal_group"
      lump_personal_group
    else
      person
    end
  end

  # メールアドレスを検索する。
  def mail
    picker_session = session[:picker] || {}
    params[:mail] ||= picker_session[:mail_before_picker] || {}
    params[:mail][:field_type] ||= "to"
    if params[:mail][:recipients].is_a?(Hash)
      params[:mail][:recipients] = params[:mail][:recipients].keys
    end
    # set params["commit"] too when mode changed
    if !params["commit"] && params["pick"]
      flash[:pick] = {
        :field => params[:return_field],
        :mail => params[:mail],
      }
      x_close_or_redirect_to return_to_url
      return
    end

    @mail_field_type_list = [
      [s_("rfw|Picker|Mail Field Type|To"), "to"],
      [s_("rfw|Picker|Mail Field Type|Cc"), "cc"],
      [s_("rfw|Picker|Mail Field Type|Bcc"), "bcc"],
    ]

    @select_more_columns_from_people = ",people.mail_address"
    if params[:mail][:recipients]
      if params[:mail][:recipients].is_a?(Hash)
        params[:mail][:recipients] = params[:mail][:recipients].keys
      elsif params[:mail][:recipients].is_a?(String)
        params[:mail][:recipients] = params[:mail][:recipients].split(/,/)
      end
      @recipients = Person.joints(params[:mail][:recipients]).select(&:can_be_recipient?)
    else
      @recipients = []
    end

    unless params["commit"]
      %w"company organization group".each do |key|
        if params[key]
          key_id, = params[key].keys
          case key_id
          when /\Aopen(\d+)/
            params["#{key}_id"] = $1
          when /\Apick(\d+)/
            params["#{key}_id"] = $1
            @recipients += key.classify.constantize.find($1.to_i).people.select(&:can_be_recipient?)
          end
        end
      end
      if params["person"]
        key_id, = params["person"].keys
        person = Person.find_by_id(key_id)
        if person && person.can_be_recipient?
          @recipients << person
        end
      end
      @recipients.uniq!
      if params["unset"]
        key_id, = params["unset"].keys
        key_id = key_id.to_i
        @recipients.delete_if do |person|
          person.id == key_id
        end
      end
    end

    lump
  end

  # モデルとして指定されたテーブルを検索する。
  def record
    @record_id, @record_name, @record_code = fields = params[:record_field].split(",")
    case fields.size
    when 2, 3
      if !params[:commit] && params[:pick] # done
        record = {
          :record_id   => params[@record_id],
          :record_name => params[@record_name],
        }
        record[:record_code] = params[@record_code] if @record_code
        flash[:pick] = {
          :field => params[:return_field],
          :record => record
        }
        x_close_or_redirect_to return_to_url
        return
      end
      pass_params
      [:pick, :action].each do |key|
        @pass_params_form[key] = "record"
      end
      [:record_field, :record_model].each do |key|
        @pass_params_form[key] = params[key]
      end
    else
      raise "wrong number of subject fields"
    end
    {:name => 1, :code => 2}.each do |key, i|
      unless (value = params["record_#{key}"]).blank?
        record_model = params[:record_model].classify.constantize
        if @records
          @records &= find_like(record_model, fields[i], value)
        else
          @records = find_like(record_model, fields[i], value)
        end
      end
    end
  end

  private

  def set_mode_options
    @modes = [
      [s_("rfw|Picker|Company"), nil],
      [s_("rfw|Picker|Group"), "group"],
      [s_("rfw|Picker|Personal Group"), "personal_group"],
    ]
  end

  def lump_group
    with_picker_scope(Group, Person) do
      group = search_by_id :group
      search_under group, :people
      pass_params :group
    end
  end

  def lump_personal_group
    with_picker_scope(PersonalGroup, Person) do
      personal_group = search_by_id :personal_group
      search_under personal_group, :people
      pass_params :personal_group
    end
  end

  def return_to_url
    picker_session = session[:picker] || {}
    if return_to = PICKER_RETURN_TO[params[:return_to]]
      return return_to
    elsif picker_session[:return_to]
      return picker_session[:return_to]
    else
      return PICKER_ERROR_RETURN_TO
    end
  end

  def view_in_picker
    if /\Aview_[a-z0-9]+\z/ =~ params[:view]
      @current_view = params[:view]
    else
      @current_view = "view_picker"
    end
    case params[:return_field]
    when /person,organization,company/
      @title = s_("rfw|title|Person, Organization, and Company Picker")
    when /person,organization/
      @title = s_("rfw|title|Person and Organization Picker")
    when /organization,company/
      @title = s_("rfw|title|Organization and Company Picker")
    when /group/
      @title = s_("rfw|title|Group Picker")
    when /company/
      @title = s_("rfw|title|Company Picker")
    when /organization/
      @title = s_("rfw|title|Organization Picker")
    when /person/
      @title = s_("rfw|title|Person Picker")
    when /post/
      @title = s_("rfw|title|Post Picker")
    when /lump_people/
      @title = s_("rfw|title|Lump People Picker")
    when /lump/
      @title = s_("rfw|title|Lump Picker")
    when /mail/
      @title = s_("rfw|title|Mail Picker")
    when /record/
      @title = s_("rfw|title|Record Picker")
    else
      unless /\A(?:test|index)\z/ =~ params[:action]
        logger.error "ERROR: unknown picker #{params[:return_field].inspect} #{params[:action].inspect}"
        raise "unknown picker"
      end
    end
    @return_to_url = return_to_url
  end

  def search(table_name)
    singular = table_name.to_s.singularize
    model_class = singular.classify.constantize
    name = params["#{singular}_name"]
    if name.blank?
      items = model_class.find(:all)
    else
      items = find_like_name(model_class, name)
    end
    case table_name.to_sym
    when :companies
      items = lift_by_membership(User.current.person.companies, items)
    end
    instance_variable_set "@#{table_name}", items
    return items
  end

  def search_under(scope, table_name)
    return [] unless scope
    params["#{scope.class.to_s.underscore}_id"] = scope.id
    singular = table_name.to_s.singularize
    items = scope.__send__(table_name)
    name = params["#{singular}_name"]
    unless name.blank?
      items = find_like_name(items, name)
    end
    case table_name.to_sym
    when :organizations
      items = lift_by_membership(User.current.person.organizations, items)
    end
    instance_variable_set "@#{table_name}", items
    return items
  end

  def search_by_id(singular, scope=nil)
    name_hidden = "#{singular}_name_hidden"
    name = "#{singular}_name"
    singular_id = "#{singular}_id"
    if params[name_hidden] && params[name_hidden] != params[name]
      params[singular_id] = nil
      params[:organization_id] = nil if singular.to_sym == :company
      params[name_hidden] = params[name]
    end
    table_name = singular.to_s.pluralize
    if scope
      items = search_under scope, table_name
    else
      items = search table_name
    end
    unless params[singular_id].blank?
      target_id = params[singular_id].to_i
      it = items.find {|i| i.id == target_id}
    end
    it ||= items[0]
    return it
  end

  def pass_params(*singulars)
    @pass_params_form = {}
    [
      :return_to,
      :return_field,
      :view,
    ].each do |key|
      @pass_params_form[key] = params[key] if params[key]
    end
    singulars.each do |singular|
      ["#{singular}_id", "#{singular}_name_hidden"].each do |key|
        key = key.to_sym
        @pass_params_form[key] = params[key] if params[key]
      end
    end

    @pass_params_link = @pass_params_form.dup
    [
      :mode,
    ].each do |key|
      @pass_params_link[key] = params[key] if params[key]
    end
    singulars.each do |singular|
      key = "#{singular}_name".to_sym
      @pass_params_link[key] = params[key] if params[key]
    end
  end

  def method_scoping(model_class)
    if model_class == Person
      return {
        :find => {
          :order => "people.name,people.id",
          :readonly => true,
        }
      }
    elsif model_class == PersonalGroup
      return {
        :find => {
          :conditions => {:person_id => User.current.person_id},
          :order => "personal_groups.name,personal_groups.id",
          :readonly => true,
        }
      }
    end
    table_name = model_class.table_name
    return {
      :find => {
        :order => "#{table_name}.name,#{table_name}.id",
        :readonly => true,
      }
    }
  end

  def with_picker_scope(model_class, *rest, &block)
    if model_class
      model_class.__send__(:with_scope, method_scoping(model_class)) do
        if rest.empty?
          block.call
          render unless performed?
        else
          with_picker_scope(*rest, &block)
        end
      end
    end
  end

  def find_like_name(scope, name)
    find_like(scope, "name", name)
  end

  def find_like(scope, column, value)
    escape = '^'
    quoted = scope.connection.quote_like(value, escape)
    return scope.find(:all, :conditions => ["#{column} LIKE ? ESCAPE ?", "%#{quoted}%", escape])
  end

  def lift_by_membership(my_all_list, list)
    my_ids = my_all_list.map(&:id)
    my_list, other_list = list.partition do |item|
      my_ids.include?(item.id)
    end
    return my_list + other_list
  end
end
