/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/05
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.core.local.parser;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.AbstractPolicyModel;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.ui.texteditor.MarkerUtilities;

public abstract class PolicyParser {

	private LocalResource localResource;
	
	public PolicyParser(LocalResource localResource) {
		this.localResource = localResource;
		matchingPattern = Pattern.compile("^(" + getOrPattern(localResource.getModel().listSupportedDirectives()) + ")[\\s]+(.*)"); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	private Pattern matchingPattern = null;

	protected Pattern getMatchingPattern() {
		return matchingPattern;
	}

	protected LocalResource getLocalResource() {
		return localResource;
	}

	protected boolean parse() {

		IFile file = localResource.getFile();
	
		AbstractPolicyModel model = localResource.getModel();
		
		handleParseAboutToStart();
		
		BufferedReader bufferdReader = null;
		InputStreamReader reader = null;
		try {
			model.initialize();
			
			reader = new InputStreamReader(file.getContents());
			bufferdReader = new BufferedReader(reader);
			String line;
			int lineno = 0;
			int offset = 0;
			for (; null != (line = bufferdReader.readLine()); offset += line.length() + 1) {
				lineno++;
				PolicyElement element = parseLine(model, line, lineno, offset);
				if (element != null) {
				}
			}
			
			model.setValid(true);
			model.setSize(offset);
			
			{
				ArrayList<PolicyElement> notifyElementList = new ArrayList<PolicyElement>();
				PolicyElement[] elements = model.getElementArray();
				for (int cnt = 0; cnt < elements.length; cnt++) {
					PolicyElement element = elements[cnt];
					if (element.isCreated()) {
						notifyElementList.add(element);
					} else {
						if (!element.isDeleted() && !element.isChecked()) {
							element.setDeleted(true);
							notifyElementList.add(element);
						}
					}
				}
			}

			updateMarker(localResource);
			
			handleParseCompleted();

		} catch (Exception e) {
			model.setValid(false);
			TomoyoCorePlugin.logException(e);
			return false;
		} finally {
			try {
				if (bufferdReader != null)
					bufferdReader.close();				
			} catch (Exception e) {
			}
			try {
				if (reader != null)
					reader.close();				
			} catch (Exception e) {
			}
		}

		return true;
	}

	private void updateMarker(LocalResource localResource) {
    	IResource resource = localResource.getFile();
    	try {
    		resource.deleteMarkers(null, true, IResource.DEPTH_ZERO);
    	} catch (CoreException e) {
    		TomoyoCorePlugin.logException(e);
    	}
		try {
			PolicyElement[] elements = PolicyCacheManager.getInstance().findElements(resource.getProject(), localResource.getModel().getClass());
			if (elements == null)
				return;
			for (int cnt = 0; cnt < elements.length; cnt++) {
				PolicyElement element = elements[cnt];
				if (element.hasNote()) {
					Map<String, Object> attributes = new HashMap<String, Object>();
					MarkerUtilities.setCharStart(attributes, element.getOffset());
					MarkerUtilities.setCharEnd(attributes, element.getOffset() + 1);
					MarkerUtilities.setLineNumber(attributes, element.getLine());
					MarkerUtilities.setMessage(attributes, element.getNote());
					switch (element.getNoteID()) {
					case PolicyElement.NOTE_INFO:
						attributes.put(IMarker.SEVERITY, new Integer(IMarker.SEVERITY_INFO));
						break;
					case PolicyElement.NOTE_WARNNING:
						attributes.put(IMarker.SEVERITY, new Integer(IMarker.SEVERITY_WARNING));
						break;
					case PolicyElement.NOTE_ERROR:
						attributes.put(IMarker.SEVERITY, new Integer(IMarker.SEVERITY_ERROR));
						break;
					default:
						continue;
					}
					MarkerUtilities.createMarker(resource, attributes,
		                  "org.eclipse.core.resources.problemmarker");
/*
					IMarker marker = resource.createMarker("org.eclipse.core.resources.problemmarker");
					switch (element.getNoteID()) {
					case PolicyElement.NOTE_INFO:
						marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO);
						break;
					case PolicyElement.NOTE_WARNNING:
						marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
						break;
					case PolicyElement.NOTE_ERROR:
						marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
						break;
					}
					marker.setAttribute(IMarker.MESSAGE, element.getNote());
					marker.setAttribute(IMarker.LINE_NUMBER, element.getLine());
*/					
				}
			}
		} catch (CoreException e) {
			TomoyoCorePlugin.logException(e);
		}
	}
	
	protected abstract PolicyElement parseLine(AbstractPolicyModel model, String line, int lineno, int column);

	protected abstract void handleParseAboutToStart();

	protected abstract void handleParseCompleted();

	protected static String getOrPattern(String[] pats) {
		StringBuffer buffer = new StringBuffer();
		for (int cnt = 0; cnt < pats.length; cnt++) {
			buffer.append(pats[cnt]);
			buffer.append("|"); //$NON-NLS-1$
		}
		return buffer.deleteCharAt(buffer.length() - 1).toString();
	}

}
