/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.text.outline;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.AbstractPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.ICoreEditorOutlilePage;
import jp.sourceforge.tomoyo.ui.editor.text.PolicyEditor;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;

public abstract class PolicyOutlinePage extends ContentOutlinePage implements ICoreEditorOutlilePage {

	protected PolicyEditor editor;
	
    /**
     * Create a new instance of the reciver using adapatable
     * as the model.
     */
    public PolicyOutlinePage(PolicyEditor editor) {
    	this.editor = editor;
    }

    //--------------------------------------------------------------------------------------
    // Setup
    //--------------------------------------------------------------------------------------

	/** 
     * Creates the control and registers the popup menu for this page
     * Menu id "org.eclipse.ui.examples.propertysheet.outline"
     */
    public void createControl(Composite parent) {
    	super.createControl(parent);

		setupViewer();
		setupViewerPoupuMenu();
		
    	getControl().addFocusListener(new FocusListener() {
			public void focusGained(FocusEvent e) {
				bring2TopPropertySheet();
			}
			public void focusLost(FocusEvent e) {
			}
    	});
    	
    	getTreeViewer().setInput(getModel());

		restoreUserPreference();
    }
    
	private List<Action> actionList = null;

	public List<Action> getToolBarActions() {
		if (actionList == null) {
			actionList = new ArrayList<Action>();
			actionList.add(expandAction);
			actionList.add(collapseAction);
		}
		return actionList;
	}

	protected void setupViewer() {
		TreeViewer viewer = getTreeViewer();
    	viewer.setContentProvider(createContentProvider());
    	viewer.setLabelProvider(createLabelProvider());
    	viewer.addSelectionChangedListener(this);
    	viewer.getControl().addKeyListener(new ViewerExpandCollapseKeyListener(viewer));
    	viewer.getControl().addMouseListener(new CoreEditorOutlineMouseListener(viewer));
    	viewer.getControl().addFocusListener(new TreeFocusAdapter(viewer));
	}

	private void setupViewerPoupuMenu() {
		TreeViewer viewer = getTreeViewer();
    	MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
    	menuManager.setRemoveAllWhenShown(true);
    	setupMenuBar(menuManager);
		Menu menu = menuManager.createContextMenu(viewer.getTree());
		viewer.getTree().setMenu(menu);
    }
	
	protected void refreshViewer() {
		TreeViewer viewer = getTreeViewer();
		if (viewer == null || viewer.getTree().isDisposed())
			return;
		viewer.refresh(true);
	}

	private void setupMenuBar(MenuManager menuManager) {
		IMenuListener menuListener = createMenuListener();
		if (menuListener != null)
			menuManager.addMenuListener(menuListener);
	}

	protected abstract IMenuListener createMenuListener();

	protected void restoreUserPreference() {
	}

	private void bring2TopPropertySheet() {
		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		/*
		 * note
		 * 
		 * you will have WARNING message below with commented out source.
		 * WARNING: Prevented recursive attempt to activate part org.eclipse.ui.views.PropertySheet while still in the middle of activating part org.eclipse.ui.views.ContentOutline
		 */
		/*
		IViewPart viewPart = page.findView(IPageLayout.ID_PROP_SHEET);
		page.bringToTop(viewPart);
		 */
		try {
			page.showView(IPageLayout.ID_PROP_SHEET, null, IWorkbenchPage.VIEW_VISIBLE);
		} catch (PartInitException e) {
			Activator.logException(e);
		}		
	}

	protected class TreeFocusAdapter extends FocusAdapter {
		private TreeViewer viewer;
		public TreeFocusAdapter(TreeViewer viewer) {
			this.viewer = viewer;
		}
		public void focusGained(FocusEvent e) {
			updateSelectionProvider(viewer);
		}
		private void updateSelectionProvider(TreeViewer viewer) {
			PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart().getSite().setSelectionProvider(viewer);
		}
	}
	
	//--------------------------------------------------------------------------------------
    // Content & label providers
    //--------------------------------------------------------------------------------------
    
    protected abstract PolicyOutlineContentProvider createContentProvider();
    
    protected abstract IBaseLabelProvider createLabelProvider();
    
    //--------------------------------------------------------------------------------------
    // Content model
    //--------------------------------------------------------------------------------------
    
    private Object getModel() {
		return editor.getDocumentModel();
    }

    //--------------------------------------------------------------------------------------
    // Notification
    //--------------------------------------------------------------------------------------

    public void notifyPolicyUpdated() {
		Control control = getTreeViewer().getControl();
		if (control != null && !control.isDisposed()) {
	    	getTreeViewer().refresh();
		}
    }
    
    //--------------------------------------------------------------------------------------
    // Event handlings
    //--------------------------------------------------------------------------------------
    
    /*
     * An flug to avoid call loop.
     */
    private boolean ignoreSelectionChanged = false;
    
	public void updateCursor(int offset) {
		try {	
			ignoreSelectionChanged = true;
			Object model = getModel();
			if (model instanceof AbstractPolicyModel) {
				AbstractPolicyModel document = (AbstractPolicyModel)model;
				PolicyElement element = document.within(offset);
				if (element != null) {
					StructuredSelection currentSelection = (StructuredSelection)getTreeViewer().getSelection();
					if (currentSelection.size() > 1)
						return;
					StructuredSelection newSelection = new StructuredSelection(element);
					getTreeViewer().setSelection(newSelection);
				}
			}
		} finally {
			ignoreSelectionChanged = false;
		}
	}
	
	/*
	 * (non-Javadoc) Method declared on ContentOutlinePage
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		super.selectionChanged(event);
//		storeSelection();
		if (ignoreSelectionChanged)
			return;
		ISelection selection = event.getSelection();
		if (selection.isEmpty()) {
			editor.resetHighlightRange();
			return;
		}
		if (selection instanceof StructuredSelection) {
			StructuredSelection structured = (StructuredSelection)selection;
			Iterator<?> it = structured.iterator();
			while (it.hasNext()) {
				Object obj = it.next();
				if (obj instanceof PolicyElement) {
					PolicyElement element = (PolicyElement)obj;
					if (element instanceof Domain) {
						Domain domain = (Domain)element;
						if (domain.isSkelton())	// ignore skenton element which don't exist in document.
							continue;
					}
					editor.getSelectionProvider().setSelection(new TextSelection(element.getOffset(), element.length()));
					// using 'editor.selectAndReveal' causes outline page flicking.
					editor.selectAndReveal(element.getOffset(), element.length());
				}
			}
		}
	}

	//------------------------------------------------------------------------------------------------
	// Viewer selection management 
	//------------------------------------------------------------------------------------------------
	/*
	private ISelection selectionCache;
	
	private void storeSelection() {
		selectionCache = getTreeViewer().getSelection();
	}

	private void restoreSelection() {
		TreeViewer viewer = getTreeViewer();
		viewer.getTree().setRedraw(false);
		try {
			if (selectionCache == null)
				return;
	    	StructuredSelection selections = (StructuredSelection)selectionCache;
	    	Iterator it = selections.iterator();
	    	while (it.hasNext()) {
	    		PolicyElement element = (PolicyElement)it.next();
	    		if (element.getParent() != null) {
	        		if (element.getParent().getParent() != null)
	        			viewer.setExpandedState(element.getParent().getParent(), true);
	        		viewer.setExpandedState(element.getParent(), true);
	    		}
	    		viewer.setExpandedState(element, true);
	    		viewer.setSelection(selectionCache, true);
	    	}
		} finally {
		}
		viewer.getTree().setRedraw(true);
	}
	*/
	//------------------------------------------------------------------------------------------------
	// Actions
	//------------------------------------------------------------------------------------------------

    protected Action expandAction = new Action("Expand all", Action.AS_PUSH_BUTTON) {
    	public ImageDescriptor getImageDescriptor() {
    		return Activator.getImageDescriptor("expandall.gif"); //$NON-NLS-1$
		}
		public void run() {
    		TreeViewer viewer = getTreeViewer();
    		viewer.getTree().setRedraw(false);
    		viewer.expandAll();
    		viewer.getTree().setRedraw(true);
    	}
    };
    
    protected Action collapseAction = new Action("Collapse all", Action.AS_PUSH_BUTTON) {
    	public ImageDescriptor getImageDescriptor() {
    		return Activator.getImageDescriptor("collapseall.gif"); //$NON-NLS-1$
		}
    	public void run() {
    		TreeViewer viewer = getTreeViewer();
    		viewer.getTree().setRedraw(false);
    		viewer.collapseAll();
    		viewer.getTree().setRedraw(true);
    	}
    };

	//------------------------------------------------------------------------------------------------
	// Utilities
	//------------------------------------------------------------------------------------------------

	protected IFile getFile() {
		return ((FileEditorInput)editor.getEditorInput()).getFile();
	}
	
	protected IProject getProject() {
		return getFile().getProject();
	}


}
