/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form.outline;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.core.Utilities;
import jp.sourceforge.tomoyo.core.local.model.AbstractPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElementDefinition;
import jp.sourceforge.tomoyo.core.local.model.domain.Profile;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileGroup;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.parser.PolicyParserManager;
import jp.sourceforge.tomoyo.core.local.resource.DomainPolicy;
import jp.sourceforge.tomoyo.core.local.resource.ExceptPolicy;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.local.resource.SystemPolicy;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.ICoreEditorOutlilePage;



import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;

/**
 * Page for the content outliner
 */
public class ProjectOutlinePage extends ContentOutlinePage implements ICoreEditorOutlilePage {

	private FormEditor editor = null;
	
    /**
     * Create a new instance of the reciver using adapatable
     * as the model.
     */
	public ProjectOutlinePage(FormEditor editor) {
		super();
		this.editor = editor;
	}
	
	/** 
     * Creates the control and registers the popup menu for this page
     * Menu id "org.eclipse.ui.examples.propertysheet.outline"
     */
    public void createControl(Composite parent) {
		super.createControl(parent);

		setupViewer();
		setupViewerPoupuMenu();
    
		PolicyParserManager.getInstance().addObserver(new PolicyUpdateObserver());
		ProfileManager.getInstance().addObserver(new ProfileUpdateObserver());

    	getTreeViewer().setInput(getModel());

    	restoreUserPreference();
    }
    
	private void restoreUserPreference() {
		if (profileElementModel != null)
			getTreeViewer().expandToLevel(profileElementModel, 1);
	}
	
	private List<Action> actionList = null;

	public List<Action> getToolBarActions() {
		if (actionList == null) {
			actionList = new ArrayList<Action>();
			actionList.add(new ClearNotificationAction());
			actionList.add(new CollapseTreeAction());
		}
		return actionList;
	}

	private void setupViewer() {
		TreeViewer viewer = getTreeViewer();
		viewer.setContentProvider(createContentProvider());
		viewer.setLabelProvider(createLabelProvider());
    	viewer.addSelectionChangedListener(new TreeViewerSelectionChangeLister());
    	viewer.getControl().addKeyListener(new ViewerExpandCollapseKeyListener(viewer));
    	viewer.getControl().addMouseListener(new CoreEditorOutlineMouseListener(viewer));
	}

    private void setupViewerPoupuMenu() {
    	TreeViewer viewer = getTreeViewer();
    	MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
    	menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new ProjectOutlineMenuListener(viewer));
		Menu menu = menuManager.createContextMenu(viewer.getTree());
		viewer.getTree().setMenu(menu);
    }

	private void refreshViewer() {
		TreeViewer viewer = getTreeViewer();
		if (viewer == null || viewer.getTree().isDisposed())
			return;
		viewer.refresh(true);
	}

	//-----------------------------------------------------------------------------------------
	// Actions
	//-----------------------------------------------------------------------------------------

	private class ClearNotificationAction extends Action {
		public ClearNotificationAction() {
			super("", Activator.getImageDescriptor("clear.gif")); //$NON-NLS-1$
			setToolTipText(Messages.ProjectOutlinePage_ClearActionTooltip);
		}
    	public void setChecked(boolean checked) {
			super.setChecked(checked);
		}
		public void run() {
			resetUpdates();
			refreshViewer();
		}
    };

    private void resetUpdates() {
		resetUpdates(true, true);
    }
    
    private void resetUpdates(boolean local, boolean server) {
		ProjectOutlineElement rootModel = (ProjectOutlineElement)getTreeViewer().getInput();
		resetUpdates(rootModel, local, server);
    }
    
	private void resetUpdates(ProjectOutlineElement element, boolean local, boolean server) {
		if (local)
			element.resetLocalUpdated();
		if (server)
			element.clearServerUpdatedCount();
		for (int cnt = 0; cnt < element.getChildCount(); cnt++) {
			ProjectOutlineElement child = element.getChild(cnt);
			resetUpdates(child, local, server);
		}
	}

	private class CollapseTreeAction extends Action {
		public CollapseTreeAction() {
			super("", Activator.getImageDescriptor("collapseall.gif")); //$NON-NLS-1$ //$NON-NLS-2$
			setToolTipText(Messages.ProjectOutlinePage_CollapseActionTooltip);
		}
    	public void setChecked(boolean checked) {
			super.setChecked(checked);
		}
		public void run() {
    		TreeViewer viewer = getTreeViewer();
    		viewer.collapseAll();
		}
    };
    
	//-----------------------------------------------------------------------------------------
	// Observers and threads
	//-----------------------------------------------------------------------------------------

    private class PolicyUpdateObserver implements Observer {
		public void update(Observable o, Object arg) {
			if (o instanceof PolicyParserManager) {
				final LocalResource localResource = (LocalResource)arg;
				Activator.getStandardDisplay().asyncExec(new Runnable() {
					public void run() {
						if (localResource instanceof DomainPolicy)
							domainPoliyModel.clearServerUpdatedCount();
						if (localResource instanceof ExceptPolicy)
							exceptPoliyModel.clearServerUpdatedCount();
						if (!Utilities.is20(getProject())) {
							if (localResource instanceof SystemPolicy)
								systemPoliyModel.clearServerUpdatedCount();
						}
						refreshViewer();
					}
				});
			}
		}
    }
    
    private class ProfileUpdateObserver implements Observer {
		public void update(Observable o, Object arg) {
			Display.getDefault().asyncExec(new ProfileUpdateThread(o, arg));
		}
    }

    private class ProfileUpdateThread implements Runnable {
    	private Observable o;
    	private Object arg;
    	public ProfileUpdateThread(Observable o, Object arg) {
    		this.o = o;
    		this.arg = arg;
    	}
		public void run() {
			if (o instanceof ProfileManager) {
				Hashtable<?, ?> table = (Hashtable<?, ?>)arg;
				IProject project = (IProject)table.get(IProject.class);
				
    			profileElementModel.removeChildren();
    			
    			ProfileGroup[] newGroups = ProfileManager.getInstance().getProjectProfile(project).toArray(true);
    			for (int cnt = 0; cnt < newGroups.length; cnt++) {
    				ProfileGroup newGroup = newGroups[cnt];
   					profileElementModel.addChild(new ProjectOutlineElement(newGroup));
    			}
				refreshViewer();
			}
		}
    }
    
	//-----------------------------------------------------------------------------------------
	// Treeviewer listeners
	//-----------------------------------------------------------------------------------------

    private class TreeViewerSelectionChangeLister implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
	    	StructuredSelection selections = (StructuredSelection)event.getSelection();
	    	ProjectOutlineElement firstElement = (ProjectOutlineElement)selections.getFirstElement();
	    	if (firstElement == null)
	    		;
		}
    }

	//-----------------------------------------------------------------------------------------
	// Models
	//-----------------------------------------------------------------------------------------
    
    private ProjectOutlineElement profileElementModel;
    private ProjectOutlineElement domainPoliyModel;
    private ProjectOutlineElement exceptPoliyModel;
    private ProjectOutlineElement systemPoliyModel;
    
    private Object getModel() {
    	IProject project = getProject();
    	
    	ProjectOutlineElement rootModel = new ProjectOutlineElement(project);
    	
    	profileElementModel = new ProjectOutlineElement(Profile.class);
    	rootModel.addChild(profileElementModel);
    	
		ProfileGroup[] groups = ProfileManager.getInstance().getProjectProfile(project).toArray(true);
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			profileElementModel.addChild(new ProjectOutlineElement(group));
		}
    	
		LocalResource domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		domainPoliyModel = createPolicyModel(rootModel, domainPolicy);
		
		LocalResource exceptPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getExceptPolicy();
		exceptPoliyModel = createPolicyModel(rootModel, exceptPolicy);
		
		if (!Utilities.is20(getProject())) {
			LocalResource systemPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getSystemPolicy();
			systemPoliyModel = createPolicyModel(rootModel, systemPolicy);
		}

		return rootModel;
	}
    
    private ProjectOutlineElement createPolicyModel(ProjectOutlineElement rootModel, LocalResource localResource) {
    	ProjectOutlineElement policyElement = new ProjectOutlineElement(localResource.getModel());
    	rootModel.addChild(policyElement);
    	ArrayList<PolicyElementDefinition> elementDefinitions = localResource.getModel().listElementDefinitions();
    	for (int cnt = 0; cnt < elementDefinitions.size(); cnt++) {
    		PolicyElementDefinition elementDefinition = elementDefinitions.get(cnt);
       		policyElement.addChild(new ProjectOutlineElement(elementDefinition));
    	}
    	return policyElement;
    }

	//-----------------------------------------------------------------------------------------
	// Content providers
	//-----------------------------------------------------------------------------------------
	
	private ProjectContentProvider createContentProvider() {
		return new ProjectContentProvider();
	}

	private class ProjectContentProvider implements ITreeContentProvider {
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof ProjectOutlineElement) {
				ProjectOutlineElement element = (ProjectOutlineElement)parentElement;
				return element.getChildren();
			} else {
				return new Object[0];
			}
		}
		public Object getParent(Object element) {
			return null;
		}
		public boolean hasChildren(Object element) {
			return getChildren(element).length > 0;
		}
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}
		public void dispose() {
		}
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	//-----------------------------------------------------------------------------------------
	// Label providers
	//-----------------------------------------------------------------------------------------
	
	private IBaseLabelProvider createLabelProvider() {
		ProjectLabelProvider labelProvider = new ProjectLabelProvider();
		ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
		return new TableDecoratingLabelProvider(labelProvider, decorator);
	}

	protected class ProjectLabelProvider extends LabelProvider {

		public ProjectLabelProvider() {
		}
		
		public String getText(Object obj) {
			ProjectOutlineElement element = (ProjectOutlineElement)obj;
			String caption = getCaption(element);
			int newCount = getCount(element);
			try {
				if (element.isLocalUpdated()) {
					int oldCount = element.getOldCount();
					return decorateCount(caption, oldCount, newCount);
				} else
					return decorateCount(caption, newCount);
			} finally {
				element.setCount(newCount);
			}
		}
		
		private String getCaption(ProjectOutlineElement element) {
			Object object = element.getObject();
			if (object.equals(Profile.class)) {
				PolicyElementDefinition definition = ProcResourceManager.getInstance().getProcResourceSet(
						getProject()).getDomainPolicy().getModel().findDefinition(Profile.class);
				return definition.getCaption();
			}
			if (object instanceof ProfileGroup) {
				ProfileGroup group = (ProfileGroup)object;
				return group.getLongName();
			}
			if (object instanceof AbstractPolicyModel) {
				AbstractPolicyModel policyModel = (AbstractPolicyModel)object;
				return policyModel.getPolicyModelName();
			}
			if (object instanceof PolicyElementDefinition) {
				PolicyElementDefinition definition = (PolicyElementDefinition)object;
				return definition.getCaption();
			}
			return element.getObject() == null ? null : element.getObject().toString();
		}

		private int getCount(ProjectOutlineElement element) {
			Object object = element.getObject();
			IProject project = getProject();
			if (object.equals(Profile.class)) {
				return ProfileManager.getInstance().getProjectProfile(project).getGroupCount();
			}
			if (object instanceof ProfileGroup) {
				ProfileGroup group = (ProfileGroup)object;
				return PolicyCacheManager.getInstance().countDomain(project, group);
			}
			if (object instanceof AbstractPolicyModel) {
				AbstractPolicyModel policyModel = (AbstractPolicyModel)object;
				return PolicyCacheManager.getInstance().count(project, policyModel.getClass(), false);
			}
			if (object instanceof PolicyElementDefinition) {
				PolicyElementDefinition definition = (PolicyElementDefinition)object;
				return PolicyCacheManager.getInstance().count(project, definition.getElementClass(), false);
			}
			return 0;
		}

		private String decorateCount(String elementName, int count) {
			return elementName + " (" + count + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		private String decorateCount(String elementName, int oldCount, int newCount) {
			return elementName + " (" + oldCount + " > " + newCount + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		
		public Image getImage(Object obj) {
			ProjectOutlineElement element = (ProjectOutlineElement)obj;
			Object object = element.getObject();
			if (object.equals(Profile.class)) {
				PolicyElementDefinition definition = ProcResourceManager.getInstance().getProcResourceSet(
						getProject()).getDomainPolicy().getModel().findDefinition(Profile.class);
				return definition.getImage();
			}
			if (object instanceof AbstractPolicyModel) {
				AbstractPolicyModel policyModel = (AbstractPolicyModel)object;
				return policyModel.getImage();
			}
			if (object instanceof ProfileGroup) {
//				ProfileGroup group = (ProfileGroup)object;
				return Activator.getImage("policy_element.gif"); //$NON-NLS-1$
			}
			if (object instanceof PolicyElementDefinition) {
				PolicyElementDefinition definition = (PolicyElementDefinition)object;
				return definition.getImage();
			}
			return null;
		}
	}
	
	private class TableDecoratingLabelProvider extends DecoratingLabelProvider {

		private LabelProvider provider;
		private ILabelDecorator decorator;

		/**
		 * @param provider
		 * @param decorator
		 */
		public TableDecoratingLabelProvider(ILabelProvider provider, ILabelDecorator decorator) {
			super(provider, decorator);
			this.provider = (LabelProvider)provider;
			this.decorator = decorator;
		}

		public Image getColumnImage(Object element, int columnIndex) {
			Image image = provider.getImage(element);
			if (decorator != null) {
				Image decorated = decorator.decorateImage(image, element);
				if (decorated != null) {
					return decorated;
				}
			}
			return image;
		}

		public String getColumnText(Object element, int columnIndex) {
			String text = provider.getText(element);
			if (decorator != null) {
				String decorated = decorator.decorateText(text, element);
				if (decorated != null) {
					return decorated;
				}
			}
			return text;
		}

	}
	
	protected IProject getProject() {
		return ((FileEditorInput)editor.getEditorInput()).getFile().getProject();
	}

}
