/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form.domain;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.ui.editor.Activator;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.PlatformUI;


public class DomainSearchDialog extends TitleAreaDialog implements ModifyListener, SelectionListener {

	private static DomainSearchDialog instance;
	
	public static DomainSearchDialog getInstance(Shell parentShell) {
		if (instance == null)
			instance = new DomainSearchDialog(parentShell);
		return instance;
	}
	
	private DomainSearchDialog(Shell parentShell) {
		super(parentShell);
	}

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.DomainSearchDialog_ShellTitle);
    }
    
	protected int getShellStyle() {
//		return SWT.MAX | SWT.RESIZE;
		return SWT.SHELL_TRIM;
	}

	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.DomainSearchDialog_DialogTitle);
		setTitleImage(Activator.getImage("search_wiz.gif")); //$NON-NLS-1$
		setMessage(Messages.DomainSearchDialog_DialogMessage);
		
		Composite composite = (Composite)super.createDialogArea(parent);

		Composite innerComposite = new Composite(composite, SWT.NULL);
		innerComposite.setLayout(new GridLayout());
		innerComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		createRegularSearchComposite(innerComposite);
		createHistorySearchComposite(innerComposite);
	    
		restoreDefault();

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
			"jp.sourceforge.tomoyo.doc.open_search_domain");		 //$NON-NLS-1$

		return composite;
	}
	
	protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
		if (id == 0)
			return super.createButton(parent, id, Messages.DomainSearchDialog_CloseButtonText, defaultButton);
		else
			return null;
	}

	protected Control createRegularSearchComposite(Composite parent) {
		Group container = new Group(parent, SWT.NULL);
		container.setText(Messages.DomainSearchDialog_NormalSearchSectionTitle);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		container.setLayout(layout);
	    container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
	    
		createSearchCombo(container);
//		createSearchControl1(container);
		createGroups(container);
		createSearchButton(container);
		
	    return container;
	}

	private TableViewer historyViewer;
	
	private Control createHistorySearchComposite(Composite parent) {
		Group container = new Group(parent, SWT.NULL);
		container.setText(Messages.DomainSearchDialog_HistorySearchSectionTitle);
		container.setBackground(parent.getBackground());
		
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;

		container.setLayout(layout);
		container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	    
		historyViewer = new TableViewer(container, SWT.BORDER);
	    
		Table table = historyViewer.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
	    
		TableColumn column = new TableColumn(table,SWT.LEFT);
		column.setText(Messages.DomainSearchDialog_ColumnSearchKeyword);
		column.setWidth(300);
		column.setAlignment(SWT.LEFT);
	    
		column = new TableColumn(table,SWT.LEFT);
		column.setText(Messages.DomainSearchDialog_ColumnSearchMethod);
		column.setWidth(200);
		column.setAlignment(SWT.LEFT);
	    
		historyViewer.setContentProvider(new HistoryViewerContentProvider());
		historyViewer.setLabelProvider(new HistoryViewerLabelProvider());
/*
	    historyViewer.addSelectionChangedListener(new HistoryViewerSelectionChangedListener(viewer));
    	viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleSelectionChanged((TreeViewer)event.getSource());
			}
    	});
 * 
 */
		historyViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				handleViewerDoubleClick(event);
			}
		});

		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		gridData.heightHint = 200;
		historyViewer.getControl().setLayoutData(gridData);
	    
		return container;
	}
	
	private void restoreDefault() {
		restoreHistory();
	    
		if (searchCombo.getItemCount() > 0 ) {
			String lastKeyword = searchCombo.getItem(0);
	    	searchCombo.setText(lastKeyword);
		}
	    
		historyViewer.setInput(searchCombo);
	    
		downwardButton.setSelection(true);
		matchExecFullPathButton.setSelection(true);
		
		searchButton.setFocus();
	}

	public int open() {
		return super.open();
	}

	private Combo searchCombo;
	
	private void createSearchCombo(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		container.setLayoutData(data);
		Label searchLabel = new Label(container, SWT.NULL);
		searchLabel.setText(Messages.DomainSearchDialog_SearchLabelText);
		searchCombo = new Combo(container, SWT.NULL);
		searchCombo.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		searchCombo.addModifyListener(this);
	}

	private void createGroups(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		container.setLayoutData(data);
		createOrientationGroup(container);
		createMatchingStrategyGroup(container);
	}
	
	private Button upwardButton;
	private Button downwardButton;
	
	private void createOrientationGroup(Composite parent) {
		Group group = new Group(parent, SWT.NULL);
		group.setText(Messages.DomainSearchDialog_SearchOrientationGroupText);
		GridLayout layout = new GridLayout();
		group.setLayout(layout);
		group.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		upwardButton = new Button(group, SWT.RADIO);
		upwardButton.setText(Messages.DomainSearchDialog_SearchOrientationUpwardText);
		downwardButton = new Button(group, SWT.RADIO);
		downwardButton.setText(Messages.DomainSearchDialog_SearchOrientationDownwardText);
	}
	
	private Button matchExecFullPathButton;
	private Button matchExecPathButton;
	private Button includeButton;

	private Hashtable<Object, Button> buttonStore = new Hashtable<Object, Button>();
	
	private void createMatchingStrategyGroup(Composite parent) {
		Group group = new Group(parent, SWT.NULL);
		group.setText(Messages.DomainSearchDialog_MatchingStrategyGroupText);
		GridLayout layout = new GridLayout();
		group.setLayout(layout);
		group.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		
		matchExecFullPathButton = new Button(group, SWT.RADIO);
		matchExecFullPathButton.setText(Messages.DomainSearchDialog_MatchingStrategyFullPathText);
		matchExecFullPathButton.setData("fullpath"); //$NON-NLS-1$
		matchExecPathButton = new Button(group, SWT.RADIO);
		matchExecPathButton.setText(Messages.DomainSearchDialog_MatchingStrategyExecPathText);
		matchExecPathButton.setData("execpath"); //$NON-NLS-1$
		includeButton = new Button(group, SWT.RADIO);
		includeButton.setText(Messages.DomainSearchDialog_MatchingStrategyIncludeText);
		includeButton.setData("includes"); //$NON-NLS-1$
		
		buttonStore.put(matchExecFullPathButton.getData(), matchExecFullPathButton);
		buttonStore.put(matchExecPathButton.getData(), matchExecPathButton);
		buttonStore.put(includeButton.getData(), includeButton);
	}

	private Button searchButton;
	
	private void createSearchButton(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		container.setLayoutData(data);

		Label spacer = new Label(container, SWT.NULL);
		spacer.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
	    
		searchButton = new Button(container, SWT.NULL);
		searchButton.setText(Messages.DomainSearchDialog_SearchButtonText);
		GridData layoutData = new GridData();
		layoutData.widthHint = 150;
		searchButton.setLayoutData(layoutData);
		searchButton.addSelectionListener(this);
	}

	protected Point getInitialSize() {
		return new Point(600, 500);
	}

	private class HistoryViewerContentProvider implements IStructuredContentProvider {
		private Object newInput = null;
		public HistoryViewerContentProvider() {
		}
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			this.newInput = newInput;
		}
		public void fireFilterChanged() {
		}
		public void dispose() {
		}
		public Object[] getElements(Object parent) {
			Combo combo = (Combo)newInput;
			String[] texts = combo.getItems();
			String[] types = (String[])combo.getData();
			ArrayList<String[]> list = new ArrayList<String[]>();
			for (int cnt = 0; cnt < texts.length; cnt++) {
				list.add(new String[] {texts[cnt], types[cnt]});
			}
			return list.toArray();
		}
	}
	
	private class HistoryViewerLabelProvider extends LabelProvider implements ITableLabelProvider, ITableColorProvider {
		public String getColumnText(Object obj, int index) {
			String[] strArr = (String[])obj;
			if (index == 0)
				return strArr[index];
			if (index == 1) {
				String typeName = strArr[index];
				Button button = (Button)buttonStore.get(typeName);
				return button.getText();
			}
			return null;
		}
		public Image getColumnImage(Object obj, int index) {
			return getImage(obj);
		}
		public Image getImage(Object obj) {
			return null;
		}
		public Color getForeground(Object element, int columnIndex) {
			return null;
		}
		public Color getBackground(Object element, int columnIndex) {
			return null;
		}
	}
	
	//---------------------------------------------------------------------------------------
	// Observable object
	//---------------------------------------------------------------------------------------

	private Observable observable = new Observable() {
		public void notifyObservers(Object arg) {
			super.setChanged();
			super.notifyObservers(arg);
		}
	};
	
	public void addObserver(Observer o) {
		observable.addObserver(o);
	}
	
	//---------------------------------------------------------------------------------------
	// Listeners
	//---------------------------------------------------------------------------------------

	public void modifyText(ModifyEvent e) {
		searchButton.setEnabled(searchCombo.getText().trim().length() > 0);
	}

	private void handleViewerDoubleClick(DoubleClickEvent event) {
		TableViewer viewer = (TableViewer)event.getViewer();
		StructuredSelection selection = (StructuredSelection)viewer.getSelection();
		if (selection.size() == 1) {
			String[] strArr = (String[])selection.getFirstElement();
			Hashtable<String, Object> table = new Hashtable<String, Object>();
			table.put("search", strArr[0]); //$NON-NLS-1$
			table.put("method", strArr[1]); //$NON-NLS-1$ //$NON-NLS-2$
			table.put("orient", Boolean.valueOf(upwardButton.getSelection())); //$NON-NLS-1$
			observable.notifyObservers(table);
		}
	}

	public void widgetDefaultSelected(SelectionEvent e) {
	}
	
	public void widgetSelected(SelectionEvent e) {
		Hashtable<String, Object> table = new Hashtable<String, Object>();
		String text = searchCombo.getText();
		String type = null;
		table.put("search", text); //$NON-NLS-1$
		if (matchExecFullPathButton.getSelection())
			type = (String)matchExecFullPathButton.getData();
		if (matchExecPathButton.getSelection())
			type = (String)matchExecPathButton.getData();
		if (includeButton.getSelection())
			type = (String)includeButton.getData();
		table.put("method", type); //$NON-NLS-1$ //$NON-NLS-2$
		table.put("orient", Boolean.valueOf(upwardButton.getSelection())); //$NON-NLS-1$

		observable.notifyObservers(table);

		storeHistory(text, type);
		historyViewer.setInput(searchCombo);
	}

	private static final String KEY_SEARCH_TEXT_HISTORY = "search.text2"; //$NON-NLS-1$
	private static final String KEY_SEARCH_TYPE_HISTORY = "search.type2"; //$NON-NLS-1$
	
	private void storeHistory(String text, String type) {
		if (hasHistory(text, type))
			return;
		String[] textSrc = searchCombo.getItems();
		String[] typeSrc = (String[])searchCombo.getData();
		if (typeSrc == null)
			typeSrc = new String[0];
		String[] textDest;
		String[] typeDest;
		if (searchCombo.getItemCount() > 9) {
			textDest = new String[10];
        	textDest[0] = text;
        	System.arraycopy(textSrc, 0, textDest, 1, 9);
        	typeDest = new String[10];
        	typeDest[0] = type;
        	System.arraycopy(typeSrc, 0, typeDest, 1, 9);
		} else {
        	textDest = new String[searchCombo.getItemCount() + 1];
        	textDest[0] = text;
        	System.arraycopy(textSrc, 0, textDest, 1, searchCombo.getItemCount());
        	typeDest = new String[searchCombo.getItemCount() + 1];
        	typeDest[0] = type;
        	System.arraycopy(typeSrc, 0, typeDest, 1, searchCombo.getItemCount());
		}
		IDialogSettings settings = Activator.getDefault().getDialogSettings(getClass().getName());
		settings.put(KEY_SEARCH_TEXT_HISTORY, textDest);
		settings.put(KEY_SEARCH_TYPE_HISTORY, typeDest);
		searchCombo.setItems(textDest);
		searchCombo.setText(text);
		searchCombo.setData(typeDest);
	}
	
	private boolean hasHistory(String text, String type) {
		for (int cnt = 0; cnt < searchCombo.getItemCount(); cnt++) {
			String[] types = (String[])searchCombo.getData();
			if (text.equals(searchCombo.getItem(cnt))) {
				if (types[cnt].equals(type))
					return true;
			}
		}
		return false;
	}	

	private void restoreHistory() {
        IDialogSettings settings = Activator.getDefault().getDialogSettings(getClass().getName());
        if (settings == null)
        	return;
        String[] textArrary = settings.getArray((KEY_SEARCH_TEXT_HISTORY));
        if (textArrary == null)
        	return;
        searchCombo.setItems(textArrary);
        String[] typeArrary = settings.getArray((KEY_SEARCH_TYPE_HISTORY));
        if (typeArrary == null)
        	return;
        searchCombo.setData(typeArrary);
	}

	public void notifyFound(int count) {
		if (count == 0)
			setMessage(Messages.DomainSearchDialog_TextNotFound, IMessageProvider.WARNING);
		else
			setMessage(count + Messages.DomainSearchDialog_MatchFoundMessage_suffix, IMessageProvider.INFORMATION);
	}
	
}
