/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form;

import java.util.List;
import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.ServerAction;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.CoreEditor;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;

public abstract class ResourceFormPage extends PlainFormPage {

	public ResourceFormPage(FormEditor editor, String id, String title) {
		super(editor, id, title);
	}
	
	protected void setupObservers() {
		super.setupObservers();
		
		ProcResourceManager.getInstance().addObserver(new LocalResourceUpdateObserver());
	}

	protected abstract LocalResource getLocalResource();

	protected List<Action> getToolBarActions() {
		List<Action> actionList = super.getToolBarActions();
		actionList.add(0, createOpenEditorAction(getLocalResource()));
		actionList.add(0, createSynchronizeAction(getLocalResource()));
		return actionList;
	}

	// !!! important !!!  Without overwriting this method, you have spreading form area.
	protected void createToolBarActions(IManagedForm managedForm) {
		super.createToolBarActions(managedForm);
	}
	
	protected Button createButton(Composite parent, Image image, int style) {
		Button button = new Button(parent, style);
		button.setImage(image);
		button.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		handleButtonSelected((Button)e.getSource());
        	}
        });
		GridData data = new GridData();
		data.horizontalIndent = 0;
		data.verticalIndent = 0;
		button.setLayoutData(data);
		return button;
    }
	
	protected void handleButtonSelected(Button button) {
	}

	protected Text createText(Composite parent, int style) {
		Text text = new Text(parent, style);
		text.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				handleModifyText((Text)e.getSource());
			}
        });
		GridData data = new GridData();
		data.horizontalIndent = 0;
		data.verticalIndent = 0;
		text.setLayoutData(data);
		return text;
    }

	protected void handleModifyText(Text text) {
	}

	//-----------------------------------------------------------------------------------------
	// Actions
	//-----------------------------------------------------------------------------------------

	protected OpenEditorAction openEditorAction = null;
	
	private Action createOpenEditorAction(LocalResource localResource) {
		if (openEditorAction == null)
			openEditorAction = new OpenEditorAction(localResource);
		return openEditorAction;
	}

	protected class OpenEditorAction extends Action {
		private LocalResource localResource;
		public OpenEditorAction(LocalResource localResource) {
			super("", Action.AS_PUSH_BUTTON);
			this.localResource = localResource;
			setToolTipText(Messages.PlainFormPage_FormToolBarActionOpenEditorToolTip);
			setImageDescriptor(Activator.getImageDescriptor("edit.gif")); //$NON-NLS-1$
		}
		public void run() {
			CoreEditor editor = (CoreEditor)getEditor();
			editor.openEditorPage(localResource);
		}
	}
	
	protected SynchronizeAction syncAction = null;
	
	private Action createSynchronizeAction(LocalResource localResource) {
		if (syncAction == null)
			syncAction = new SynchronizeAction(localResource);
		return syncAction;
	}

	protected class SynchronizeAction extends ServerAction {
		private LocalResource localResource;
		public SynchronizeAction(LocalResource localResource) {
			super(getProject(), Action.AS_UNSPECIFIED);
			this.localResource = localResource;
			setToolTipText(Messages.PlainFormPage_FormToolBarActionSynchronizeToolTip);
			setImageDescriptor(Activator.getImageDescriptor("synchronize.gif")); //$NON-NLS-1$
		}
		public void run() {
			boolean syncOK = ProcResourceManager.getInstance().synchronize(getProject(), localResource);
			if (syncOK)
				;
			else
				;
		}
	}
	
	//-----------------------------------------------------------------------------------------
	// Utililties (viewer selection management)
	//-----------------------------------------------------------------------------------------
	
	protected void setSelection(TreeViewer viewer, PolicyElement element) {
		if (viewer == null)
			return;
		StructuredSelection selection = new StructuredSelection(element);
		try {
			viewer.getTree().setRedraw(false);
			viewer.setSelection(selection);
			viewer.reveal(selection.getFirstElement());
		} finally {
			viewer.getTree().setRedraw(true);
		}		
	}

	protected void storeViewerSelectioned(Viewer viewer, StructuredSelection selection) {
		if (selection == null || selection.size() == 0)
			return;
		viewer.setData("selectioned", selection);
	}

	protected void restoreViewerSelectioned(TreeViewer viewer) {
		StructuredSelection selection = (StructuredSelection)viewer.getData("selectioned");
		if (selection == null)
			return;
		Object object = selection.getFirstElement();

		if (object instanceof PolicyElement) {
			PolicyElement element = (PolicyElement)object;
			
			PolicyElement foundElement = PolicyCacheManager.getInstance().findElement(getProject(), element);
			if (foundElement == null)
				return;

			viewer.setSelection(new StructuredSelection(foundElement), true);
		} else {
			viewer.setSelection(new StructuredSelection(object), true);
		}
	}
	
	//-----------------------------------------------------------------------------------------
	// Notifications
	//-----------------------------------------------------------------------------------------
	
	private class LocalResourceUpdateObserver implements Observer {
    	public void update(Observable o, Object arg) {
			final LocalResource localResource = (LocalResource)arg;
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					notifyLocalResourceUpdated(localResource);
				}
			});
		}
    }
    
	protected void notifyLocalResourceUpdated(LocalResource localResource) {
		setDirty(true);
	}

	protected void notifyConnectionUpdated(IProject project, boolean isConnected) {
		super.notifyConnectionUpdated(project, isConnected);
		if (syncAction != null) {
			syncAction.setEnabled(isConnected);
		}
	}

	public void notifyPageChanged(PlainFormPage editorPage) {
		super.notifyPageChanged(editorPage);
		if (editorPage == this) {
			back2homeAction.setEnabled(true);
		}
	}
	
    //-----------------------------------------------------------------------------------------
	// Selection provider related
	//-----------------------------------------------------------------------------------------

	protected class TreeFocusAdapter extends FocusAdapter {
		private TreeViewer viewer;
		public TreeFocusAdapter(TreeViewer viewer) {
			this.viewer = viewer;
		}
		public void focusGained(FocusEvent e) {
			updateSelectionProvider(viewer);
		}
	}
	
	private TreeViewer lastFocusedViewer;
	
	protected void updateSelectionProvider(TreeViewer viewer) {

		if (viewer == lastFocusedViewer)
			return;
		lastFocusedViewer = viewer;

		if (PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().isPageZoomed()) {
		} else {
			showPropertySheet();
		}
		
		getSite().setSelectionProvider(viewer);
	}

	private void showPropertySheet() {
		try {
			PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().showView(IPageLayout.ID_PROP_SHEET);
		} catch (PartInitException e) {
			Activator.logException(e);
		}
	}
	
    //-----------------------------------------------------------------------------------------
	// Dirty marker
	//-----------------------------------------------------------------------------------------
	
	private boolean dirty = false;
	
	public boolean isDirty() {
		return dirty;
	}

	protected void setDirty(boolean value) {
//		dirty = value;
		if (dirty) {
//			setPartName("###");
//			firePropertyChange(PROP_TITLE);
		} else {
			;
		}
//		firePropertyChange(PROP_DIRTY);
//		notifyDirtyStateChanged(dirty);
	}
	
	protected void notifyDirtyStateChanged(boolean isDirty) {
	}
	
}
