/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.dialog;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Comparator;

import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.model.status.ProjectProfile;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.ConcreteCommand;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.form.domain.DomainPolicyPage;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.PolicyOutlineMenuListener;


import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;

public class CcsTreeDialog extends TitleAreaDialog {

	private IProject project;
	
	public CcsTreeDialog(Shell parentShell, IProject project) {
		super(parentShell);
		this.project = project;
	}
	
	private IProject getProject() {
		return project;
	}
	
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID,
            IDialogConstants.OK_LABEL, true);
    }

    protected void configureShell(Shell shell) {
    	super.configureShell(shell);
    	shell.setText(Messages.CcsTreeDialog_ShellTitle);
    	shell.setImage(Activator.getImage("new.gif")); //$NON-NLS-1$
    }

    /**
     * [Known problem]
     * In dual Monitor environments, dialog spreads to the maxmum size of current display.
     */
    protected Point getInitialSize() {
    	int w_percent = 50;
    	int h_percent = 85;
    	Rectangle rect = Activator.getStandardDisplay().getClientArea();
    	
    	int width = rect.width * w_percent / 100;
    	int height = rect.height * h_percent / 100;
    	
    	return new Point(width, height);
    }
	
	protected int getShellStyle() {
		return SWT.MAX | SWT.RESIZE;
	}

	private static final String DEFAULT_MESSAGE = Messages.CcsTreeDialog_DialogMessage;
	
	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.CcsTreeDialog_DialogTitle);
		setTitleImage(Activator.getImage("newelm_wiz.gif")); //$NON-NLS-1$
		setMessage(DEFAULT_MESSAGE, IMessageProvider.NONE);

		Composite dialogArea = (Composite)super.createDialogArea(parent);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		dialogArea.setLayoutData(data);

		createCCSTreeComposite(dialogArea);
		
		/*
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
			"jp.sourceforge.tomoyo.doc.open_ccstree_dialog");		
		*/
		
		restoreDefault();
	    
		return dialogArea;
	}
	
	private void restoreDefault() {
		ccstreeViewer.setInput(getInput());
    	ccstreeViewer.expandAll();
	}


	private Object getInput() {
		ConcreteCommand command = CommandManager.getInstance().exec(
				project,
				CommandManager.createCCSToolPath(project, CommandManager.CMD_CCSTREE)
				);
		if (command.isSuccessful()) {
			return command.getSTDOut();
		}
		return null;
	}
	
	//------------------------------------------------------------------------------------------------
	// Candidate composite
	//------------------------------------------------------------------------------------------------
	
	private TreeViewer ccstreeViewer;
	
	protected Control createCCSTreeComposite(Composite parent) {
		Composite container = new Composite(parent, SWT.BORDER);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = false;
		container.setLayout(layout);
	    GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
	    container.setLayoutData(data);
		
	    ccstreeViewer = new TreeViewer(container, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
	    ccstreeViewer.setContentProvider(new CCSTreeViewerContentProvider());
	    ccstreeViewer.setLabelProvider(new CCSTreeViewerLabelProvider(project));
	    ccstreeViewer.addSelectionChangedListener(new CCSTreeViewerSelectionChangeLister());
	    ccstreeViewer.getTree().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	    ccstreeViewer.getControl().addKeyListener(new ViewerExpandCollapseKeyListener(ccstreeViewer));
	    ccstreeViewer.getControl().addMouseListener(new CCSTreeViewerMouseListener(ccstreeViewer));
	    ccstreeViewer.setSorter(new CCSTreeViewerSorter());

	    createCCSTreeContoler(container);
	    
	    setupMenuListener(container, ccstreeViewer);

	    return container;
	}
	
	private class Process {
		private int level;
		private int profile;
		private String process;
		private int pid;
		private String domain;
		public Process(Process init, Process prev, String line) {
			super();
			int idxTrunk = line.indexOf("+- "); //$NON-NLS-1$
			if (idxTrunk > 0) {
				int blank = idxTrunk - 5;
				level = (blank / 4) + 1;
			} else {
				level = 0;
			}
			profile = Integer.parseInt(line.substring(0, 3).trim());
			if (init == null) {
				process = line.split("[\\s]+")[2]; //$NON-NLS-1$
			} else {
				process = line.split("[\\s]+")[3]; //$NON-NLS-1$
			}
			pid = getPID(line);
			domain = getDomain(line);
			
			if (init == null) {
			} else {
				if (level == prev.getLevel()) {
					prev.getParent().addChild(this);
				} else {
					if (level > prev.getLevel()) {
						prev.addChild(this);
					} else {
						init.addChild(this);
					}
				}
			}
		}
		private int getPID(String line) {
			int str = line.indexOf(" (") + 2; //$NON-NLS-1$
			int end = line.indexOf(") "); //$NON-NLS-1$
			return Integer.parseInt(line.substring(str, end));
		}
		private String getDomain(String line) {
			int end = line.indexOf(") ") + 2; //$NON-NLS-1$
			return line.substring(end);
		}
		public int getLevel() {
			return level;
		}
		public int getProfile() {
			return profile;
		}
		public String getProcess() {
			return process;
		}
		public int getPid() {
			return pid;
		}
		public String getDomain() {
			return domain;
		}
		private Process parent;
		private void setParent(Process parent) {
			this.parent = parent;
		}
		private Process getParent() {
			return parent;
		}
		private ArrayList<Process> children = new ArrayList<Process>();
		public void addChild(Process child) {
			children.add(child);
			child.setParent(this);
		}
		public ArrayList<Process> getChildren() {
			return children;
		}
	}
	
	private class CCSTreeViewerContentProvider implements ITreeContentProvider {
		public Object[] getChildren(Object parentElement) {
			ArrayList<Process> children = new ArrayList<Process>();
			if (parentElement instanceof String) {
				Process init = (Process)lines.get(0);
				children.add(init);
				return children.toArray();
			}
			if (parentElement instanceof Process) {
				Process process = (Process)parentElement;
				return process.getChildren().toArray();
			}
			return children.toArray();
		}
		public Object getParent(Object element) {
			return null;
		}
		public boolean hasChildren(Object element) {
			return getElements(element).length > 0;
		}
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}
		public void dispose() {
		}
		private ArrayList<Process> lines = new ArrayList<Process>();
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			String stdout = (String)newInput;
			if (stdout == null)
				return;
			
			lines.clear();

			StringReader reader = new StringReader(stdout);
			BufferedReader bufferdReader = new BufferedReader(reader);
			Process init = null;
			Process prev = null;
			String line;
			try {
				for (; null != (line = bufferdReader.readLine());) {
					Process process = new Process(init, prev, line);
					if (init == null)
						init = process;
					prev = process;
					lines.add(process);
				}
			} catch (IOException e) {
				Activator.logException(e);
			}
		}
	}
	
	private class CCSTreeViewerLabelProvider extends DomainPolicyOutlineLabelProvider implements IColorProvider {

		public CCSTreeViewerLabelProvider(IProject project) {
			super(project);
		}

		public Image getImage(Object object) {
			if (object instanceof Process) {
				Process process = (Process)object;
				Domain domain = PolicyCacheManager.getInstance().findDomain(project, process.getDomain());
				if (domain == null)
					return Activator.getImage("domain_notfound.gif"); //$NON-NLS-1$
				else
					return super.getImage(domain);
			}
			return null;
		}

		public String getText(Object object) {
			if (object instanceof Process) {
				Process process = (Process)object;
				StringBuffer text = new StringBuffer();
				text.append(process.getProcess());
				text.append(" "); //$NON-NLS-1$
				text.append("("); //$NON-NLS-1$
				text.append(process.getPid());
				text.append(")"); //$NON-NLS-1$
				if (showProfile) {
					text.append(" "); //$NON-NLS-1$
					text.append("["); //$NON-NLS-1$
					text.append(process.getProfile());
					text.append("]"); //$NON-NLS-1$
				}
				if (showDomain) {
					text.append(" "); //$NON-NLS-1$
					text.append(process.getDomain());
				}
				return text.toString();
			}
			return super.getText(object);
		}

		private boolean showDomain;
		
		public void showDomain(boolean b) {
			this.showDomain = b;
		}

		private boolean showProfile;
		
		public void showProfile(boolean b) {
			this.showProfile = b;
		}

		private boolean enableColoring;
		
		public void setColoringEnabled(boolean b) {
			this.enableColoring = b;
		}
		
		public Color getBackground(Object element) {
			if (enableColoring)
				return getColor(element, false);
			else
				return null;
		}

		public Color getForeground(Object element) {
			if (enableColoring)
				return getColor(element, true);
			else
				return null;
		}
		
		private Color getColor(Object object, boolean foreGround) {
			if (object instanceof Process) {
				Process process = (Process)object;
				int groupNo = process.getProfile();
				ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(project);
				RGB rgb;
				if (foreGround)
					rgb = pp.getGroup(groupNo).getForecolor();
				else
					rgb = pp.getGroup(groupNo).getBackcolor();
				if (rgb == null)
					return null;
				return Activator.getDefault().getColor(rgb);
			}
			return null;
		}
		
	}
	
	private class CCSTreeViewerSelectionChangeLister implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
			TreeViewer viewer = (TreeViewer)event.getSource();
			ISelection selection = viewer.getSelection();
			Object object = ((IStructuredSelection)selection).getFirstElement();
			if (object == null)
				return;
			Process process = (Process)object;
			if (parent != null) {
				String strDomain = process.getDomain();
				Domain domain = PolicyCacheManager.getInstance().findDomain(project, strDomain);
				if (domain == null)
					setMessage(Messages.CcsTreeDialog_DomainNotFoundMessage + " : " + strDomain, IMessageProvider.WARNING); //$NON-NLS-1$
				else {
					setMessage(DEFAULT_MESSAGE, IMessageProvider.NONE);
					parent.setSelection(domain);
				}
			}
		}
	}
	
	private class CCSTreeViewerMouseListener implements MouseListener {

		private TreeViewer viewer;
		
		public CCSTreeViewerMouseListener(TreeViewer viewer) {
			this.viewer = viewer;
		}
		
		public void mouseDoubleClick(MouseEvent e) {
			if (e.button == 3)	// ignore right double click
				return;
			ISelection selection = viewer.getSelection();
			Object object = ((IStructuredSelection)selection).getFirstElement();
			if (viewer.getExpandedState(object))
				viewer.collapseToLevel(object, 1);
			else
				viewer.expandToLevel(object, 1);
		}

		public void mouseDown(MouseEvent e) {
		}
		
		public void mouseUp(MouseEvent e) {
		}

	}

	private class CCSTreeViewerSorter extends ViewerSorter implements Comparator<Object> {
		private boolean enabled;
		public CCSTreeViewerSorter() {
		}
		public int compare(Viewer viewer, Object e1, Object e2) {
			return compare(e1, e2);
		}
		public int compare(Object e1, Object e2) {
			Process p1 = (Process)e1;
			Process p2 = (Process)e2;
			if (enabled)
				return p1.getProcess().compareTo(p2.getProcess());
			else
				return 0;
		}
		public void setEnabled(boolean b) {
			this.enabled = b;
		}
	}
	
	private Button expand;
	private Button collapse;
	
	private Button showDomain;
	private Button showProfile;
	private Button coloring;
	
	private Button enableSort;
	private Button refresh;
	
	private void createCCSTreeContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout(1, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.marginLeft = 0;
		layout.marginRight = 0;
		container.setLayout(layout);
		GridData data = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
		container.setLayoutData(data);
		
		expand = this.createButton(container, Activator.getImage("expandall.gif"), SWT.NULL); //$NON-NLS-1$
		expand.setToolTipText(Messages.CcsTreeDialog_ExpandTreeButtonTooltip);
		collapse = this.createButton(container, Activator.getImage("collapseall.gif"), SWT.NULL); //$NON-NLS-1$
		collapse.setToolTipText(Messages.CcsTreeDialog_CollapseTreeButtonTooltip);
		
		showDomain = this.createButton(container, Activator.getImage("domain.gif"), SWT.TOGGLE); //$NON-NLS-1$
		showDomain.setToolTipText(Messages.CcsTreeDialog_ShowDomainButtonTooltip);
		showProfile = this.createButton(container, Activator.getImage("profile.gif"), SWT.TOGGLE); //$NON-NLS-1$
		showProfile.setToolTipText(Messages.CcsTreeDialog_ShowProfileButtonTooltip);
		coloring = createButton(container, Activator.getImage("coloring.gif"), SWT.TOGGLE); //$NON-NLS-1$
		coloring.setToolTipText(Messages.CcsTreeDialog_EnableColorDecorationButtonTooltip);

		enableSort = this.createButton(container, Activator.getImage("sort_asc.gif"), SWT.TOGGLE); //$NON-NLS-1$
		enableSort.setToolTipText(Messages.CcsTreeDialog_SortOrderButtonTooltip);
		refresh = this.createButton(container, Activator.getImage("refresh.gif"), SWT.NULL); //$NON-NLS-1$
		refresh.setToolTipText(Messages.CcsTreeDialog_RefreshButtonTooltip);
	}

	private Button createButton(Composite parent, Image image, int style) {
		Button button = new Button(parent, style);
		button.setImage(image);
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
        		handleRadioButtonSelected((Button)e.getSource());
        	}
        });
		GridData data = new GridData();
		data.horizontalIndent = 0;
		data.verticalIndent = 0;
		button.setLayoutData(data);
		return button;
    }
	
	private void handleRadioButtonSelected(Button button) {
		if (button == expand) {
			ccstreeViewer.expandAll();
		}
		if (button == collapse) {
			ccstreeViewer.collapseAll();
			ccstreeViewer.expandToLevel(2);
		}
		if (button == showDomain) {
			CCSTreeViewerLabelProvider labelProvider = (CCSTreeViewerLabelProvider)ccstreeViewer.getLabelProvider();
			labelProvider.showDomain(button.getSelection());
			ccstreeViewer.refresh();
		}
		if (button == showProfile) {
			CCSTreeViewerLabelProvider labelProvider = (CCSTreeViewerLabelProvider)ccstreeViewer.getLabelProvider();
			labelProvider.showProfile(button.getSelection());
			ccstreeViewer.refresh();
		}
		if (button == coloring) {
			CCSTreeViewerLabelProvider labelProvider = (CCSTreeViewerLabelProvider)ccstreeViewer.getLabelProvider();
			labelProvider.setColoringEnabled(button.getSelection());
			ccstreeViewer.refresh();
		}
		if (button == enableSort) {
			CCSTreeViewerSorter sorter = (CCSTreeViewerSorter)ccstreeViewer.getSorter();
			sorter.setEnabled(button.getSelection());
			ccstreeViewer.refresh();
		}
		if (button == refresh) {
			restoreDefault();
		}
	}

	private void setupMenuListener(Composite parent, TreeViewer viewer) {
    	MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
    	menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new CCSTreeViewerMenuListener(getProject(), viewer));
		Menu menu = menuManager.createContextMenu(parent);
		viewer.getTree().setMenu(menu);
    }

	private class CCSTreeViewerMenuListener extends PolicyOutlineMenuListener {
		public CCSTreeViewerMenuListener(IProject project, StructuredViewer viewer) {
			super(project, viewer);
		}
		public void menuAboutToShow(IMenuManager manager) {
			IStructuredSelection selection = (IStructuredSelection)viewer.getSelection();
			if (selection.getFirstElement() instanceof Process) {
				Process process = (Process)selection.getFirstElement();
				String domainText = process.getDomain();
				Domain domain = PolicyCacheManager.getInstance().findDomain(project, domainText);
				manager.add(createKeepDomainAction(new Domain[] {domain}));
//				manager.add(new Separator());
//				manager.add(createInializerAction(new Domain[] {domain}));
			}
		}
	}

	private DomainPolicyPage parent;
	
	public void setParent(DomainPolicyPage parent) {
		this.parent = parent;
	}

}
