/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.core.local.parser;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.AbstractPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.model.domain.Profile;
import jp.sourceforge.tomoyo.core.local.model.domain.Rwx_1;
import jp.sourceforge.tomoyo.core.local.model.domain.Rwx_3;
import jp.sourceforge.tomoyo.core.local.model.domain.Rwx_5;
import jp.sourceforge.tomoyo.core.local.model.domain.Rwx_7;
import jp.sourceforge.tomoyo.core.local.model.except.AbstractKeepDomainElement;
import jp.sourceforge.tomoyo.core.local.model.except.ConditionalElement;
import jp.sourceforge.tomoyo.core.local.model.except.InitializeDomain;
import jp.sourceforge.tomoyo.core.local.model.except.Initializer;
import jp.sourceforge.tomoyo.core.local.model.except.KeepDomain;
import jp.sourceforge.tomoyo.core.local.model.except.NoInitializeDomain;
import jp.sourceforge.tomoyo.core.local.model.except.NoInitializer;
import jp.sourceforge.tomoyo.core.local.model.except.NoKeepDomain;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;

import org.eclipse.core.resources.IProject;

public class DomainPolicyParser extends PolicyParser {

	public DomainPolicyParser(LocalResource localResource) {
		super(localResource);
	}

	public static Pattern DomainPattern = Pattern.compile("^<kernel>.*"); //$NON-NLS-1$
	public static Pattern ProfilePattern = Pattern.compile("^use_profile[\\s]+([0-9]+)"); //$NON-NLS-1$

	private Domain cacheParentDomain;
	
	protected PolicyElement parseLine(AbstractPolicyModel model, String line, int lineno, int column) {
		// search for domains.
		Matcher domainMatcher = DomainPattern.matcher(line);
		if (domainMatcher.matches()) {
			
			if ((cacheParentDomain = (Domain)PolicyCacheManager.getInstance().findElement(model.getProject(), Domain.class, line)) == null) {
				cacheParentDomain = new Domain(line, lineno, column);
				cacheParentDomain.setCreated(model.isValid());
				model.addChild(cacheParentDomain);
				
				PolicyCacheManager.getInstance().addElement(model.getProject(), cacheParentDomain);
			} else {
				cacheParentDomain.setLineno(lineno);
				cacheParentDomain.setOffset(column);
				cacheParentDomain.setCreated(false);
			}
			
			cacheParentDomain.setChecked(true);
			
			return cacheParentDomain;
		}
		// search for access controls.
		Matcher accessMatcher = getMatchingPattern().matcher(line);
		if (accessMatcher.matches()) {
			if (cacheParentDomain != null) {
				String directive = accessMatcher.group(1);
				AccessPermission permission;
				if ((permission = (AccessPermission)PolicyCacheManager.getInstance().findAccessPermission(model.getProject(), cacheParentDomain, line)) == null) {
					permission = (AccessPermission)getLocalResource().getModel().getDefinitionByDirective(directive).createElement(directive, line, lineno, column);
					permission.setCreated(model.isValid());
					if (permission != null) {
						cacheParentDomain.addChild(permission);
						PolicyCacheManager.getInstance().addElement(model.getProject(), permission);
					}
					if (permission == null) {
						TomoyoCorePlugin.logErrorMessage("Unexpacted line:" + line);
					}
				} else {
					permission.setLineno(lineno);
					permission.setOffset(column);
					permission.setCreated(permission.isDeleted());
				}
				
				permission.setChecked(true);
				
				return permission;
			}
		}
		// search for profile definition.
		Matcher profileMatcher = ProfilePattern.matcher(line);
		if (profileMatcher.matches()) {
			if (cacheParentDomain != null) {
				Profile profile = new Profile(line, lineno, column);
				cacheParentDomain.addChild(profile);
				
				profile.setChecked(true);
			}
		}
		
		return null;
	}

	protected void handleParseAboutToStart() {
	}

	protected void handleParseCompleted() {
		IProject project = getLocalResource().getProject();
		
		PolicyCacheManager.getInstance().createDomainTranstion(project);
		
		decorateDomain(project);
	}

	//-----------------------------------------------------------------------------------------
	// Domain decoration
	//-----------------------------------------------------------------------------------------

	private void decorateDomain(IProject project) {
		PolicyElement[] domains = PolicyCacheManager.getInstance().findElements(project, Domain.class);
		if (domains == null)
			return;
		decorateInitializer(project, domains);
		decorateKeepDomain(project, domains);
		decorateReachablePossibility(project, domains);
	}

	private void decorateInitializer(IProject project, PolicyElement[] domains) {
		PolicyElement[] no_initializers = PolicyCacheManager.getInstance().findElements(project, new Class[] { NoInitializer.class, NoInitializeDomain.class });
		no_initializers = filterUnchecked(no_initializers);
		
		PolicyElement[] initializers = PolicyCacheManager.getInstance().findElements(project, new Class[] { Initializer.class, InitializeDomain.class });
		initializers = filterDuplication(initializers);
		initializers = filterUnchecked(initializers);
		initializers = filterOverwrides(initializers, no_initializers);
		
		decorateInitializerTarget(project, domains, initializers);
		decorateInitializerSource(project, initializers);
	}
	
	private void decorateKeepDomain(IProject project, PolicyElement[] domains) {
		
		PolicyElement[] noKeepDomains = PolicyCacheManager.getInstance().findElements(project, NoKeepDomain.class);
		noKeepDomains = filterUnchecked(noKeepDomains);
		
		PolicyElement[] keepDomains = PolicyCacheManager.getInstance().findElements(project, KeepDomain.class);
		keepDomains = filterDuplication(keepDomains);
		keepDomains = filterUnchecked(keepDomains);
		keepDomains = filterOverwrides(keepDomains, noKeepDomains);
		
		for (int dcnt = 0; dcnt < domains.length; dcnt++) {
			Domain domain = (Domain)domains[dcnt];
			domain.clearKeepDomain();
			
			if (domain.isDeleted())
				continue;
			
			String domainText = domain.getText();
			for (int kcnt = 0; kcnt < keepDomains.length; kcnt++) {
				ConditionalElement initializer = (ConditionalElement)keepDomains[kcnt];
				String strKeepDomain = initializer.getTarget();
				if (domainText.equals(strKeepDomain)) {
					if (initializer.hasCondition()) {
						for (int cnt = 0; cnt < domain.getChildrenCount(); cnt++) {
							if (domain.getChild(cnt) instanceof AccessPermission) {
								AccessPermission permission = (AccessPermission)domain.getChild(cnt);
								if (initializer.getCondition().equals(permission.getContents())) {
									domain.addKeepDomain((AbstractKeepDomainElement)initializer);
									break;
								}
							}
						}
					} else {
						domain.addKeepDomain((AbstractKeepDomainElement)initializer);
						break;
					}
				}
			}
		}
	}
	
	private void decorateInitializerTarget(IProject project, PolicyElement[] domains, PolicyElement[] initializers) {
		for (int cnt = 0; cnt < domains.length; cnt++) {
			Domain domain = (Domain)domains[cnt];
			if (domain.isDeleted())
				continue;
			if (domain.getDepth() > 2)
				continue;
			domain.setInitializerTarget(null);
			if (initializers == null)
				return;
			String processName = domain.getProcessName();
			for (int pcnt = 0; pcnt < initializers.length; pcnt++) {
				ConditionalElement initializer = (ConditionalElement)initializers[pcnt];
				if (processName.equals(initializer.getTarget())) {
					domain.setInitializerTarget(initializer);
					PolicyCacheManager.getInstance().addInitializerTarget(project, domain);
					break;
				}
			}
		}
	}

	private void decorateInitializerSource(IProject project, PolicyElement[] initializers) {
		PolicyElement[] domains = PolicyCacheManager.getInstance().findElements(project, Domain.class);
		PolicyElement[] executables = PolicyCacheManager.getInstance().findElements(project, new Class[] { Rwx_1.class, Rwx_3.class, Rwx_5.class, Rwx_7.class } );
		for (int cnt = 0; cnt < initializers.length; cnt++) {
			ConditionalElement initializer = (ConditionalElement)initializers[cnt];
			for (int ecnt = 0; ecnt < executables.length; ecnt++) {
				AccessPermission executable = (AccessPermission)executables[ecnt];
				Domain executableDomain = (Domain)executable.getParent();
				if (isInitizlizerSource(executable, initializer)) {
					executable.setInitializerSource(true);
					executableDomain.addInitializerExecutable(executable);
				}
			}
			for (int dcnt = 0; dcnt < domains.length; dcnt++) {
				Domain domain = (Domain)domains[dcnt];
				if (domain.getDepth() < 3)
					continue;
				if (isInitizlizerSource(domain, initializer)) {
					domain.setInitializerSource(initializer);
				}
			}
		}
	}

	private boolean isInitizlizerSource(Domain domain, ConditionalElement initializer) {
		String executableText = domain.getProcessName();
		if (executableText.equals(initializer.getTarget())) {
			if (initializer.hasCondition()) {
				String condition = initializer.getCondition();
				if (initializer.isConditionDomain()) {
					if (domain.getText().equals(condition)) {
						return true;
					}
				} else {
					if (domain.getProcessName().equals(condition)) {
						return true;
					}
				}
			} else {
				return true;
			}
		}
		return false;
	}

	private boolean isInitizlizerSource(AccessPermission executable, ConditionalElement initializer) {
		String executableText = executable.getContents();
		Domain executableDomain = (Domain)executable.getParent();
		if (executableText.equals(initializer.getTarget())) {
			if (initializer.hasCondition()) {
				String condition = initializer.getCondition();
				if (initializer.isConditionDomain()) {
					if (executableDomain.getText().equals(condition)) {
						return true;
					}
				} else {
					if (executableDomain.getProcessName().equals(condition)) {
						return true;
					}
				}
			} else {
				return true;
			}
		}
		return false;
	}

	private PolicyElement[] filterDuplication(PolicyElement[] initializers) {
		Hashtable<String, ConditionalElement> table = new Hashtable<String, ConditionalElement>();
		for (int cnt = 0; cnt < initializers.length; cnt++) {
			ConditionalElement initializer = (ConditionalElement)initializers[cnt];
			table.put(initializer.getContents(), initializer);
		}
		Collection<ConditionalElement> c = table.values();
		return (PolicyElement[])c.toArray(new PolicyElement[c.size()]);
	}
	
	private PolicyElement[] filterUnchecked(PolicyElement[] initializers) {
		Hashtable<String, ConditionalElement> table = new Hashtable<String, ConditionalElement>();
		for (int cnt = 0; cnt < initializers.length; cnt++) {
			ConditionalElement initializer = (ConditionalElement)initializers[cnt];
//			if (initializer.isChecked())
			if (!initializer.isDeleted())
				table.put(initializer.getContents(), initializer);
		}
		Collection<ConditionalElement> c = table.values();
		return (PolicyElement[])c.toArray(new PolicyElement[c.size()]);
	}
	
	private PolicyElement[] filterOverwrides(PolicyElement[] initializers, PolicyElement[] no_initializers) {
		Hashtable<String, ConditionalElement> table = new Hashtable<String, ConditionalElement>();
		for (int cnt = 0; cnt < initializers.length; cnt++) {
			ConditionalElement initializer = (ConditionalElement)initializers[cnt];
			if (isOverwrided(initializer, no_initializers))
				continue;
			table.put(initializer.getContents(), initializer);
		}
		Collection<ConditionalElement> c = table.values();
		return (PolicyElement[])c.toArray(new PolicyElement[c.size()]);
	}

	private boolean isOverwrided(ConditionalElement initializer, PolicyElement[] no_initializers) {
		for (int ecnt = 0; ecnt < no_initializers.length; ecnt++) {
			ConditionalElement no_initializer = (ConditionalElement)no_initializers[ecnt];
			if (no_initializer.hasCondition()) {
				if (no_initializer.isConditionDomain()) {
					if (initializer.getCondition().equals(no_initializer.getCondition()))
						return true;
				} else {
					if (initializer.getContents().equals(no_initializer.getContents()))
						return true;
				}
			} else {
				if (initializer.getTarget().equals(no_initializer.getContents()))
					return true;
			}
		}
		return false;
	}
	
	private void decorateReachablePossibility(IProject project, PolicyElement[] domains) {
		for (int cnt = 0; cnt < domains.length; cnt++) {
			Domain domain = (Domain)domains[cnt];
			Domain parentDomain = PolicyCacheManager.getInstance().getParentDomain(project, domain);
			decorateReachablePossibility(parentDomain, domain);
		}
	}
	
	private void decorateReachablePossibility(Domain parentDomain, Domain domain) {
		if (domain.getDepth() == 1) {
			domain.setUnreachableReason(Domain.REACHABLE);
			return;
		}
		if (parentDomain == null) {
			domain.setUnreachableReason(Domain.UNREACHABLE_REASON_NO_PARENTS);
			return;
		} else {
			if (domain.getDepth() > 2) {
				if (domain.isInitializerSource()) {
					domain.setUnreachableReason(Domain.UNREACHABLE_REASON_INITIALIZED);
					return;
				}
				if (parentDomain.isInitializerSource()) {
					domain.setUnreachableReason(Domain.UNREACHABLE_REASON_PARENT_INITIALIZED);
					return;
				}
				/*
				if (parentDomain.hasInitializerSource()) {
					AccessPermission[] permissions = parentDomain.getInitializerSource();
					for (int cnt = 0; cnt < permissions.length; cnt++) {
						if (permissions[cnt].getContents().equals(domain.getProcessName())) {
							return;
						}
					}
				}
				*/
			}
			if (parentDomain.isInitializerTarget()) {
				if (parentDomain.getDepth() > 2) {
					domain.setUnreachableReason(Domain.UNREACHABLE_REASON_INITIALIZED);
					return;
				}
			}
			if (isKeptDomain(parentDomain, domain)) {
				domain.setUnreachableReason(Domain.UNREACHABLE_REASON_PARENT_KEPTDOMAIN);
				return;
			}
			if (!parentDomain.isReachable()) {
				domain.setUnreachableReason(Domain.UNREACHABLE_REASON_PARENT_UNREACHABLE);
				return;
			}
		}
		domain.setUnreachableReason(Domain.REACHABLE);
	}

	@SuppressWarnings("unchecked")
	private boolean isKeptDomain(Domain parentDomain, Domain domain) {
		ArrayList keepDomainList = parentDomain.getKeepDomainList();
		if (parentDomain.isKeepDomain()) {
			for (int cnt = 0; cnt < keepDomainList.size(); cnt++) {
				AbstractKeepDomainElement keepDomain = (AbstractKeepDomainElement)keepDomainList.get(cnt);
				if (keepDomain.getTarget().equals(parentDomain.getText())) {
					if (keepDomain.hasCondition()) {
						String conditionText = keepDomain.getCondition();
						if (keepDomain.isConditionDomain()) {
							if (domain.getText().equals(conditionText))
								return true;
						} else {
							if (domain.getProcessName().equals(conditionText))
								return true;
						}
					} else {
						return true;
					}
				}
			}
		}
		return false;
	}

}
