/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.base.ssh.connection;

import java.io.UnsupportedEncodingException;
import java.util.Hashtable;
import java.util.Observable;
import java.util.Observer;

import org.eclipse.core.resources.IProject;
import org.eclipse.swt.widgets.Composite;

import jp.sourceforge.tomoyo.base.ssh.Activator;
import jp.sourceforge.tomoyo.base.ssh.ui.UIUtilities;
import jp.sourceforge.tomoyo.core.PersistentPropertyManager;
import jp.sourceforge.tomoyo.core.ProjectProperty;
import jp.sourceforge.tomoyo.core.ProjectPropertyManager;
import jp.sourceforge.tomoyo.core.extensions.IConnectionAdapter;
import jp.sourceforge.tomoyo.core.extensions.IConnectionSettings;
import jp.sourceforge.tomoyo.core.server.AbstractCommand;
import jp.sourceforge.tomoyo.core.server.ConcreteCommand;
import jp.sourceforge.tomoyo.core.server.Connection;
import jp.sourceforge.tomoyo.core.server.ConnectionManager;
import jp.sourceforge.tomoyo.core.server.ICommand;
import jp.sourceforge.tomoyo.core.server.ReadCommand;
import jp.sourceforge.tomoyo.core.server.WriteCommand;

public class SSHConnectionAdapter implements IConnectionAdapter, Observer {

	public static final String ADAPTER_ID = SSHConnectionAdapter.class.getName();
	
	public String getID() {
		return ADAPTER_ID;
	}

	//----------------------------------------------------------------
	// Public methods (inbound)
	//----------------------------------------------------------------

	public static IConnectionSettings getConnectionSettings(IProject project) {
		if (project == null || !project.isOpen())
			return null;

		SSHConnectionSettings settings = new SSHConnectionSettings();
		settings.project = project;
		try {
			ProjectProperty properties = ProjectPropertyManager.getInstance().getProperty(project);
			settings.hostname = properties.getProperty(PersistentPropertyManager.PROPERTY_SERVER_HOSTNAME);
			settings.username = properties.getProperty(PersistentPropertyManager.PROPERTY_LOGIN_USERNAME);
			settings.portno = Integer.parseInt(properties.getProperty(PersistentPropertyManager.PROPERTY_SERVER_PORT));
			settings.isPublicKeyAuthentication = properties.usePublicKeyAuthentication();
			settings.pkeyfile = properties.getPrivateKeyFile();
			settings.password = properties.getProperty(PersistentPropertyManager.PROPERTY_LOGIN_PASSWORD);
			settings.knownHosts = properties.getKnownHostsFile();
			
			String confirmConnect = properties.getProperty(PersistentPropertyManager.PROPERTY_MANAGER_CONFIRM_CONNECT);
			settings.confirmConnection = (confirmConnect == null) ? true : confirmConnect.equals("true");
		} catch (Exception e) {
			Activator.log(e);
			return null;
		}
		
		return settings;
	}

	public boolean connect(IProject project) {
		Connection connection = getConnection(getConnectionSettings(project));
		if (connection.isConnected())
			return true;
		return connection.connect();
	}

	@Override
	public boolean exec(ConcreteCommand cCommand) {
		return _exec(cCommand);
	}

	@Override
	public boolean read(ReadCommand rCommand) {
		return _exec(rCommand);
	}

	@Override
	public boolean write(WriteCommand wCommand) {
		return _exec(wCommand);
	}

	private boolean _exec(ICommand command) {
		if (!_checkConnection(command))
			return false;
		
		Connection connection = getConnection(getConnectionSettings(command.getProject()));
		CommandThread thread = new CommandThread((SSHSession)connection, command);
		Activator.getStandardDisplay().syncExec(thread);

		return _checkStatus(command, thread);
	}

	private boolean _checkConnection(ICommand cCommand) {
		if (isConnected(cCommand.getProject())) {
			return true;
		} else {
			return connect(cCommand.getProject());
		}
	}

	private boolean _checkStatus(ICommand command, CommandThread thread) {
		if (thread.isSuccessful()) {
			if (command.isSuccessful()) {
				;
			} else {
				AbstractCommand aCommand = (AbstractCommand)command;
				String additionalMessage = null;
				if (command instanceof ConcreteCommand) {
					ConcreteCommand cCommand = (ConcreteCommand)command;
					additionalMessage = " command = " + CommandThread.toCommandString(cCommand);
				} else if (command instanceof ReadCommand) {
					ReadCommand rCommand = (ReadCommand)command;
					additionalMessage = " source = " + rCommand.getSource();
				} else if (command instanceof WriteCommand) {
					WriteCommand wCommand = (WriteCommand)command;
					additionalMessage = " target = " + wCommand.getTarget();
				}
				Activator.log(new Exception(
						"Exec command failure : status = " + command.getSystemStatusCode()
						+ additionalMessage
						+ " output = " + convert(aCommand.getSTDOut())
						+ " error message = " + convert(aCommand.getSTDErr())));
			}
			return true;
		} else {
			if (command instanceof ConcreteCommand) {
				ConcreteCommand cCommand = (ConcreteCommand)command;
				Activator.log(new Exception("Exec command failure : status = " + cCommand.getSystemStatusCode() + " command = " + CommandThread.toCommandString(cCommand)));
			} else if (command instanceof ReadCommand) {
				ReadCommand rCommand = (ReadCommand)command;
				Activator.log(new Exception("Read command failure : status = " + rCommand.getSystemStatusCode() + " source = " + rCommand.getSource()));
			} else if (command instanceof WriteCommand) {
				WriteCommand wCommand = (WriteCommand)command;
				Activator.log(new Exception("Write command failure : status = " + wCommand.getSystemStatusCode() + " target = " + wCommand.getTarget()));
			}
			return false;
		}
	}

	private String convert(String source) {
		try {
			return new String(source.getBytes(), "EUC-JP");
		} catch (UnsupportedEncodingException e) {
			Activator.log(e);
			return source;
		}
	}
	
	
/*
	public boolean asyncExecCommand(SimpleCommand command) {
		try {
			if (!isConnected(command.getProject())) {
				boolean isConnected = connect(command.getProject());
				if (!isConnected)
					return false;
			}
			IConnectionSettings settings = getConnectionSettings(command.getProject());
			Connection connection = getConnection(settings);
			AsyncCommandThread thread = new AsyncCommandThread((SSHSession)connection, command);
			new Thread(thread).start();
			if (command.isSuccess()) {
				return true;
			} else {
				Activator.logErrorMessage("Command exec failure : status = " + command.getStatus() + " cmd = " + command.getCommand());
				return false;
			}
		} catch (Exception e) {
			Activator.logException(e);
			return false;
		} finally {
		}
	}
*/
	/*
	private boolean wasConnected(IProject project) {
		ArrayList connectionHistoryList = (ArrayList)connectionHistoryCache.get(project);
		if (connectionHistoryList.size() == 0)
			return false;
		ConnectionStatus connectionStatus = (ConnectionStatus)connectionHistoryList.get(connectionHistoryList.size() - 1);
		return connectionStatus.isConnected();
	}
	*/

	public void cleanup(IProject project) {
		Connection connection = getConnection(getConnectionSettings(project));
		if (connection.isConnected())
			connection.disconnect();
	}

	public boolean isConnected(IProject project) {
		Connection connection = getConnection(getConnectionSettings(project));
		return connection.isConnected();
	}

	public String getErrorMessage(IProject project) {
		Connection connection = getConnection(getConnectionSettings(project));
		return connection.getErrorMessage();
	}

	public void disconnect(IProject project) {
		Connection connection = getConnection(getConnectionSettings(project));
		connection.disconnect();
	}

	public void createSettingComposite(IProject project, Composite parent) {
		new UIUtilities(project).createSettingComposite(parent);
	}
	
	public String getSelfDomain(IProject project) {
		ConcreteCommand cCommand = new ConcreteCommand(project, "cat < /proc/ccs/info/self_domain");
		Connection connection = getConnection(getConnectionSettings(cCommand.getProject()));
		CommandThread thread = new CommandThread((SSHSession)connection, cCommand);
		Activator.getStandardDisplay().syncExec(thread);
		if (cCommand.isSuccessful())
			return cCommand.getSTDOut();
		else
			return null;
	}

	//----------------------------------------------------------------
	// Internal methods
	//----------------------------------------------------------------

	private static Hashtable<IProject, SSHSession> connectionCache = new Hashtable<IProject, SSHSession>();

	private Connection getConnection(IConnectionSettings settings) {
		if (connectionCache.get(settings.getProject()) == null) {
			SSHSession connection = new SSHSession((SSHConnectionSettings)settings);
			connectionCache.put(settings.getProject(), connection);
			connection.addObserver(this);
		} else {
			SSHSession connection = connectionCache.get(settings.getProject());
			connection.settings = (SSHConnectionSettings)settings;
		}
		return connectionCache.get(settings.getProject());
	}

	//----------------------------------------------------------------
	// Public method(s) (outbound)
	//----------------------------------------------------------------
	
	public void update(Observable o, Object arg) {
		IConnectionSettings settings = (IConnectionSettings)arg;
		ConnectionManager.getInstance().updateConnectionState(settings.getProject(), isConnected(settings.getProject()));
	}

}
