/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "file.h"
#include <shlwapi.h>
#include "abort.h"
#include "general.h"
#include "history.h"
#include "resource.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:ファイル入力関数群                                                       *
*                                                                             *
******************************************************************************/
typedef BOOL (*IsCharSet_t)(LPSTR, DWORD);
typedef int (*EnterCharSet_t)(LPSTR, DWORD);
typedef VOID (*TranslateCharSet_t)(LPLINEBUF, LPSTR, DWORD);


static ABORTDLG AbortDlg;


/*  ja:UNICODE(リトルエンディアン)を判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:UNICODE,FALSE:その他                                       */
static BOOL
IsUnicodeLittle (LPSTR lpcFile,
                 DWORD dwSize)
{
  return dwSize >= 2 && dwSize % 2 == 0
                            && lpcFile[0] == '\xff' && lpcFile[1] == '\xfe';
}


/*  ja:UNICODE(ビックエンディアン)を判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:UNICODE,FALSE:その他                                       */
static BOOL
IsUnicodeBig (LPSTR lpcFile,
              DWORD dwSize)
{
  return dwSize >= 2 && dwSize % 2 == 0
                            && lpcFile[0] == '\xfe' && lpcFile[1] == '\xff';
}


/*  ja:UTF-7を判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:UTF-7,FALSE:その他                                         */
static BOOL
IsUtf7 (LPSTR lpcFile,
        DWORD dwSize)
{
  int i = 0, nBits;
  BOOL fBase64 = FALSE;
  DWORD dwBits = 0;

  while (i < dwSize)
    {
      if (i % WCOMMON_BUFFER_SIZE == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
        }
      if (fBase64)
        {
          if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
            {
              dwBits <<= 6;
              nBits += 6;
              if ('A' <= lpcFile[i] && lpcFile[i] <= 'Z')
                dwBits |= lpcFile[i] - 'A';
              else if ('a' <= lpcFile[i] && lpcFile[i] <= 'z')
                dwBits |= lpcFile[i] - 'a' + 26;
              else if ('0' <= lpcFile[i] && lpcFile[i] <= '9')
                dwBits |= lpcFile[i] - '0' + 52;
              else if (lpcFile[i] == '+')
                dwBits |= 62;
              else
                dwBits |= 63;
              i++;
              if (nBits >= 16)
                nBits -= 16;
            }
          else
            {
              if (lpcFile[i] == '-')
                i++;
              if (dwBits & (1 << nBits) - 1)
                return FALSE;
              fBase64 = FALSE;
            }
        }
      else
        {
          if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
            {
              i += 2;
            }
          else if (lpcFile[i] == '+')
            {
              i++;
              nBits = 0;
              fBase64 = TRUE;
            }
          else if (lpcFile[i] == '\t'
                            || lpcFile[i] == cCRLF[0] || lpcFile[i] == cCRLF[1]
                            || ' ' <= lpcFile[i] && lpcFile[i] <= '*'
                            || ',' <= lpcFile[i] && lpcFile[i] <= '['
                            || ']' <= lpcFile[i] && lpcFile[i] <= '}')
            {
              i++;
            }
          else
            {
              return FALSE;
            }
        }
    }
  return TRUE;
}


/*  ja:UTF-8を判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:UTF-8,FALSE:その他                                         */
static BOOL
IsUtf8 (LPSTR lpcFile,
        DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % WCOMMON_BUFFER_SIZE == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                            && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if ((BYTE)lpcFile[i] <= 0x7f)
        i++;
      else if (i + 1 < dwSize
                && (0xc2 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf
                && 0x80 <= (BYTE)lpcFile[i + 1] && (BYTE)lpcFile[i + 1] <= 0xbf
                || (BYTE)lpcFile[i] == 0xc0 && (BYTE)lpcFile[i + 1] == 0x80))
        i += 2;
      else if (i + 2 < dwSize
                && 0xe0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xef
                && 0x80 <= (BYTE)lpcFile[i + 1] && (BYTE)lpcFile[i + 1] <= 0xbf
                && 0x80 <= (BYTE)lpcFile[i + 2] && (BYTE)lpcFile[i + 2] <= 0xbf
                                && lpcFile[i] & 0x0f | lpcFile[i + 1] & 0x20)
        i += 3;
      else
        return FALSE;
    }
  return TRUE;
}


/*  ja:ISO-2022-JPを判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:ISO-2022-JP,FALSE:その他                                   */
static BOOL
IsIso2022Jp (LPSTR lpcFile,
             DWORD dwSize)
{
  int i = 0;
  BOOL fKanji = FALSE;

  while (i < dwSize)
    {
      if (i % WCOMMON_BUFFER_SIZE == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J'))
        {
          /* ja:漢字OUT */
          if (!fKanji)
            return FALSE;
          fKanji = FALSE;
          i += 3;
        }
      else if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B'))
        {
          /* ja:漢字IN */
          if (fKanji)
            return FALSE;
          fKanji = TRUE;
          i += 3;
        }
      else
        {
          if (fKanji)
            {
              int j;

              if (i + 1 >= dwSize)
                return FALSE;
              for (j = 0; j < 2; j++)
                {
                  if ((BYTE)lpcFile[i] < 0x21 || 0x7e < (BYTE)lpcFile[i])
                    return FALSE;
                  i++;
                }
            }
          else
            {
              if (0x80 <= (BYTE)lpcFile[i])
                return FALSE;
              i++;
            }
        }
    }
  return !fKanji;
}


/*  ja:EUC-JPを判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:EUC-JP,FALSE:その他                                        */
static BOOL
IsEucJp (LPSTR lpcFile,
         DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % WCOMMON_BUFFER_SIZE == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if (0xa1 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfe)
        {
          i++;
          if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xfe < (BYTE)lpcFile[i])
            return FALSE;
          i++;
        }
      else if ((BYTE)lpcFile[i] == 0x8f)
        {
          int j;

          i++;
          for (j = 0; j < 2; j++)
            {
              if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xfe < (BYTE)lpcFile[i])
                return FALSE;
              i++;
            }
        }
      else
        {
          if ((BYTE)lpcFile[i] == 0x8e)
            {
              i++;
              if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xdf < (BYTE)lpcFile[i])
                return FALSE;
            }
          i++;
        }
    }
  return TRUE;
}


/*  ja:SHIFT_JISを判別する
    lpcFile,ファイル
     dwSize,バイト数
        RET,TRUE:SHIFT_JIS,FALSE:その他                                     */
static BOOL
IsShiftJis (LPSTR lpcFile,
            DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % WCOMMON_BUFFER_SIZE == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if (0x81 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0x9f
                    || 0xe0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfc)
        {
          i++;
          if (i >= dwSize || (BYTE)lpcFile[i] < 0x40
                        || (BYTE)lpcFile[i] == 0x7f || 0xfc < (BYTE)lpcFile[i])
            return FALSE;
          i++;
        }
      else
        {
          i++;
        }
    }
  return TRUE;
}


/*  ja:UNICODE(リトルエンディアン)でCRLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeLittleCrlf (LPSTR lpcFile,
                        DWORD dwSize)
{
  int i;

  for (i = 0; i + 3 < dwSize
            && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != '\0'
            || lpcFile[i + 2] != cCRLF[1] || lpcFile[i + 3] != '\0'); i += 2);
  if (i + 3 >= dwSize)
    i += 2;
  return i;
}


/*  ja:UNICODE(リトルエンディアン)でCR
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeLittleCr (LPSTR lpcFile,
                      DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != '\0'); i += 2);
  return i;
}


/*  ja:UNICODE(リトルエンディアン)でLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeLittleLf (LPSTR lpcFile,
                      DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != cCRLF[1] || lpcFile[i + 1] != '\0'); i += 2);
  return i;
}


/*  ja:UNICODE(ビックエンディアン)でCRLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeBigCrlf (LPSTR lpcFile,
                     DWORD dwSize)
{
  int i;

  for (i = 0; i + 3 < dwSize
            && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[0]
            || lpcFile[i + 2] !='\0' || lpcFile[i + 3] != cCRLF[1]); i += 2);
  if (i + 3 >= dwSize)
    i += 2;
  return i;
}


/*  ja:UNICODE(ビックエンディアン)でCR
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeBigCr (LPSTR lpcFile,
                   DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[0]); i += 2);
  return i;
}


/*  ja:UNICODE(ビックエンディアン)でLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterUnicodeBigLf (LPSTR lpcFile,
                   DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[1]); i += 2);
  return i;
}


/*  ja:マルチバイトでCRLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterMultiByteCrlf (LPSTR lpcFile,
                    DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
            && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != cCRLF[1]); i++);
  if (i + 1 >= dwSize)
    i++;
  return i;
}


/*  ja:マルチバイトでCR
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterMultiByteCr (LPSTR lpcFile,
                  DWORD dwSize)
{
  int i;

  for (i = 0; i < dwSize && lpcFile[i] != cCRLF[0]; i++);
  return i;
}


/*  ja:マルチバイトでLF
    lpcFile,ファイル
     dwSize,バイト数
        RET,文字数                                                          */
static int
EnterMultiByteLf (LPSTR lpcFile,
                  DWORD dwSize)
{
  int i;

  for (i = 0; i < dwSize && lpcFile[i] != cCRLF[1]; i++);
  return i;
}


/*  ja:UNICODE(リトルエンディアン)を変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateUnicodeLittle (LPLINEBUF p,
                        LPSTR     lpcFile,
                        DWORD     dwSize)
{
  p->nLength = WideCharToMultiByte (CP_ACP, 0,
                (LPWSTR)lpcFile, dwSize / sizeof (WCHAR), NULL, 0, NULL, NULL);
  p->lpszText = MemoryAlloc (p->nLength * sizeof (CHAR));
  p->nLength = WideCharToMultiByte (CP_ACP, 0,
                                    (LPWSTR)lpcFile, dwSize / sizeof (WCHAR),
                                    p->lpszText, p->nLength, NULL, NULL);
}


/*  ja:UNICODE(ビックエンディアン)を変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateUnicodeBig (LPLINEBUF p,
                     LPSTR     lpcFile,
                     DWORD     dwSize)
{
  int i;
  LPSTR lpcText;

  lpcText = MemoryAlloc (dwSize * sizeof (CHAR));
  for (i = 0; i < dwSize; i += 2)
    {
      lpcText[i] = lpcFile[i + 1];
      lpcText[i + 1] = lpcFile[i];
    }
  TranslateUnicodeLittle (p, lpcText, dwSize);
  MemoryFree (lpcText);
}


/*  ja:UTF-7を変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateUtf7 (LPLINEBUF p,
               LPSTR     lpcFile,
               DWORD     dwSize)
{
  int i = 0, j = 0, nBits;
  BOOL fBase64 = FALSE;
  DWORD dwBits = 0;
  LPSTR lpcText;

  while (i < dwSize)
    if (fBase64)
      {
        if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
          {
            nBits += 6;
            i++;
            if (nBits >= 16)
              {
                j++;
                nBits -= 16;
              }
          }
        else
          {
            if (lpcFile[i] == '-')
              i++;
            fBase64 = FALSE;
          }
      }
    else
      {
        if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
          {
            i += 2;
            j++;
          }
        else if (lpcFile[i] == '+')
          {
            i++;
            nBits = 0;
            fBase64 = TRUE;
          }
        else
          {
            i++;
            j++;
          }
      }
  lpcText = MemoryAlloc (j * sizeof (WCHAR));
  i = j = 0;
  fBase64 = FALSE;
  while (i < dwSize)
    if (fBase64)
      {
        if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
          {
            dwBits <<= 6;
            nBits += 6;
            if ('A' <= lpcFile[i] && lpcFile[i] <= 'Z')
              dwBits |= lpcFile[i] - 'A';
            else if ('a' <= lpcFile[i] && lpcFile[i] <= 'z')
              dwBits |= lpcFile[i] - 'a' + 26;
            else if ('0' <= lpcFile[i] && lpcFile[i] <= '9')
              dwBits |= lpcFile[i] - '0' + 52;
            else if (lpcFile[i] == '+')
              dwBits |= 62;
            else
              dwBits |= 63;
            i++;
            if (nBits >= 16)
              {
                lpcText[j++] = dwBits >> nBits - 16 & 0xff;
                lpcText[j++] = dwBits >> nBits - 8 & 0xff;
                nBits -= 16;
              }
          }
        else
          {
            if (lpcFile[i] == '-')
              i++;
            fBase64 = FALSE;
          }
      }
    else
      {
        if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
          {
            lpcText[j++] = '+';
            lpcText[j++] = '\0';
            i += 2;
          }
        else if (lpcFile[i] == '+')
          {
            i++;
            nBits = 0;
            fBase64 = TRUE;
          }
        else
          {
            lpcText[j++] = lpcFile[i++];
            lpcText[j++] = '\0';
          }
      }
  TranslateUnicodeLittle (p, lpcText, j);
  MemoryFree (lpcText);
}


/*  ja:UTF-8を変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateUtf8 (LPLINEBUF p,
               LPSTR     lpcFile,
               DWORD     dwSize)
{
  int i = 0, j = 0;
  LPSTR lpcText;

  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xc0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf)
        i += 2;
      else if (i + 2 < dwSize && 0xe0 <= (BYTE)lpcFile[i])
        i += 3;
      else
        i++;
      j++;
    }
  lpcText = MemoryAlloc (j * sizeof (WCHAR));
  i = j = 0;
  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xc0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf)
        {
          lpcText[j] = lpcFile[i] << 6 | lpcFile[i + 1] & 0x3f;
          lpcText[j + 1] = lpcFile[i] >> 2 & 0x07;
          i += 2;
        }
      else if (i + 2 < dwSize && 0xe0 <= (BYTE)lpcFile[i])
        {
          lpcText[j] = lpcFile[i + 1] << 6 | lpcFile[i + 2] & 0x3f;
          lpcText[j + 1] = lpcFile[i] << 4 | (lpcFile[i + 1] & 0x3f) >> 2;
          i += 3;
        }
      else
        {
          lpcText[j] = lpcFile[i++];
          lpcText[j + 1] = '\0';
        }
      j += 2;
    }
  TranslateUnicodeLittle (p, lpcText, j);
  MemoryFree (lpcText);
}


/*  ja:ISO-2022-JPを変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateIso2022Jp (LPLINEBUF p,
                    LPSTR     lpcFile,
                    DWORD     dwSize)
{
  int i = 0, j = 0;
  BOOL fKanji = FALSE;

  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  while (i < dwSize)
    {
      if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J'))
        {
          /* ja:漢字OUT */
          fKanji = FALSE;
          i += 3;
        }
      else if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B'))
        {
          /* ja:漢字IN */
          fKanji = TRUE;
          i += 3;
        }
      else if (i + 1 < dwSize && fKanji)
        {
          p->lpszText[j] = lpcFile[i];
          p->lpszText[j + 1] = lpcFile[i + 1];
          p->lpszText[j + 1] += p->lpszText[j] & 1 ? 0x1f : 0x7d;
          if ((BYTE)p->lpszText[j + 1] >= 0x7f)
            p->lpszText[j + 1]++;
          p->lpszText[j] = ((BYTE)p->lpszText[j] - 0x21 >> 1) + 0x81;
          if ((BYTE)p->lpszText[j] > 0x9f)
            p->lpszText[j] += 0x40;
          i += 2;
          j += 2;
        }
      else
        {
          p->lpszText[j++] = lpcFile[i++];
        }
    }
  p->nLength = j;
}


/*  ja:EUC-JPを変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateEucJp (LPLINEBUF p,
                LPSTR     lpcFile,
                DWORD     dwSize)
{
  int i = 0, j = 0;

  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xa1 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfe)
        {
          p->lpszText[j] = lpcFile[i] & ~0x80;
          p->lpszText[j + 1] = lpcFile[i + 1] & ~0x80;
          p->lpszText[j + 1] += p->lpszText[j] & 1 ? 0x1f : 0x7d;
          if ((BYTE)p->lpszText[j + 1] >= 0x7f)
            p->lpszText[j + 1]++;
          p->lpszText[j] = ((BYTE)p->lpszText[j] - 0x21 >> 1) + 0x81;
          if ((BYTE)p->lpszText[j] > 0x9f)
            p->lpszText[j] += 0x40;
          i += 2;
          j += 2;
        }
      else if ((BYTE)lpcFile[i] == 0x8f)
        {
          i += 3;
          p->lpszText[j++] = '?';
        }
      else
        {
          if (i + 1 < dwSize && (BYTE)lpcFile[i] == 0x8e)
            i++;
          p->lpszText[j++] = lpcFile[i++];
        }
    }
  p->nLength = j;
}


/*  ja:SHIFT_JISを変換する
          p,ラインバッファ
    lpcFile,ファイル
     dwSize,バイト数                                                        */
static VOID
TranslateShiftJis (LPLINEBUF p,
                   LPSTR     lpcFile,
                   DWORD     dwSize)
{
  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  p->nLength = dwSize;
  MemoryCopy (p->lpszText, lpcFile, dwSize * sizeof (CHAR));
}


/*  ja:改行コードを取得する
     lpcFile,ファイル
      dwSize,バイト数
    uCharSet,キャラクターセット
         RET,0:CRLF,1:LF,2:CR                                               */
static UINT
NegotiateRetCode (LPSTR lpcFile,
                  DWORD dwSize,
                  UINT  uCharSet)
{
  int i = 0, nCRLF[3] = {0, 0, 0};

  switch (uCharSet)
    {
      case 0:/* ja:UNICODE(リトルエンディアン) */
        while (i < dwSize)
          if (i <= dwSize - 4
                    && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == '\0'
                    && lpcFile[i + 2] == cCRLF[1] && lpcFile[i + 3] == '\0')
            {
              i += 4;
              nCRLF[0]++;
            }
          else if (i <= dwSize - 2
                        && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == '\0')
            {
              i += 2;
              nCRLF[1]++;
            }
          else if (i <= dwSize - 2
                        && lpcFile[i] == cCRLF[1] && lpcFile[i + 1] == '\0')
            {
              i += 2;
              nCRLF[2]++;
            }
          else
            {
              i += 2;
            }
        break;
      case 1:/* ja:UNICODE(ビックエンディアン) */
        while (i < dwSize)
          if (i <= dwSize - 4
                    && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[0]
                    && lpcFile[i + 2] == '\0' && lpcFile[i + 3] == cCRLF[1])
            {
              i += 4;
              nCRLF[0]++;
            }
          else if (i <= dwSize - 2
                        && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[0])
            {
              i += 2;
              nCRLF[1]++;
            }
          else if (i <= dwSize - 2
                        && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[1])
            {
              i += 2;
              nCRLF[2]++;
            }
          else
            {
              i += 2;
            }
          break;
      default:/* ja:ISO-2022-JP,UTF-7,UTF-8,EUC-JP,SHIFT_JIS */
        while (i < dwSize)
          if (i <= dwSize - 2
                    && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == cCRLF[1])
            {
              i += 2;
              nCRLF[0]++;
            }
          else if (lpcFile[i] == cCRLF[0])
            {
              i++;
              nCRLF[1]++;
            }
          else if (lpcFile[i] == cCRLF[1])
            {
              i++;
              nCRLF[2]++;
            }
          else
            {
              i++;
            }
    }
  return nCRLF[0] >= nCRLF[1] && nCRLF[0] >= nCRLF[2]
                                        ? 0 : (nCRLF[1] >= nCRLF[2] ? 1 : 2);
}


/*  ja:TXTファイルを開く
           ptw,TXTウインドウ情報
     dwCharSet,キャラクターセットの自動判別候補
    fNegotiate,TRUE:自動改行判別,FALSE:デフォルト
      uRetCode,改行コード
       RET,TRUE:正常終了,FALSE:エラー                                       */
BOOL
OpenTextFile (LPTEXTWND ptw,
              DWORD     dwCharSet,
              BOOL      fNegotiate,
              UINT      uRetCode)
{
  BOOL fResult = TRUE;
  DWORD dwSize = INVALID_FILE_SIZE;
  HANDLE hFile = INVALID_HANDLE_VALUE, hMap = NULL;
  HINSTANCE hInstance;
  LPSTR lpcFile = NULL;
  LPTSTR lpszAbort;

  hInstance = GetModuleHandle (NULL);
  ptw->lpStart = NULL;
  hDlgCancel = NULL;
  hFile = CreateFile (ptw->lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile != INVALID_HANDLE_VALUE)
    {
      dwSize = GetFileSize (hFile, NULL);
      if (dwSize != INVALID_FILE_SIZE)
        {
          if (dwCharSet == 0)
            dwCharSet = 0x2135467;
          ptw->uCharSet = (dwCharSet & 15) - 1;
          ptw->uRetCode = uRetCode;
          ptw->nMax = 1;
          ptw->nOff = 0;
          ptw->lpStart = MemoryAlloc (sizeof (LINEBUF));
          if (dwSize > 0)
            {
              hMap = CreateFileMapping (hFile, NULL,
                                                    PAGE_READONLY, 0, 0, NULL);
              if (hMap)
                {
                  lpcFile = MapViewOfFile (hMap, FILE_MAP_READ, 0, 0, 0);
                  if (lpcFile)
                    {
                      EnableWindow (hWndMain, FALSE);
                      AbortDlg.fSignal = FALSE;
                      AbortDlg.hEvent = NULL;
                      AbortDlg.lpszText = lpszAbort
                                        = LoadText (hInstance, IDS_JOB_READ);
                      hDlgCancel = CreateDialogParamGUI (hInstance,
                                MAKEINTRESOURCE (DIALOG_G),
                                hWndClient, AbortDlgProc, (LPARAM)&AbortDlg);
                    }
                }
            }
        }
    }
  if (hDlgCancel)
    {
      int i = 0;
      DWORD dwRetLen;
      LPLINEBUF p;
      EnterCharSet_t EnterText;
      TranslateCharSet_t TranslateText;
      const static IsCharSet_t IsCharSet[7] = {IsUnicodeLittle, IsUnicodeBig,
                            IsUtf7, IsUtf8, IsIso2022Jp, IsEucJp, IsShiftJis};
      const static EnterCharSet_t EnterCharSet[7][3] = {
        {EnterUnicodeLittleCrlf, EnterUnicodeLittleCr, EnterUnicodeLittleLf},
        {EnterUnicodeBigCrlf,    EnterUnicodeBigCr,    EnterUnicodeBigLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf}};
      const static TranslateCharSet_t TranslateCharSet[7] = {
        TranslateUnicodeLittle, TranslateUnicodeBig, TranslateUtf7,
        TranslateUtf8, TranslateIso2022Jp, TranslateEucJp, TranslateShiftJis};

      /* ja:キャラクターセット */
      if ((dwCharSet & ~15) != 0)
        {
          DWORD dwAuto;

          for (dwAuto = dwCharSet; !AbortDlg.fSignal && dwAuto & 15;
                                                                dwAuto >>= 4)
            if (IsCharSet[(dwAuto & 15) - 1](lpcFile, dwSize))
              {
                ptw->uCharSet = (dwAuto & 15) - 1;
                break;
              }
        }

      /* ja:改行コード */
      if (fNegotiate)
        ptw->uRetCode = NegotiateRetCode (lpcFile, dwSize, ptw->uCharSet);

      /* ja:読み込む */
      switch (ptw->uCharSet)
        {
          case 0:/* ja:UNICODE(リトルエンディアン) */
            if (dwSize >= 2 && lpcFile[0] == '\xff' && lpcFile[1] == '\xfe')
              i = 2;
            dwSize &= ~1;
            break;
          case 1:/* ja:UNICODE(ビックエンディアン) */
            if (dwSize >= 2 && lpcFile[0] == '\xfe' && lpcFile[1] == '\xff')
              i = 2;
            dwSize &= ~1;
        }
      p = ptw->lpStart;
      dwRetLen = (ptw->uRetCode == 0 ? 2 : 1) * (ptw->uCharSet <= 1 ? 2 : 1);
      EnterText = EnterCharSet[ptw->uCharSet][ptw->uRetCode];
      TranslateText = TranslateCharSet[ptw->uCharSet];
      while (i < dwSize)
        {
          int nLength;
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
          nLength = EnterText (lpcFile + i, dwSize - i);
          if (nLength > 0)
            TranslateText (p, lpcFile + i, nLength);
          i += nLength;
          if (i + dwRetLen <= dwSize)
            {
              LPLINEBUF q;

              q = MemoryAlloc (sizeof (LINEBUF));
              q->lpszText = NULL;
              q->nLength = 0;
              q->fMargin = FALSE;
              q->prev = p;
              q->next = p->next;
              p->next = q;
              p = q;
              ptw->nMax++;
              i += dwRetLen;
            }
        }
    }
  else if (dwSize != 0)
    {
      LPLINEBUF p, q;

      for (p = ptw->lpStart; p; p = q)
        {
          q = p->next;
          MemoryFree (p->lpszText);
          MemoryFree (p);
        }
      fResult = FALSE;
    }
 if (hFile != INVALID_HANDLE_VALUE)
    {
      if (hMap)
        {
          if (lpcFile)
            UnmapViewOfFile (lpcFile);
          CloseHandle (hMap);
        }
      CloseHandle (hFile);
    }
  /* ja:右マージンで折り返す */
  if (fResult)
    ModifyMargin (ptw);
  /* ja:終了処理 */
  if (hDlgCancel)
    {
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      MemoryFree (lpszAbort);
    }
  if (!fResult)
    {
      LPTSTR lpszText;

      lpszText = LoadText (hInstance, IDS_FILE_OPEN);
      MessageBox (hWndClient, lpszText,
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (lpszText);
    }
  return fResult;
}


/******************************************************************************
*                                                                             *
* ja:ファイル出力関数群                                                       *
*                                                                             *
******************************************************************************/
typedef BOOL (*ConvertCharSet_t)(HANDLE, LPLINEBUF);


/*  ja:UNICODE(リトルエンディアン)に変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertUnicodeLittle (HANDLE    hFile,
                      LPLINEBUF p)
{
  int nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_ACP, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_ACP, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  fResult = WriteFile (hFile, lpwText,
                                    nLength * sizeof (WCHAR), &dwWrite, NULL)
                                        && nLength * sizeof (WCHAR) == dwWrite;
  MemoryFree (lpwText);
  return fResult;
}


/*  ja:UNICODE(ビックエンディアン)に変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertUnicodeBig (HANDLE    hFile,
                   LPLINEBUF p)
{
  int i, nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_ACP, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_ACP, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  for (i = 0; i < nLength; i++)
    lpwText[i] = lpwText[i] >> 8 | lpwText[i] << 8;
  fResult = WriteFile (hFile, lpwText,
                                    nLength * sizeof (WCHAR), &dwWrite, NULL)
                                        && nLength * sizeof (WCHAR) == dwWrite;
  MemoryFree (lpwText);
  return fResult;
}


/*  ja:UTF-7に変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertUtf7 (HANDLE    hFile,
             LPLINEBUF p)
{
  int i, j = 0, n, nLength;
  BOOL fMode, fResult;
  DWORD dwWrite;
  LPSTR lpcText;
  LPWSTR lpwText;
  static int nBits;
  static BOOL fBase64;
  static DWORD dwBits;
  const static CHAR cBase64[64] = {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
                                   'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
                                   'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
                                   'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
                                   'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
                                   'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
                                   'w', 'x', 'y', 'z', '0', '1', '2', '3',
                                   '4', '5', '6', '7', '8', '9', '+', '/'};

  if (!p->prev || !p->prev->fMargin)
    fBase64 = FALSE;
  nLength = MultiByteToWideChar (CP_ACP, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_ACP, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  fMode = fBase64;
  n = nBits;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] == '\t' || lpwText[i] == cCRLF[0] || lpwText[i] == cCRLF[1]
                || lpwText[i] == ' ' || '\'' <= lpwText[i] && lpwText[i] <= ')'
                || '+' <= lpwText[i] && lpwText[i] <=':' || lpwText[i] == '?'
                || 'A' <= lpwText[i] && lpwText[i] <='Z'
                || 'a' <= lpwText[i] && lpwText[i] <='z')
      {
        if (fMode)
          {
            if (n > 0)
              j++;
            if (lpwText[i] == '+' || lpwText[i] == '-' || lpwText[i] == '/'
                                    || '0' <= lpwText[i] && lpwText[i] <= '9'
                                    || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                                    || 'a' <= lpwText[i] && lpwText[i] <= 'z')
              j++;
            fMode = FALSE;
          }
        j++;
        if (lpwText[i] == '+')
          j++;
      }
    else
      {
        if (!fMode)
          {
            j++;
            n = 0;
            fMode = TRUE;
          }
        n += 16;
        while (n >= 6)
          {
            j++;
            n -= 6;
          }
      }
  if (fMode && (!p->fMargin || !p->next))
    {
      if (n > 0)
        j++;
      if (!p->next)
        j++;
    }
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  j = 0;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] == '\t' || lpwText[i] == cCRLF[0] || lpwText[i] == cCRLF[1]
                || lpwText[i] == ' ' || '\'' <= lpwText[i] && lpwText[i] <= ')'
                || '+' <= lpwText[i] && lpwText[i] <= ':' || lpwText[i] == '?'
                || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                || 'a' <= lpwText[i] && lpwText[i] <= 'z')
      {
        if (fBase64)
          {
            if (nBits > 0)
              lpcText[j++] = cBase64[dwBits << 6 - nBits & 0x3f];
            if (lpwText[i] == '+' || lpwText[i] == '-' || lpwText[i] == '/'
                                    || '0' <= lpwText[i] && lpwText[i] <= '9'
                                    || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                                    || 'a' <= lpwText[i] && lpwText[i] <= 'z')
              lpcText[j++] = '-';
            fBase64 = FALSE;
          }
        lpcText[j++] = lpwText[i];
        if (lpwText[i] == '+')
          lpcText[j++] = '-';
      }
    else
      {
        if (!fBase64)
          {
            lpcText[j++] = '+';
            nBits = 0;
            fBase64 = TRUE;
          }
        dwBits = dwBits << 16 | lpwText[i];
        nBits += 16;
        while (nBits >= 6)
          {
            nBits -= 6;
            lpcText[j++] = cBase64[dwBits >> nBits & 0x3f];
          }
      }
  if (fBase64 && (!p->fMargin || !p->next))
    {
      if (nBits > 0)
        lpcText[j++] = cBase64[dwBits << 6 - nBits & 0x3f];
      if (!p->next)
        lpcText[j++] = '-';
    }
  MemoryFree (lpwText);
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:UTF-8に変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertUtf8 (HANDLE    hFile,
             LPLINEBUF p)
{
  int i, j = 0, nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPSTR lpcText;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_ACP, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_ACP, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  for (i = 0; i < nLength; i++)
    if (lpwText[i] <= 0x007f)
      j++;
    else if (lpwText[i] <= 0x07ff)
      j += 2;
    else
      j += 3;
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  j = 0;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] <= 0x007f)
      {
        lpcText[j++] = lpwText[i];
      }
    else if (lpwText[i] <= 0x07ff)
      {
        lpcText[j++] = 0xc0 | lpwText[i] >> 6;
        lpcText[j++] = 0x80 | lpwText[i] & 0x3f;
      }
    else
      {
        lpcText[j++] = 0xe0 | lpwText[i] >> 12;
        lpcText[j++] = 0x80 | lpwText[i] >> 6 & 0x3f;
        lpcText[j++] = 0x80 | lpwText[i] & 0x3f;
      }
  MemoryFree (lpwText);
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:ISO-2022-JPに変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertIso2022Jp (HANDLE    hFile,
                  LPLINEBUF p)
{
  int i = 0, j = 0;
  BOOL fMode, fResult;
  CHAR cKI[3] = {'\x1b', '$', 'B'}, cKO[3] = {'\x1b', '(', 'B'};
  DWORD dwWrite;
  LPSTR lpcText;
  static BOOL fKanji;

  if (!p->prev || !p->prev->fMargin)
    fKanji = FALSE;
  fMode = fKanji;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                                && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        if (!fMode)
          {
            j += 3;
            fMode = TRUE;
          }
        i += 2;
        j += 2;
      }
    else
      {
        if (fMode)
          {
            j += 3;
            fMode = FALSE;
          }
        i++;
        j++;
      }
  if (fMode && (!p->fMargin || !p->next))
    j += 3;
  lpcText = MemoryAlloc (j * sizeof (CHAR) + 10000);
  i = j = 0;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                                && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        if (!fKanji)
          {
            MemoryCopy (lpcText + j, cKI, sizeof (CHAR) * 3);
            j += 3;
            fKanji = TRUE;
          }
        lpcText[j] = p->lpszText[i++];
        lpcText[j + 1] = p->lpszText[i++];
        lpcText[j] -= (BYTE)lpcText[j] >= 0xe0 ? 0xc1 : 0x81;
        lpcText[j] <<= 1;
        if ((BYTE)lpcText[j + 1] >= 0x9f)
          {
            lpcText[j] += 0x22;
            lpcText[j + 1] -= 0x7e;
          }
        else
          {
            lpcText[j] += 0x21;
            lpcText[j + 1] -= (BYTE)lpcText[j + 1] >= 0x80 ? 0x20 : 0x1f;
          }
        j += 2;
      }
    else
      {
        if (fKanji)
          {
            MemoryCopy (lpcText + j, cKO, sizeof (CHAR) * 3);
            j += 3;
            fKanji = FALSE;
          }
        lpcText[j++] = p->lpszText[i++];
      }
  if (fKanji && (!p->fMargin || !p->next))
    {
      MemoryCopy (lpcText + j, cKO, sizeof (CHAR) * 3);
      j += 3;
    }
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:EUC-JPに変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertEucJp (HANDLE    hFile,
              LPLINEBUF p)
{
  int i = 0, j = 0;
  BOOL fResult;
  DWORD dwWrite;
  LPSTR lpcText;

  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        i += 2;
        j += 2;
      }
    else if (0xa1 <= (BYTE)p->lpszText[i] && (BYTE)p->lpszText[i] <= 0xfe)
      {
        i++;
        j += 2;
      }
    else
      {
        i++;
        j++;
      }
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  i = j = 0;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        lpcText[j] = p->lpszText[i++];
        lpcText[j + 1] = p->lpszText[i++];
        lpcText[j] -= (BYTE)lpcText[j] >= 0xe0 ? 0xc1 : 0x81;
        lpcText[j] <<= 1;
        if ((BYTE)lpcText[j + 1] >= 0x9f)
          {
            lpcText[j] += 0x22;
            lpcText[j + 1] -= 0x7e;
          }
        else
          {
            lpcText[j] += 0x21;
            lpcText[j + 1] -= (BYTE)lpcText[j + 1] >= 0x80 ? 0x20 : 0x1f;
          }
        lpcText[j] |= 0x80;
        lpcText[j + 1] |= 0x80;
        j += 2;
      }
    else
      {
        if (0xa1 <= (BYTE)p->lpszText[i] && (BYTE)p->lpszText[i] <= 0xfe)
          lpcText[j++] = 0x8e;
          lpcText[j++] = p->lpszText[i++];
      }
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:SHIFT_JISに変換する
    hFile,ファイルハンドル
        p,ラインバッファ
      RET,TRUE:正常終了,FALSE:エラー                                        */
static BOOL
ConvertShiftJis (HANDLE    hFile,
                 LPLINEBUF p)
{
  DWORD dwWrite;

  return WriteFile (hFile, p->lpszText,
                                    p->nLength * sizeof (CHAR), &dwWrite, NULL)
                                    && p->nLength * sizeof (CHAR) == dwWrite;
}


/*  ja:TXTファイルを保存する
    lpszFile,ファイル名
         ptw,TXTウインドウ情報
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL
SaveTextFile (LPCTSTR   lpszFile,
              LPTEXTWND ptw)
{
  BOOL fResult;
  HANDLE hFile;
  HINSTANCE hInstance;

  hInstance = GetModuleHandle (NULL);
  if (ptw->fRecycle && PathFileExists (lpszFile))
    {
      /* ja:既にあるファイルをごみ箱に移動 */
      LPTSTR lpszDelete;
      SHFILEOPSTRUCT shfo;

      lpszDelete = MemoryAlloc ((lstrlen (lpszFile) + 2) * sizeof (TCHAR));
      lstrcpy (lpszDelete, lpszFile);
      MemorySet (&shfo, 0, sizeof (SHFILEOPSTRUCT));
      shfo.hwnd = hWndClient;
      shfo.wFunc = FO_DELETE;
      shfo.pFrom = lpszDelete;
      shfo.fFlags = FOF_ALLOWUNDO | FOF_NOCONFIRMATION;
      SHFileOperation (&shfo);
      MemoryFree (lpszDelete);
    }
  /* ja:開く */
  hFile = CreateFile (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  fResult = hFile != INVALID_HANDLE_VALUE;
  if (fResult)
    {
      BOOL fEOF = FALSE;
      CHAR cText[4];
      DWORD dwSize, dwWrite;
      LPTSTR lpszAbort;
      LPLINEBUF p;
      ConvertCharSet_t ConvertText;
      const static ConvertCharSet_t ConvertCharSet[7] = {
                ConvertUnicodeLittle, ConvertUnicodeBig, ConvertUtf7,
                ConvertUtf8, ConvertIso2022Jp, ConvertEucJp, ConvertShiftJis};

      /* ja:ダイアログボックス */
      EnableWindow (hWndMain, FALSE);
      AbortDlg.fSignal = FALSE;
      AbortDlg.hEvent = NULL;
      AbortDlg.lpszText = lpszAbort = LoadText (hInstance, IDS_JOB_WRITE);
      hDlgCancel = CreateDialogParamGUI (hInstance, MAKEINTRESOURCE (DIALOG_G),
                                hWndClient, AbortDlgProc, (LPARAM)&AbortDlg);

      /* ja:書く */
      switch (ptw->uCharSet)
        {
          case 0:/* ja:UNICODE(リトルエンディアン) */
            dwSize = 2;
            cText[0] = '\xff';
            cText[1] = '\xfe';
            fResult = WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
            cText[1] = '\0';
            switch (ptw->uRetCode)
              {
                case 1: cText[0] = cCRLF[0]; break;
                case 2: cText[0] = cCRLF[1]; break;
                default:
                  dwSize = 4;
                  cText[0] = cCRLF[0];
                  cText[2] = cCRLF[1];
                  cText[3] = '\0';
              }
            break;
          case 1:/* ja:UNICODE(ビックエンディアン) */
            dwSize = 2;
            cText[0] = '\xfe';
            cText[1] = '\xff';
            fResult = WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
            cText[0] = '\0'; 
            switch (ptw->uRetCode)
              {
                case 1: cText[1] = cCRLF[0]; break;
                case 2: cText[1] = cCRLF[1]; break;
                default:
                  dwSize = 4;
                  cText[1] = cCRLF[0];
                  cText[2] = '\0';
                  cText[3] = cCRLF[1];
              }
            break;
          default:/* en:ISO-2022-JP,UTF-7,UTF-8,EUC-JP,SHIFT_JIS */
            fResult = TRUE;
            switch (ptw->uRetCode)
              {
                case 1:  dwSize = 1; cText[0] = cCRLF[0]; break;
                case 2:  dwSize = 1; cText[0] = cCRLF[1]; break;
                default: dwSize = 2; cText[0] = cCRLF[0]; cText[1] = cCRLF[1];
              }
        }
      /* ja:本文 */
      p = ptw->lpStart;
      while (p->prev)
        p = p->prev;
      ConvertText = ConvertCharSet[ptw->uCharSet];
      while (fResult && p)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (AbortDlg.fSignal)
            break;
          if (p->nLength > 0)
            fResult = ConvertText (hFile, p);
          if (fResult && !p->fMargin && p->next)
            {
              fResult = WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
              fEOF = FALSE;
            }
          else if (p->nLength > 0)
            {
              fEOF = p->lpszText[p->nLength - 1] == '\x1a';
            }
          p = p->next;
        }
      /* ja:最後に^Zを付け加える */
      if (fResult && !AbortDlg.fSignal && !fEOF && ptw->fEOF)
        {
          switch (ptw->uCharSet)
            {
              case 0: dwSize = 2; cText[0] = '\x1a'; cText[1] = '\0'; break;
              case 1: dwSize = 2; cText[0] = '\0'; cText[1] = '\x1a'; break;
              default: dwSize = 1; cText[0] = '\x1a';
            }
          fResult = WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
        }
      /* ja:終了処理 */
      CloseHandle (hFile);
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      MemoryFree (lpszAbort);
      /* ja:書き込みエラー */
      if (!fResult)
        {
          LPTSTR lpszText;

          lpszText = LoadText (hInstance, IDS_FILE_WRITE);
          MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
          MemoryFree (lpszText);
        }
      if (AbortDlg.fSignal)
        fResult = FALSE;
    }
  else
    {
      LPTSTR lpszText;

      lpszText = LoadText (hInstance, IDS_FILE_OPEN);
      MessageBox (hWndClient, lpszText,
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (lpszText);
    }
  return fResult;
}


/******************************************************************************
*                                                                             *
* ja:ファイル関数群                                                           *
*                                                                             *
******************************************************************************/
typedef struct _SAMEFILE
{
  int nCommand, nCount, nSame;
  HWND hWnd;
  LPCTSTR lpszFile;                 /* ja:ファイル名とフルパス */
} SAMEFILE, *PSAMEFILE, *LPSAMEFILE;


/*  ja:
    0:同名同パスのファイルの数を数える
    1:同名のファイルの数を数える
    2:同名同パスのファイルをフルパス+数字表示に変更する
    3:同名のファイルをフルパス表示に変更する
    4:同名同パスのファイルをフルパス表示に変更する
    5:同名のファイルを通常表示に変更する                                    */
static BOOL
CALLBACK SameFileEnumProc (HWND   hWnd,
                           LPARAM lParam)
{
  LPTSTR lpszClass;
  LPSAMEFILE lpSameFile;

  lpSameFile = (LPSAMEFILE)lParam;
  lpszClass = GetClassNameNew (hWnd);
  if (lpSameFile->hWnd != hWnd && lstrcmp (lpszClass, TEXTCLASS) == 0)
    {
      LPTEXTWND ptw;

      ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
      switch (lpSameFile->nCommand)
        {
          case 0:/* ja:同名同パスのファイルの数を数える */
            if (lstrcmpi (lpSameFile->lpszFile, ptw->lpszFile) == 0)
              {
                if (lpSameFile->nSame < ptw->nSame)
                  lpSameFile->nSame = ptw->nSame;
                lpSameFile->nCount++;
              }
            break;
          case 1:/* ja:同名のファイルの数を数える */
            if (lstrcmpi (PathFindFileName (lpSameFile->lpszFile),
                          PathFindFileName (ptw->lpszFile)) == 0)
              lpSameFile->nCount++;
            break;
          case 2:/* ja:同名同パスのファイルをフルパス+数字表示に変更する */
            if (lstrcmpi (lpSameFile->lpszFile, ptw->lpszFile) == 0)
              {
                if (ptw->nSame < 0)
                  {
                    LPTSTR lpszText;

                    ptw->nSame = 0;
                    lpszText = StringJoin (NULL, ptw->lpszFile, _T(":0"),
                                                                        NULL);
                    SetWindowText (hWnd, lpszText);
                    MemoryFree (lpszText);
                  }
                if (lpSameFile->nSame < ptw->nSame)
                  lpSameFile->nSame = ptw->nSame;
                lpSameFile->nCount++;
              }
            break;
          case 3:/* ja:同名のファイルをフルパス表示に変更する */
            if (lstrcmpi (PathFindFileName (lpSameFile->lpszFile),
                          PathFindFileName (ptw->lpszFile)) == 0)
              {
                lpSameFile->nCount++;
                if (ptw->nSame < 0)
                  SetWindowText (hWnd, ptw->lpszFile);
              }
            break;
          case 4:/* ja:同名同パスのファイルをフルパス表示に変更する */
            if (lstrcmpi (lpSameFile->lpszFile, ptw->lpszFile) == 0)
              {
                ptw->nSame = -1;
                lpSameFile->nCount++;
                SetWindowText (hWnd, ptw->lpszFile);
              }
            break;
          case 5:/* ja:同名のファイルを通常表示に変更する */
            {
              LPTSTR lpszName;

              lpszName = PathFindFileName (ptw->lpszFile);
              if (lstrcmpi (PathFindFileName (lpSameFile->lpszFile),
                                                                lpszName) == 0)
                {
                  ptw->nSame = -1;
                  lpSameFile->nCount++;
                  SetWindowText (hWnd, lpszName);
                }
            }
        }
    }
  MemoryFree (lpszClass);
  return TRUE;
}


/*  ja:ファイルを加える
        hWnd,加えようとするウインドウ
    lpszFile,加えようとするファイルのフルパス
     lpnSame,フルパスに付属する数値
         RET,加えようとするファイルのラベル                                 */
LPTSTR
AddEditFile (HWND     hWnd,
             LPCTSTR  lpszFile,
             int     *lpnSame)
{
  LPTSTR lpszTitle;
  SAMEFILE SameFile;

  *lpnSame = -1;
  SameFile.nCommand = 2;
  SameFile.nCount = 0;
  SameFile.nSame = -1;
  SameFile.hWnd = hWnd;
  SameFile.lpszFile = lpszFile;
  EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
  if (SameFile.nCount > 0)
    {
      /* ja:同名同パスがあるとき */
      *lpnSame = SameFile.nSame + 1;
      wasprintf (&lpszTitle, _T("%s:%d"), lpszFile, *lpnSame);
    }
  else
    {
      /* ja:同名同パスはないとき */
      SameFile.nCommand = 3;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
      /* ja:同名異パスがあるとき or 通常 */
      lpszTitle = StringDuplicate (SameFile.nCount > 0 ? lpszFile
                                              : PathFindFileName (lpszFile));
    }
  return lpszTitle;
}


/*  ja:ファイルを削除する
        hWnd,削除しようとするウインドウ
    lpszFile,削除しようとするファイルのフルパス                             */
VOID
DeleteEditFile (HWND    hWnd,
                LPCTSTR lpszFile)
{
  SAMEFILE SameFile;

  /* ja:同名異パスを調べる */
  SameFile.nCommand = 1;
  SameFile.nCount = 0;
  SameFile.hWnd = hWnd;
  SameFile.lpszFile = lpszFile;
  EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
  if (SameFile.nCount == 1)
    {
      /* ja:同名異パスのファイルを通常表示に変更する */
      SameFile.nCommand = 5;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
    }
  else
    {
      /* ja:同名同パスを調べる */
      SameFile.nCommand = SameFile.nCount = 0;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
      if (SameFile.nCount == 1)
        {
          /* ja:同名同パスのファイルをフルパス表示に変更する */
          SameFile.nCommand = 4;
          EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&SameFile);
        }
    }
}


/*  ja:ファイルを開く
    lpszFile,ファイル名
         RET,ウインドウ,NULL:エラー                                         */
HWND
OpenEditFile (LPCTSTR lpszFile)
{
  HINSTANCE hInstance;
  HWND hWnd;
  MDICREATESTRUCT mcs;
  LPTSTR lpszTitle;
  LPTEXTWND ptw;

  hInstance = GetModuleHandle (NULL);
  ptw = MemoryAlloc (sizeof (TEXTWND));
  ptw->ptSelect.x = -1;
  if (!lpszFile)
    {
      /* ja:新規 */
      int i;
      LPTSTR *ppszArray, lpszExt, lpszFormat;

      ppszArray = StringSplitDelimiter (lpFileType[0].lpszExt, _T(";"));
      lpszExt = ppszArray ? PathFindExtension (ppszArray[0]) : _T("");
      lpszFormat = LoadText (hInstance, IDS_OTHER_CREATE);
      wasprintf (&ptw->lpszFile, lpszFormat, nFileCount++ % 10000, lpszExt);
      MemoryFree (ppszArray);
      MemoryFree (lpszFormat);
      ptw->fCreate = TRUE;
      ptw->nID = lpFileType[0].fAssociate ? lpFileType[0].nID : -1;
      ptw->nMargin = lpFileType[0].nMargin;
      ptw->nTab = lpFileType[0].nTab;
      ptw->fAutoIndent = lpFileType[0].fAutoIndent;
      ptw->fCode = lpFileType[0].fCode;
      ptw->fCRLF = lpFileType[0].fCRLF;
      ptw->fEOF = lpFileType[0].fEOF;
      ptw->fLimit = lpFileType[0].fLimit;
      ptw->fOverWrite = lpFileType[0].fOverWrite;
      ptw->fRecycle = lpFileType[0].fRecycle;
      ptw->fSpace = lpFileType[0].fSpace;
      ptw->fSysColor = lpFileType[0].fSysColor;
      ptw->fTabConv = lpFileType[0].fTabConv;
      ptw->fGline = lpFileType[0].fGline;
      ptw->fMline = lpFileType[0].fMline;
      ptw->fUline = lpFileType[0].fUline;
      ptw->fVline = lpFileType[0].fVline;
      ptw->uRetCode = lpFileType[0].uRetCode;
      ptw->uCharSet = (lpFileType[0].dwCharSet & 15) - 1;
      MemoryCopy (ptw->crColor, lpFileType[0].crColor, sizeof (COLORREF) * 12);
      ptw->LogFont = lpFileType[0].LogFont;
    }
  else
    {
      int i;

      ptw->lpszFile = GetLongFileNew (lpszFile);
      for (i = 0; i < nFileType; i++)
        {
          LPTSTR *ppszArray;

          ppszArray = StringSplitDelimiter (lpFileType[i].lpszExt, _T(";"));
          if (ppszArray)
            {
              int j;
              BOOL fMatch = FALSE;

              for (j = 0; ppszArray[j]; j++)
                if (PathMatchSpec (ptw->lpszFile, ppszArray[j]))
                  {
                    fMatch = TRUE;
                    break;
                  }
              MemoryFree (ppszArray);
              if (fMatch)
                break;
            }
        }
      if (i >= nFileType)
        for (i = 0; i < nFileType; i++)
          if (lstrcmp (lpFileType[i].lpszExt, _T("*")) == 0)
            break;
      if (i >= nFileType)
        i = 0;
      ptw->nID = lpFileType[i].fAssociate ? lpFileType[i].nID : -1;
      ptw->nMargin = lpFileType[i].nMargin;
      ptw->nTab = lpFileType[i].nTab;
      ptw->fAutoIndent = lpFileType[i].fAutoIndent;
      ptw->fCode = lpFileType[i].fCode;
      ptw->fCRLF = lpFileType[i].fCRLF;
      ptw->fEOF = lpFileType[i].fEOF;
      ptw->fLimit = lpFileType[i].fLimit;
      ptw->fOverWrite = lpFileType[i].fOverWrite;
      ptw->fRecycle = lpFileType[i].fRecycle;
      ptw->fSpace = lpFileType[i].fSpace;
      ptw->fSysColor = lpFileType[i].fSysColor;
      ptw->fTabConv = lpFileType[i].fTabConv;
      ptw->fGline = lpFileType[i].fGline;
      ptw->fMline = lpFileType[i].fMline;
      ptw->fUline = lpFileType[i].fUline;
      ptw->fVline = lpFileType[i].fVline;
      MemoryCopy (ptw->crColor, lpFileType[i].crColor, sizeof (COLORREF) * 12);
      ptw->LogFont = lpFileType[i].LogFont;
      if (PathFileExists (ptw->lpszFile) && !OpenTextFile (ptw,
                                                    lpFileType[i].dwCharSet,
                                                    lpFileType[i].fNegotiate,
                                                    lpFileType[i].uRetCode))
        {
          MemoryFree (ptw);
          return NULL;
        }
    }
  if (!ptw->lpStart)
    {
      /* ja:ファイルが0バイトのとき */
      ptw->nOff = 0;
      ptw->nMax = 1;
      ptw->lpStart = MemoryAlloc (sizeof (LINEBUF));
    }
  lpszTitle = AddEditFile (NULL, ptw->lpszFile, &ptw->nSame);
  mcs.szClass = TEXTCLASS;
  mcs.szTitle = lpszTitle;
  mcs.hOwner = hInstance;
  mcs.x = mcs.y = mcs.cx = mcs.cy = CW_USEDEFAULT;
  mcs.style = WS_HSCROLL | WS_VSCROLL;
  mcs.lParam = (LPARAM)ptw;
  hWnd = (HWND)SendMessage (hWndClient, WM_MDIGETACTIVE, 0, 0);
  if (IsWindow (hWnd))
    {
      if (IsZoomed (hWnd))
        mcs.style |= WS_MAXIMIZE;
    }
  else if (fZoomed)
    {
      mcs.style |= WS_MAXIMIZE;
    }
  hWnd = (HWND)SendMessage (hWndClient, WM_MDICREATE, 0, (LPARAM)&mcs);
  MemoryFree (lpszTitle);
  /* ja:新規作成ではなく、ファイルが存在する時には履歴に加える */
  if (lpszFile && PathFileExists (lpszFile))
    SetHistory (ptw->lpszFile, hWndMain, nHistory, MENUTOP, MENUFILE);
  return hWnd;
}
