package pantheon.withDao;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.util.ArrayList;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

import pantheon.ItemService;
import pantheon.LoginService;
import pantheon.model.Item;
import pantheon.model.ItemProperty;
import pantheon.model.ItemType;
import pantheon.model.Label;
import pantheon.model.PropertyType;
import pantheon.model.User;
import pantheon.user.UserType;

/**
 * @TransactionConfigurationと@Transactionalで各メソッド実行後は自動的にロールバックされる
 * 
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml" })
@TransactionConfiguration
@Transactional
public class ItemServiecWithDaoTest {

	@Autowired
	private LoginService loginService;

	@Autowired
	private ItemService itemService;

	@Test
	public void testInsert() {

		// *************************
		// ユーザ追加
		// *************************
		assertNull(loginService.findByName("user1"));
		loginService.createUser(UserType.ROLE_USER, "user1", "testUser1", "メモ１");

		User user = loginService.findByName("user1");
		assertNotNull(user);
		long userId = user.getId();

		// *************************
		// その他item追加
		// *************************
		long itemId;
		{
			Item item = new Item();
			item.setName("動的言語");
			item.setType(ItemType.NOT_CLASSIFICATION);

			ItemProperty property = new ItemProperty();
			property.setIndex(0);
			property.setLabel(Label.NOT_CLASSIFICATION);
			property.setType(PropertyType.POINT);
			property.setFromYear(2000);
			property.setText("データ１");

			item.setPropertyList(new ArrayList<ItemProperty>());
			item.getPropertyList().add(property);

			assertNull(itemService.findByName("動的言語"));
			itemId = itemService.insert(item, userId);
		}

		{
			// 最新の第2版を取得
			Item item = itemService.findByIdWithProperties(itemId);
			assertThat(1, equalTo(item.getVersion()));

			// プロパティが追加されたことを確認
			assertThat(1, equalTo(item.getPropertyList().size()));
			ItemProperty property = item.getPropertyList().get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2000, equalTo(property.getFromYear()));
			assertThat("データ１", equalTo(property.getText()));
		}
	}

	@Test
	public void testUpdate() {

		// *************************
		// ユーザ追加
		// *************************
		assertNull(loginService.findByName("user1"));
		loginService.createUser(UserType.ROLE_USER, "user1", "testUser1", "メモ１");

		User user = loginService.findByName("user1");
		assertNotNull(user);
		long userId = user.getId();

		// *************************
		// その他item追加
		// *************************
		long itemId;
		{
			Item item = new Item();
			item.setName("動的言語");
			item.setType(ItemType.NOT_CLASSIFICATION);

			assertNull(itemService.findByName("動的言語"));
			itemId = itemService.insert(item, userId);
		}

		// *************************
		// その他item変更
		// *************************
		{
			Item item = itemService.findByIdWithProperties(itemId);
			assertNotNull(item);
			assertThat("動的言語", equalTo(item.getName()));
			assertThat(1, equalTo(item.getVersion()));

			// 変更情報追加
			item.setName("動的プログラミング言語");

			ItemProperty property = new ItemProperty();
			property.setItemId(itemId);
			property.setIndex(0);
			property.setLabel(Label.NOT_CLASSIFICATION);
			property.setType(PropertyType.POINT);
			property.setFromYear(2000);
			property.setText("データ１");

			item.setPropertyList(new ArrayList<ItemProperty>());
			item.getPropertyList().add(property);

			itemService.update(item, userId);
		}

		{
			// 初版の名称では検索できなくなったことを確認
			assertNull(itemService.findByName("動的言語"));

			// 最新の第2版を取得
			Item item = itemService.findByIdWithProperties(itemId);
			assertThat(2, equalTo(item.getVersion()));

			// 最新版では、プロパティが追加されたことを確認
			ItemProperty property = item.getPropertyList().get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2000, equalTo(property.getFromYear()));
			assertThat("データ１", equalTo(property.getText()));
		}

		// *************************
		// さらに1プロパティを追加
		// *************************
		{
			Item item = itemService.findByIdWithProperties(itemId);

			// 変更情報追加
			ItemProperty property = new ItemProperty();
			property.setItemId(itemId);
			property.setIndex(1);
			property.setLabel(Label.NOT_CLASSIFICATION);
			property.setType(PropertyType.POINT);
			property.setFromYear(2000);
			property.setText("データ２");
			item.getPropertyList().add(property);

			itemService.update(item, userId);
		}

		// *************************
		// 第3版取得
		// *************************
		{
			Item item = itemService.findByIdWithProperties(itemId);
			assertThat(3, equalTo(item.getVersion()));

			// 最新版では、プロパティが追加されたことを確認
			assertThat(2, equalTo(item.getPropertyList().size()));

			assertThat("データ１", equalTo(item.getPropertyList().get(0).getText()));
			assertThat("データ２", equalTo(item.getPropertyList().get(1).getText()));
		}

		// *************************
		// 変更履歴取得
		// *************************
		{
			// 初版
			Item item = itemService.findByIdWithProperties(itemId, 1);
			assertThat(1, equalTo(item.getVersion()));
			assertThat(0, equalTo(item.getPropertyList().size()));
		}

		{
			// 2版
			Item item = itemService.findByIdWithProperties(itemId, 2);
			assertThat(2, equalTo(item.getVersion()));
			assertThat(1, equalTo(item.getPropertyList().size()));
			
			assertThat("データ１", equalTo(item.getPropertyList().get(0).getText()));
		}

		{
			// 3版
			Item item = itemService.findByIdWithProperties(itemId, 3);
			assertThat(3, equalTo(item.getVersion()));
			assertThat(2, equalTo(item.getPropertyList().size()));
			
			assertThat("データ１", equalTo(item.getPropertyList().get(0).getText()));
			assertThat("データ２", equalTo(item.getPropertyList().get(1).getText()));
		}
	}
}
