package pantheon.withDao;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

import pantheon.BookService;
import pantheon.ItemService;
import pantheon.LoginService;
import pantheon.PeopleService;
import pantheon.model.Book;
import pantheon.model.Item;
import pantheon.model.ItemProperty;
import pantheon.model.Label;
import pantheon.model.People;
import pantheon.model.PropertyType;
import pantheon.model.User;
import pantheon.user.UserType;

/**
 * @TransactionConfigurationと@Transactionalで各メソッド実行後は自動的にロールバックされる
 * 
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml" })
@TransactionConfiguration
@Transactional
public class DaoTest {

	@Autowired
	private LoginService loginService;

	@Autowired
	private PeopleService peopleService;

	@Autowired
	private BookService bookService;

	@Autowired
	private ItemService itemService;

	@Test
	public void testFindByItem() {
		// *************************
		// ユーザ追加
		// *************************
		assertNull(loginService.findByName("ダニール"));
		loginService.createUser(UserType.ROLE_USER, "ダニール", "testUser1", "メモ１");

		User user = loginService.findByName("ダニール");
		assertNotNull(user);
		long userId = user.getId();

		// *************************
		// ユーザ名変更
		// *************************
		loginService.updateUser(userId, "ダニール・オリバー", "");
		assertNull(loginService.findByName("ダニール"));
		assertNotNull(loginService.findByName("ダニール・オリバー"));

		// *************************
		// 書籍追加
		// *************************
		long book1Id;
		{
			Book book = new Book();
			book.setName("銀河百科事典");
			book.setFromYear(12050);// この年は適当

			assertNull(bookService.findByName("銀河百科事典"));
			book1Id = bookService.insert(book, userId);
		}

		// *************************
		// 人物追加
		// *************************
		long peopleId;
		{
			People hari = new People();
			hari.setName("セルダン");
			hari.setFromYear(11988);
			hari.setToYear(12069);

			assertNull(peopleService.findByName("セルダン"));
			peopleId = peopleService.insert(hari, new Long[] { book1Id }, userId);
		}

		// *************************
		// 登録確認
		// *************************
		{
			// 関連確認
			List<Item> bookList = itemService.findAllTargetItem(peopleId, Label.WRITTEN);
			assertThat(1, equalTo(bookList.size()));
			assertThat("銀河百科事典", equalTo(bookList.get(0).getName()));

			List<Item> peopleList = itemService.findAllItemByTargetItem(Label.WRITTEN, book1Id);
			assertThat(1, equalTo(peopleList.size()));
			assertThat("セルダン", equalTo(peopleList.get(0).getName()));
		}

		{
			Item hari = itemService.findByIdWithProperties(peopleId);
			assertThat(1, equalTo(hari.getPropertyList().size()));
			assertThat(peopleId, equalTo(hari.getPropertyList().get(0).getItemId()));
			assertThat(book1Id, equalTo(hari.getPropertyList().get(0).getTargetItemId()));
		}

		// *************************
		// 書籍1変更
		// *************************
		{
			Book book = bookService.findByName("銀河百科事典");
			assertNotNull(book);

			book.setName("銀河百科事典　初版");
			bookService.update(book, userId);

			assertNull(bookService.findByName("銀河百科事典"));
		}

		Book encyclopaediaGalactica = bookService.findByName("銀河百科事典　初版");
		assertNotNull(encyclopaediaGalactica);

		// *************************
		// 書籍2追加
		// *************************
		long book2Id;
		{
			Book book2 = new Book();
			book2.setName("心理歴史学");
			book2.setFromYear(12051);// この年は適当
			book2Id = bookService.insert(book2, userId);
		}

		// *************************
		// 人物変更
		// *************************
		{
			Item hari = itemService.findByIdWithProperties(peopleId);
			assertNotNull(hari);

			hari.setName("ハリ・セルダン");
			peopleService.update(hari, new Long[] { encyclopaediaGalactica.getId(), book2Id, }, userId);

			assertNull(peopleService.findByName("セルダン"));
			assertNotNull(peopleService.findByName("ハリ・セルダン"));
		}

		// *************************
		// 変更確認
		// *************************
		{
			List<Item> bookList = itemService.findAllTargetItem(Long.valueOf(peopleId), Label.WRITTEN);
			assertThat(2, equalTo(bookList.size()));
			assertThat("銀河百科事典　初版", equalTo(bookList.get(0).getName()));
			assertThat("心理歴史学", equalTo(bookList.get(1).getName()));
		}

		// *************************
		// 履歴確認
		// *************************
		{
			// 初版
			Item people = itemService.findById(peopleId, 1);
			assertThat("セルダン", equalTo(people.getName()));

			List<Item> bookList = itemService.findAllTargetItem(peopleId, Label.WRITTEN, 1);
			assertThat(1, equalTo(bookList.size()));
			assertThat("銀河百科事典　初版", equalTo(bookList.get(0).getName()));
			// TODO 履歴作成時の版のデータを返すように（ここでは"初版"無しが正しい）
		}

		{
			// ２版
			Item people = itemService.findById(peopleId, 2);
			assertThat("ハリ・セルダン", equalTo(people.getName()));

			List<Item> bookList = itemService.findAllTargetItem(Long.valueOf(peopleId), Label.WRITTEN, 2);
			assertThat(2, equalTo(bookList.size()));
			assertThat("銀河百科事典　初版", equalTo(bookList.get(0).getName()));
			assertThat("心理歴史学", equalTo(bookList.get(1).getName()));
		}
	}

	@Test
	public void testItemProperty() {

		long itemId = 7L;
		int itemVersion = 4;
		{
			Item item = new Item();
			item.setId(itemId);
			item.setVersion(itemVersion);
			item.setPropertyList(new ArrayList<ItemProperty>());

			// 追加
			ItemProperty property = new ItemProperty();
			property.setItemId(itemId);
			property.setIndex(0);
			property.setLabel(Label.NOT_CLASSIFICATION);
			property.setType(PropertyType.POINT);
			property.setFromYear(2000);
			property.setText("aaa");
			item.getPropertyList().add(property);

			itemService.insertAllProperties(item);
		}

		{
			// 履歴取得
			ItemProperty property = itemService.findAllPropertyByItemId(itemId, itemVersion).get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2000, equalTo(property.getFromYear()));
			assertThat("aaa", equalTo(property.getText()));
		}

		{
			// 取得
			ItemProperty property = itemService.findAllPropertyByItemId(itemId).get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2000, equalTo(property.getFromYear()));
			assertThat("aaa", equalTo(property.getText()));

			// 変更
			property.setFromYear(2001);
			property.setText("bbb");

			// TODO ここのitem作成は本来はfind取得したもの
			Item item = new Item();
			item.setId(itemId);
			item.setVersion(itemVersion + 1);
			item.setPropertyList(new ArrayList<ItemProperty>());
			item.getPropertyList().add(property);
			itemService.insertAllProperties(item);
		}

		{
			// 取得
			ItemProperty property = itemService.findAllPropertyByItemId(itemId).get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2001, equalTo(property.getFromYear()));
			assertThat("bbb", equalTo(property.getText()));
		}

		{
			// 履歴取得
			ItemProperty property = itemService.findAllPropertyByItemId(itemId, itemVersion + 1).get(0);
			assertNotNull(property);
			assertThat(itemId, equalTo(property.getItemId()));
			assertThat(Label.NOT_CLASSIFICATION, equalTo(property.getLabel()));
			assertThat(PropertyType.POINT, equalTo(property.getType()));
			assertThat(2001, equalTo(property.getFromYear()));
			assertThat("bbb", equalTo(property.getText()));
		}
	}
}
