package pantheon.web;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

import org.apache.commons.lang.StringUtils;
import org.junit.Test;

import pantheon.model.ItemType;
import pantheon.model.People;

public class ItemValidatorTestForPeople extends AbstractValidatorTest<ItemCommand> {
	public ItemValidatorTestForPeople() {
		super(ItemCommand.class, ItemValidator.class);
	}

	private void setSuccessData() {
		command.setItemType(ItemType.PEOPLE);
		command.setIsInsert(Boolean.FALSE);
		command.setName(StringUtils.repeat("全", People.NAME_LENGTH));
		command.setKanaName(StringUtils.repeat("カ", People.NAME_LENGTH));
		command.setShortName(StringUtils.repeat("全", People.SHORT_NAME_LENGTH));
		command.setWikiName(StringUtils.repeat("全", People.NAME_LENGTH));
		command.setEnglishName(StringUtils.repeat("a", People.ENGLISH_NAME_LENGTH));
		command.setFromYear(StringUtils.repeat("1", People.YEAR_LENGTH));
		command.setToYear(StringUtils.repeat("2", People.YEAR_LENGTH));
		command.setVersion(1);
	}

	@Test
	public void testSuccess() throws Exception {

		// テスト準備
		setSuccessData();

		// 実行
		validator.validate(command, exception);

		// 検証
		assertThat(0, equalTo(exception.getErrorCount()));
	}

	@Test
	public void testBlankError() throws Exception {

		// テスト準備
		command.setItemType(ItemType.PEOPLE);
		command.setIsInsert(Boolean.FALSE);

		// 実行
		validator.validate(command, exception);

		// 検証
		assertThat(4, equalTo(exception.getErrorCount()));
		assertThat(ValidatorError.ERROR_VERSION.name(), equalTo(exception.getFieldError("version").getCode()));
		assertThat(ValidatorError.ERROR_BLANK.name(), equalTo(exception.getFieldError("name").getCode()));
		assertThat(ValidatorError.ERROR_BLANK.name(), equalTo(exception.getFieldError("fromYear").getCode()));
		assertThat(ValidatorError.ERROR_BLANK.name(), equalTo(exception.getFieldError("toYear").getCode()));

		assertThat("生誕年", equalTo(exception.getFieldError("fromYear").getArguments()[0]));
		assertThat("死没年", equalTo(exception.getFieldError("toYear").getArguments()[0]));
	}

	@Test
	public void testOverLength() throws Exception {

		// テスト準備
		setSuccessData();
		command.setName(command.getName() + "1");
		command.setKanaName(command.getKanaName() + "1");
		command.setShortName(command.getShortName() + "1");
		command.setWikiName(command.getWikiName() + "1");
		command.setEnglishName(command.getEnglishName() + "1");
		command.setFromYear(command.getFromYear() + "a");
		command.setToYear(command.getToYear() + "a");

		// 実行
		validator.validate(command, exception);

		// 検証
		assertThat(7, equalTo(exception.getErrorCount()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("name").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("kanaName").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("shortName").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("wikiName").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("englishName").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("fromYear").getCode()));
		assertThat(ValidatorError.ERROR_LENGTH.name(), equalTo(exception.getFieldError("toYear").getCode()));
	}

	@Test
	public void testNumericError() throws Exception {

		// テスト準備
		setSuccessData();
		command.setFromYear("a");
		command.setToYear("b");

		// 実行
		validator.validate(command, exception);

		// 検証
		assertThat(2, equalTo(exception.getErrorCount()));
		assertThat(ValidatorError.ERROR_NUMERIC.name(), equalTo(exception.getFieldError("fromYear").getCode()));
		assertThat(ValidatorError.ERROR_NUMERIC.name(), equalTo(exception.getFieldError("toYear").getCode()));
	}

	@Test
	public void testKanaError() throws Exception {

		// テスト準備
		setSuccessData();
		command.setKanaName("あ");

		// 実行
		validator.validate(command, exception);

		// 検証
		assertThat(1, equalTo(exception.getErrorCount()));
		assertThat(ValidatorError.ERROR_ZENKAKU_KATA_KANA.name(), equalTo(exception.getFieldError("kanaName").getCode()));
	}
}
