package pantheon.web;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.util.Assert;
import org.springframework.validation.BindException;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.view.RedirectView;

import pantheon.ItemService;
import pantheon.PageService;
import pantheon.PeopleService;
import pantheon.model.ItemType;
import pantheon.model.Page;
import pantheon.model.PageRecord;
import pantheon.model.PageType;
import pantheon.model.meta.Century;

public class UpdatePageController extends PantheonSimpleFormController<PageCommand> {

	private String errViewName;
	private PageService pageService;
	private PeopleService peopleService;
	private ItemService itemService;

	public UpdatePageController() {
		super();
	}

	@Override
	protected ModelAndView showForm(HttpServletRequest request, HttpServletResponse response, BindException errors) throws Exception {
		if (errors.hasErrors()) {
			return super.showForm(request, response, errors);
		}

		// 更新対象のページを取得
		String pId = request.getParameter("pageId");
		if (!StringUtils.isNumeric(pId)) {
			return new ModelAndView(errViewName);
		}
		int pageId = Integer.parseInt(request.getParameter("pageId"));

		Page page = pageService.findById(pageId);
		Assert.notNull(page);

		List<Long> selectedPeopleIds = new ArrayList<Long>();
		List<Long> selectedNotClassification = new ArrayList<Long>();
		for (PageRecord pageRecord : page.getRecordList()) {
			if (pageRecord.getItem().getType() == ItemType.PEOPLE) {
				selectedPeopleIds.add(pageRecord.getItem().getId());
			} else if (pageRecord.getItem().getType() == ItemType.NOT_CLASSIFICATION) {
				selectedNotClassification.add(pageRecord.getItem().getId());
			}
		}
		Long[] selectedPeopleArray = (Long[]) selectedPeopleIds.toArray(new Long[selectedPeopleIds.size()]);
		Long[] selectedNotClassificationArray = (Long[]) selectedNotClassification.toArray(new Long[selectedNotClassification.size()]);

		// 人物の前リストを取得
		Map<String, Object> myModel = referenceData(request);

		PageCommand pageCommand = new PageCommand();
		pageCommand.setId(pageId);
		pageCommand.setName(page.getName());
		pageCommand.setPageTypeCode(String.valueOf(page.getType().getCode()));
		pageCommand.setFromCentury(page.getFromCentury().getIndex());
		pageCommand.setToCentury(page.getToCentury().getIndex());
		pageCommand.setOriginalPage(page);
		pageCommand.setPeopleIds(selectedPeopleArray);
		pageCommand.setNotClassificationItemIds(selectedNotClassificationArray);

		myModel.put(COMMAND_NAME, pageCommand);

		return new ModelAndView(getFormView(), myModel);
	}

	@Override
	protected ModelAndView onSubmit(HttpServletRequest request, HttpServletResponse response, Object command, BindException errors) throws Exception {
		logger.info("UpdatePageController#onSubmit");

		PageCommand pageCommand = (PageCommand) command;

		// 名前重複チェック
		String inputName = pageCommand.getName();
		Page sameNamePage = pageService.findByName(inputName);
		if (sameNamePage != null && sameNamePage.getId() != pageCommand.getId()) {
			errors.addError(createSameNameError("name", inputName));
			return showForm(request, response, errors);
		}

		PageType pageType = PageType.getByCode(Integer.parseInt(pageCommand.getPageTypeCode()));
		Page page = new Page(pageCommand.getId(), inputName, pageType, pageCommand.getFromCentury(), pageCommand.getToCentury());

		pageService.update(page, pageCommand.getPeopleIds(), pageCommand.getNotClassificationItemIds());

		// TODO 本当は元の年表に戻りたい
		// String viewName = getSuccessView() + "?pageId=" +
		// pageCommand.getId();
		// return new ModelAndView(new RedirectView(viewName));
		return new ModelAndView(new RedirectView(getSuccessView()));

	}

	@Override
	protected Map<String, Object> referenceData(HttpServletRequest request) throws Exception {
		Map<String, Object> myModel = new HashMap<String, Object>();

		myModel.put("peopleList", peopleService.findAll());
		myModel.put("notClassificationItemList", itemService.findAllItem(ItemType.NOT_CLASSIFICATION));
		myModel.put("centuryList", Century.getAllCenturyList());
		myModel.put("pageTypeArray", PageType.values());
		return myModel;
	}

	@Required
	public void setErrViewName(String errViewName) {
		this.errViewName = errViewName;
	}

	@Required
	public void setPageService(PageService pageService) {
		this.pageService = pageService;
	}

	@Required
	public void setPeopleService(PeopleService peopleService) {
		this.peopleService = peopleService;
	}

	@Required
	public void setItemService(ItemService itemService) {
		this.itemService = itemService;
	}

}
