package pantheon.web;

import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.SimpleFormController;
import org.springframework.web.servlet.view.RedirectView;

import pantheon.LoginService;
import pantheon.model.User;
import pantheon.user.LoginUser;
import pantheon.user.UserUtility;

public class UpdateLoginUserController extends SimpleFormController {
	/** Logger for this class and subclasses */
	protected final Log logger = LogFactory.getLog(getClass());
	private LoginService loginService;

	public UpdateLoginUserController() {
		super();
		setCommandName("LoginUserCommand");
		setCommandClass(LoginUserCommand.class);
	}

	@Override
	protected ModelAndView showForm(HttpServletRequest request, HttpServletResponse response, BindException errors) throws Exception {
		logger.debug("UpdateLoginUserController#showForm()");

		if (errors.hasErrors()) {
			return super.showForm(request, response, errors);
		}

		LoginUser loginUser = UserUtility.getLoginUser(request);
		User user = loginService.findByUserId(loginUser.getId());

		LoginUserCommand loginUserCommand = new LoginUserCommand();
		loginUserCommand.setName(user.getUserName());
		loginUserCommand.setNote(user.getNote());

		Map<String, Object> model = new HashMap<String, Object>();
		model.put("LoginUserCommand", loginUserCommand);
		model.put("LoginUser", user);

		return new ModelAndView(getFormView(), model);
	}

	@Override
	protected ModelAndView onSubmit(HttpServletRequest request, HttpServletResponse response, Object command, BindException errors) throws Exception {
		logger.debug("UpdateLoginUserController#onSubmit()");

		LoginUserCommand userCommand = (LoginUserCommand) command;

		// ログインユーザ情報取得
		LoginUser loginUser = UserUtility.getLoginUser(request);

		// 名前重複チェック
		User sameNameUser = loginService.findByName(userCommand.getName());
		if (sameNameUser != null && sameNameUser.getId() != loginUser.getId()) {
			ObjectError error = new FieldError("LoginUserCommand", "name", ValidatorError.editSameNameErrMessage(userCommand.getName()));
			errors.addError(error);
			return super.showForm(request, response, errors);
		}

		loginService.updateUser(loginUser.getId(), userCommand.getName(), userCommand.getNote());
		loginUser.setUsername(userCommand.getName());

		return new ModelAndView(new RedirectView(getSuccessView()));
	}

	@Override
	protected Map<String, Object> referenceData(HttpServletRequest request) throws Exception {
		LoginUser loginUser = UserUtility.getLoginUser(request);

		Map<String, Object> model = new HashMap<String, Object>();
		model.put("LoginUser", loginUser);

		return model;
	}

	@Required
	public void setLoginService(LoginService loginService) {
		this.loginService = loginService;
	}

}
