package pantheon.web;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.SimpleFormController;
import org.springframework.web.servlet.view.RedirectView;

import pantheon.ItemService;
import pantheon.PeopleService;
import pantheon.model.Book;
import pantheon.model.Item;
import pantheon.model.ItemType;
import pantheon.user.LoginUser;
import pantheon.user.UserUtility;

public class CreateItemController extends SimpleFormController {

	final static int PORPERTY_SIZE = 10;

	/** Logger for this class and subclasses */
	protected final Log logger = LogFactory.getLog(getClass());
	private ItemType itemType;
	private ItemService itemService;
	private PeopleService peopleService;

	public CreateItemController() {
		super();
		setCommandName("ItemCommand");
		setCommandClass(ItemCommand.class);
	}

	@Override
	protected ModelAndView showForm(HttpServletRequest request, HttpServletResponse response, BindException errors) throws Exception {
		if (errors.hasErrors()) {
			return super.showForm(request, response, errors);
		}

		Map<String, Object> myModel = referenceData(request);

		ItemCommand itemCommand = new ItemCommand();
		itemCommand.setItemType(itemType);
		itemCommand.setPropertyFromYear(new String[PORPERTY_SIZE]);
		itemCommand.setPropertyText(new String[PORPERTY_SIZE]);

		myModel.put("ItemCommand", itemCommand);
		myModel.put("itemType", itemType);

		return new ModelAndView(getFormView(), myModel);
	}

	@Override
	protected boolean suppressValidation(HttpServletRequest request, Object command) {
		ItemCommand itemCommand = (ItemCommand) command;
		itemCommand.setIsInsert(Boolean.TRUE);
		itemCommand.setItemType(itemType);
		return super.suppressValidation(request, command);
	}

	@Override
	protected ModelAndView onSubmit(HttpServletRequest request, HttpServletResponse response, Object command, BindException errors) throws Exception {
		ItemCommand itemCommand = (ItemCommand) command;

		// 名前重複チェック
		if (itemService.findByName(itemCommand.getName()) != null) {
			ObjectError error = new FieldError("ItemCommand", "name", ValidatorError.editSameNameErrMessage(itemCommand.getName()));
			errors.addError(error);
			return super.showForm(request, response, errors);
		}

		LoginUser user = UserUtility.getLoginUser(request);

		Item item = new Item();
		item.setType(itemType);
		item.setName(itemCommand.getName());
		item.setKanaName(itemCommand.getKanaName());
		item.setShortName(itemCommand.getShortName());
		item.setWikiName(itemCommand.getWikiName());
		item.setEnglishName(itemCommand.getEnglishName());
		if (item.getType().isHasFromYear()) {
			item.setFromYear(Integer.parseInt(itemCommand.getFromYear()));
		}
		if (item.getType().isHasToYear()) {
			item.setToYear(Integer.parseInt(itemCommand.getToYear()));
		}

		item.setPropertyList(itemCommand.getPointPropertyList());

		if (item.getType() == ItemType.PEOPLE) {
			peopleService.insert(item, itemCommand.getPropertyBookIds(), user.getId());
		} else {
			itemService.insert(item, user.getId());
		}

		return new ModelAndView(new RedirectView(getSuccessView()));
	}

	@Override
	protected Map<String, Object> referenceData(HttpServletRequest request) throws Exception {
		Map<String, Object> myModel = new HashMap<String, Object>();
		myModel.put("itemType", this.itemType);

		if (this.itemType == ItemType.PEOPLE) {
			List<Book> bookList = itemService.findAllBook();
			myModel.put("bookList", bookList);
		}

		return myModel;
	}

	@Required
	public void setItemType(String itemType) {
		this.itemType = ItemType.valueOf(itemType);
	}

	@Required
	public void setItemService(ItemService itemService) {
		this.itemService = itemService;
	}

	@Required
	public void setPeopleService(PeopleService peopleService) {
		this.peopleService = peopleService;
	}

}
