package pantheon.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;

import pantheon.model.meta.Century;

public class Page extends AbstractPanteonModel {

	private final int id;

	private String name;

	private PageType type;

	/** ページに表示する最初の世紀 */
	private Century fromCentury;

	/** ページに表示する最後の世紀 */
	private Century toCentury;

	private List<PageRecord> recordList;

	private String jointPeopleIds;
	private String jointNotClassificationItemIds;

	/** 永続化されていない新規ページを作成するとき */
	public Page(String name, PageType type, int fromCentury, int toCentury) {
		this(-1, name, type, Century.getByIndex(fromCentury), Century.getByIndex(toCentury));
	}

	public Page(int id, String name, PageType type, int fromCentury, int toCentury) {
		this(id, name, type, Century.getByIndex(fromCentury), Century.getByIndex(toCentury));
	}

	public Page(int id, String name, PageType type, Century fromCentury, Century toCentury) {
		this.id = id;
		this.name = name;
		this.type = type;
		this.recordList = new ArrayList<PageRecord>();
		this.fromCentury = fromCentury;
		this.toCentury = toCentury;
	}

	public void setPeopleList(List<Item> peopleList) {
		Collections.sort(peopleList, new People.ItemFromToComparator());

		this.recordList = new ArrayList<PageRecord>();

		// ページ全体の表示期間
		int startYear = this.fromCentury.getFromYear();
		int endYear = this.toCentury.getToYear();
		int span = endYear - startYear + 1;// TODO
		// このプラス1はなんだっけ？from,toが同じ年でも最初と最後で1年の幅があるから？

		for (Item people : peopleList) {
			PageRecord record = new PageRecord(people);

			int sspan = people.getFromYear() - startYear;
			double startPosition = sspan / (double) span;

			int espan = people.getToYear() - startYear + 1;
			double endPosition = espan / (double) span;

			record.setStartPosition(startPosition);
			record.setEndPosition(endPosition);

			recordList.add(record);
		}
	}

	public void setNotClassificationList(List<Item> notClassificationList) {
		for (Item item : notClassificationList) {
			PageRecord record = new PageRecord(item);
			recordList.add(record);
		}
	}

	/**
	 * 人物情報レコード配下 の 書籍情報の時点 の 比率を計算する
	 */
	public void calculateBookDataInAllPeopleRecords() {

		// ページ全体の表示期間
		int startYear = this.fromCentury.getFromYear();
		int endYear = this.toCentury.getToYear();
		int span = endYear - startYear + 1;// TODO
		// このプラス1はなんだっけ？from,toが同じ年でも最初と最後で1年の幅があるから？

		for (PageRecord record : recordList) {

			List<PageRecord.BookRecord> subList = new ArrayList<PageRecord.BookRecord>();

			for (Item subItem : record.getSubItemList()) {
				// このsubItemは今のところBookを想定している

				int sspan = subItem.getFromYear() - startYear;
				double startPosition = sspan / (double) span;

				PageRecord.BookRecord bookRecord = new PageRecord.BookRecord(subItem);
				bookRecord.setStartPosition(startPosition);
				subList.add(bookRecord);
			}

			record.setSubList(subList);
		}
	}

	public void calculatePointPropertyInAllItemRecords() {

		// ページ全体の表示期間
		int startYear = this.fromCentury.getFromYear();
		int endYear = this.toCentury.getToYear();
		int span = endYear - startYear + 1;// TODO
		// このプラス1はなんだっけ？from,toが同じ年でも最初と最後で1年の幅があるから？

		for (PageRecord record : recordList) {
			if (record.getItem().getPropertyList() == null) {
				continue;
			}

			List<PageRecord.PropertyRecord> subList = new ArrayList<PageRecord.PropertyRecord>();

			for (ItemProperty itemProperty : record.getItem().getPropertyList()) {

				int sspan = itemProperty.getFromYear() - startYear;
				double startPosition = sspan / (double) span;

				PageRecord.PropertyRecord subRecord = new PageRecord.PropertyRecord(itemProperty);
				subRecord.setStartPosition(startPosition);
				subList.add(subRecord);
			}

			record.setPropertyRecordList(subList);
		}
	}

	public int getId() {
		return id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public Century getFromCentury() {
		return fromCentury;
	}

	public PageType getType() {
		return type;
	}

	public void setType(PageType type) {
		this.type = type;
	}

	public Century getToCentury() {
		return toCentury;
	}

	public int getCenturySize() {
		return toCentury.getIndex() - fromCentury.getIndex() + 1;
	}

	public List<PageRecord> getRecordList() {
		return recordList;
	}

	public int getRecordCount() {
		return recordList.size();
	}

	public int getRecordCountWithSub() {
		int sum = recordList.size();
		for (PageRecord pageRecord : recordList) {
			if (!CollectionUtils.isEmpty(pageRecord.getSubList())) {
				sum += pageRecord.getSubList().size();
			}

			if (!CollectionUtils.isEmpty(pageRecord.getPropertyRecordList())) {
				sum += pageRecord.getPropertyRecordList().size();
			}
		}
		return sum;
	}

	// TODO 紀元前、紀元後またぎ対応
	public int getCenturyCount() {
		return toCentury.getIndex() - fromCentury.getIndex() + 1;
	}

	public String getJointPeopleIds() {
		return jointPeopleIds;
	}

	public void setJointPeopleIds(String jointPeopleIds) {
		this.jointPeopleIds = jointPeopleIds;
	}

	public String getJointNotClassificationItemIds() {
		return jointNotClassificationItemIds;
	}

	public void setJointNotClassificationItemIds(String jointNotClassificationItemIds) {
		this.jointNotClassificationItemIds = jointNotClassificationItemIds;
	}
}
