package pantheon.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import pantheon.model.User;
import pantheon.user.UserType;

public class UserJdbcHsqldbDao extends PantheonJdbcDao implements UserDao {
	protected final Log logger = LogFactory.getLog(getClass());

	private final static String FIND_BY_ID = "select * from users where user_id=?";
	private final static String FIND_BY_NAME = "select * from users where username=?";
	private final static String INSERT;
	private final static String UPDATE;
	private final static String FIND_ALL_USERS_IN_PEOPLE_HISTORY;

	static {
		{
			StringBuilder sb = new StringBuilder();
			sb.append("insert into users ");
			sb.append("(user_type,username,password,note,enabled,create_date) ");
			sb.append("values (?,?,?,?,1,now())");
			INSERT = sb.toString();
		}

		{
			StringBuilder sb = new StringBuilder();
			sb.append("update users ");
			sb.append("set ");
			sb.append("  username = ?, ");
			sb.append("  note = ? ");
			sb.append("where user_id=?");
			UPDATE = sb.toString();
		}

		{
			StringBuilder sb = new StringBuilder();
			sb.append("select * from users ");
			sb.append("where user_id in (");
			sb.append("  select update_user_id ");
			sb.append("  from " + PeopleJdbcHsqldbDao.HIST_TABLE_NAME + " ");
			sb.append("  where " + PeopleJdbcHsqldbDao.FLD_ID + "=?");
			sb.append(")");
			FIND_ALL_USERS_IN_PEOPLE_HISTORY = sb.toString();
		}
	}

	@Override
	public User findByUserId(long userId) {
		List<User> list = getSimpleJdbcTemplate().query(FIND_BY_ID, new UserMapper(), userId);
		if (list.size() == 0) {
			return null;
		}

		if (list.size() == 1) {
			return list.get(0);
		}

		throw new IllegalStateException();
	}

	@Override
	public User findByUserName(String name) {
		logger.info("PeopleJdbcHsqldbDao#findByName()");

		List<User> list = getSimpleJdbcTemplate().query(FIND_BY_NAME, new UserMapper(), name);
		if (list.size() == 0) {
			return null;
		}

		if (list.size() == 1) {
			return list.get(0);
		}

		throw new IllegalStateException();
	}

	/**
	 * peopleIdを持つPeople履歴で使用されているUpdateUserIdのUserを全て取得する。
	 */
	@Override
	public List<User> findAllUsersInPeopleHistory(long peopleId) {
		return getSimpleJdbcTemplate().query(FIND_ALL_USERS_IN_PEOPLE_HISTORY, new UserMapper(), peopleId);
	}

	@Override
	public void insert(UserType type, String userName, String password, String note) {
		logger.info("UserJdbcHsqldbDao#insert()");
		int affectedCount = getSimpleJdbcTemplate().update(INSERT, type.getPersistenceValue(), userName, password, note);
		checkHitOneRecord(affectedCount);
	}

	@Override
	public void update(long userId, String userName, String note) {
		logger.info("UserJdbcHsqldbDao#update() userId=" + userId + ",userName=" + userName);
		int affectedCount = getSimpleJdbcTemplate().update(UPDATE, userName, note, userId);
		checkHitOneRecord(affectedCount);
	}

	private static class UserMapper implements ParameterizedRowMapper<User> {
		public User mapRow(ResultSet rs, int rowNum) throws SQLException {
			User user = new User();
			user.setId(rs.getInt("user_id"));
			user.setType(UserType.getByPersistenceValue(rs.getInt("user_type")));
			user.setUserName(rs.getString("username"));
			user.setPassword(rs.getString("password"));
			user.setNote(rs.getString("note"));
			user.setCreateDate(new Date(rs.getTimestamp("create_date").getTime()));

			return user;
		}
	}
}
