package pantheon.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.util.Assert;

import pantheon.model.Page;
import pantheon.model.PageType;

public class PageJdbcHsqldbDao extends PantheonJdbcDao implements PageDao {
	protected final Log logger = LogFactory.getLog(getClass());

	private final static String TABLE_NAME = "page_ex2";

	private final static String FIND_ALL = "select * from " + TABLE_NAME;
	private final static String FIND_BY_ID = "select * from " + TABLE_NAME + " where page_id=?";
	private final static String FIND_BY_NAME = "select * from " + TABLE_NAME + " where page_name=?";
	private final static String INSERT;
	public final static String UPDATE;

	static {
		{
			StringBuilder sb = new StringBuilder();
			sb.append("insert " + TABLE_NAME + "  ");
			sb.append("(page_name,page_type,from_century,to_century,use_people_ids,use_notclassification_ids) ");
			sb.append("values (?,?,?,?,?,?)");
			INSERT = sb.toString();
		}

		{
			StringBuilder sb = new StringBuilder();
			sb.append("update " + TABLE_NAME + " ");
			sb.append("set page_name=?,page_type=?,from_century=?,to_century=?,use_people_ids=?,use_notclassification_ids=? ");
			sb.append("where page_id=?");
			UPDATE = sb.toString();
		}

	}

	@Override
	public List<Page> findAll() {
		logger.debug("PageJdbcHsqldbDao#findAll()");

		List<Page> list = getSimpleJdbcTemplate().query(FIND_ALL, new PageMapper());

		return list;
	}

	@Override
	public Page findById(int pageId) {
		logger.debug("PeopleJdbcHsqldbDao#findById()");

		List<Page> list = getSimpleJdbcTemplate().query(FIND_BY_ID, new PageMapper(), pageId);
		if (list.size() == 0) {
			return null;
		}

		if (list.size() == 1) {
			Page page = list.get(0);

			return page;
		}

		throw new IllegalStateException();
	}

	@Override
	public Page findByName(String name) {
		logger.debug("PeopleJdbcHsqldbDao#findById()");

		List<Page> list = getSimpleJdbcTemplate().query(FIND_BY_NAME, new PageMapper(), name);
		if (list.size() == 0) {
			return null;
		}

		if (list.size() == 1) {
			Page page = list.get(0);

			return page;
		}

		throw new IllegalStateException();
	}

	@Override
	public void insert(Page page, Long[] peopleIds, Long[] notClassificationItemIds) {
		Assert.notNull(page);

		String jointPeopleIds = StringUtils.join(peopleIds, ",");
		String jointNotClassificationItemIds = StringUtils.join(notClassificationItemIds, ",");

		int affectedCount = getSimpleJdbcTemplate().update(INSERT,//
				page.getName(),//
				page.getType().getCode(), //
				page.getFromCentury().getIndex(),//
				page.getToCentury().getIndex(),//
				jointPeopleIds,//
				jointNotClassificationItemIds);
		checkHitOneRecord(affectedCount);
	}

	@Override
	public void update(Page page, Long[] peopleIds, Long[] notClassificationItemIds) {

		String jointPeopleIds = StringUtils.join(peopleIds, ",");
		String jointNotClassificationItemIds = StringUtils.join(notClassificationItemIds, ",");

		int affectedCount = getSimpleJdbcTemplate().update(UPDATE,//
				page.getName(),//
				page.getType().getCode(), //
				page.getFromCentury().getIndex(),//
				page.getToCentury().getIndex(),//
				jointPeopleIds,//
				jointNotClassificationItemIds,//
				page.getId());
		checkHitOneRecord(affectedCount);
	}

	private class PageMapper implements ParameterizedRowMapper<Page> {
		public Page mapRow(ResultSet rs, int rowNum) throws SQLException {
			int id = rs.getInt("page_id");
			String name = rs.getString("page_name");
			PageType type = PageType.getByCode(rs.getInt("page_type"));
			int fromCentury = rs.getInt("from_century");
			int toCentury = rs.getInt("to_century");

			Page page = new Page(id, name, type, fromCentury, toCentury);
			page.setJointPeopleIds(rs.getString("use_people_ids"));
			page.setJointNotClassificationItemIds(rs.getString("use_notclassification_ids"));

			return page;
		}
	}

}
