package pantheon.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import pantheon.model.Book;
import pantheon.model.Item;
import pantheon.model.ItemType;

public class BookJdbcHsqldbDao extends AbstractItemJdbcHsqldbDao implements BookDao {

	private final static String FIND_ALL = FIND_ALL_PREFIX + getType().getCode();
	private final static String INSERT;
	private final static String UPDATE;

	private final static ItemType getType() {
		return ItemType.BOOK;
	}

	static {
		{
			StringBuilder sb = new StringBuilder();
			sb.append("insert into " + TABLE_NAME + " (");
			sb.append(ALL_FIELDS_JOIN_COMMA_WITHOUT_ID);
			sb.append(") values (");
			sb.append(getType().getCode() + ",");// item_type
			sb.append("?,");// item_jp_name
			sb.append("?,");// item_jp_name_kana
			sb.append("?,");// item_jp_name_short
			sb.append("?,");// item_wiki_name
			sb.append("?,");// item_en_name
			sb.append("?,");// fromYear
			sb.append("null,");// toYear
			sb.append(Item.FIRST_VIRSION + ",");// version
			sb.append("?,");// update_user_id
			sb.append("now()");// update_date
			sb.append(")");
			INSERT = sb.toString();
		}

		{
			StringBuilder sb = new StringBuilder();
			sb.append("update " + TABLE_NAME + " ");
			sb.append("set ");
			sb.append("item_jp_name=?,");
			sb.append("item_jp_name_kana=?,");
			sb.append("item_jp_name_short=?,");
			sb.append("item_wiki_name=?,");
			sb.append("item_en_name=?,");
			sb.append("fromYear=?,");
			sb.append("version=version+1,");
			sb.append("update_user_id=?,");
			sb.append("update_date=now()");
			sb.append(" where " + FLD_ID + "=? and version=?");
			UPDATE = sb.toString();
		}

	}

	@Override
	public List<Book> findAll() {
		List<Book> list = getSimpleJdbcTemplate().query(FIND_ALL, new BookMapper());
		return list;
	}

	@Override
	public Book findById(String itemId) {

		List<Book> list = getSimpleJdbcTemplate().query(FIND_BY_ID, new BookMapper(), itemId);
		return getFirstRecordAfterCheck(list);
	}

	@Override
	public Book findByName(String name) {
		List<Book> list = getSimpleJdbcTemplate().query(FIND_BY_NAME, new BookMapper(), name);
		return getFirstRecordAfterCheck(list);
	}

	@Override
	public long insert(Book book, long userId) {

		// 追加
		int affectedCount = getSimpleJdbcTemplate().update(INSERT,//
				book.getName(),// item_jp_name
				book.getKanaName(),// item_jp_name_kana
				book.getShortName(),// item_jp_name_short
				book.getWikiName(),// item_wiki_name
				book.getEnglishName(),// item_en_name
				book.getFromYear(), // fromYear
				userId// update_user_id
				);
		checkHitOneRecord(affectedCount);

		// 履歴作成
		long itemId = getLastInsertId();
		long version = Item.FIRST_VIRSION;
		affectedCount = getSimpleJdbcTemplate().update(BACKUP_ITEM, itemId, version);
		checkHitOneRecord(affectedCount);

		return itemId;
	}

	@Override
	public void update(Book book, long userId) {
		if (book.getVersion() < Item.FIRST_VIRSION) {
			throw new IllegalStateException("itemId = " + book.getId());
		}

		int affectedCount = getSimpleJdbcTemplate().update(UPDATE,//
				book.getName(), // item_jp_name
				book.getKanaName(), // item_jp_name_kana
				book.getShortName(), // item_jp_name_short
				book.getWikiName(), // item_wiki_name
				book.getEnglishName(), // item_en_name
				book.getFromYear(), //
				userId,//
				book.getId(),//
				book.getVersion()//
				);
		checkHitOneRecord(affectedCount);

		// 履歴作成
		long iteId = book.getId();
		long version = book.getVersion() + 1;
		affectedCount = getSimpleJdbcTemplate().update(BACKUP_ITEM, iteId, version);
		checkHitOneRecord(affectedCount);
	}

	@Override
	public List<Book> findAllHistoryByItemId(String itemId) {
		return getSimpleJdbcTemplate().query(FIND_ALL_HISTORY, new BookMapper(), itemId);
	}

	private static class BookMapper implements ParameterizedRowMapper<Book> {
		public Book mapRow(ResultSet rs, int rowNum) throws SQLException {
			Book item = new Book();
			item.setId(rs.getInt(FLD_ID));
			item.setName(rs.getString("item_jp_name"));
			item.setKanaName(rs.getString("item_jp_name_kana"));
			item.setShortName(rs.getString("item_jp_name_short"));
			item.setWikiName(rs.getString("item_wiki_name"));
			item.setEnglishName(rs.getString("item_en_name"));
			item.setFromYear(rs.getInt("fromYear"));
			item.setVersion(rs.getInt("version"));
			item.setUpdateUserId(rs.getInt("update_user_id"));
			item.setUpdateDate(rs.getDate("update_date"));

			return item;
		}
	}
}
