package pantheon;

import java.util.ArrayList;
import java.util.List;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.transaction.annotation.Transactional;

import pantheon.dao.ItemDao;
import pantheon.dao.ItemHistoryDao;
import pantheon.dao.PeopleDao;
import pantheon.model.Item;
import pantheon.model.ItemProperty;
import pantheon.model.Label;
import pantheon.model.People;
import pantheon.model.PropertyType;

public class PeopleServiceImpl extends AbstractDefaultReadOnlyService implements PeopleService {
	private ItemDao itemDao;
	private PeopleDao peopleDao;
	private ItemHistoryDao itemHistoryDao;
	private ItemService itemService;

	@Override
	public List<People> findAll() {
		return peopleDao.findAll();
	}

	@Override
	public People findById(String id) {
		return peopleDao.findById(id);
	}

	@Override
	public People findByName(String name) {
		return peopleDao.findByName(name);
	}

	@Override
	@Transactional
	public long insert(Item item, Long[] bookIds, long userId) {

		long itemId = itemDao.insertSimple(item, userId);

		// 履歴作成
		int version = People.I_FIRST_VIRSION;
		itemHistoryDao.backup(itemId, version);

		// 書籍関連プロパティ作成
		List<ItemProperty> writtenBookProperties = creteTargetProperties(item, bookIds, Label.WRITTEN, PropertyType.RELATION);
		if (item.getPropertyList() != null) {
			// 時点プロパティを足しこむ
			writtenBookProperties.addAll(item.getPropertyList());
		}
		item.setPropertyList(writtenBookProperties);
		item.setId(itemId);
		item.setVersion(Item.I_FIRST_VIRSION);
		itemService.insertAllProperties(item);

		return itemId;
	}

	@Override
	@Transactional
	public void update(Item item, Long[] bookIds, long userId) {
		// select for update
		itemDao.findByIdForUpdate(item.getId());

		itemDao.updateSimple(item, userId);
		long itemId = item.getId();

		// 履歴作成
		int version = item.getVersion() + 1;
		itemHistoryDao.backup(itemId, version);

		// 書籍関連プロパティ作成
		List<ItemProperty> writtenBookProperties = creteTargetProperties(item, bookIds, Label.WRITTEN, PropertyType.RELATION);
		if (item.getPropertyList() != null) {
			// 時点プロパティを足しこむ
			writtenBookProperties.addAll(item.getPropertyList());
		}
		item.setPropertyList(writtenBookProperties);
		item.setId(itemId);
		item.setVersion(version);
		itemService.insertAllProperties(item);
	}

	private List<ItemProperty> creteTargetProperties(Item itme, Long[] bookIds, Label label, PropertyType type) {
		List<ItemProperty> list = new ArrayList<ItemProperty>();

		int index = 0;
		for (Long bookId : bookIds) {
			ItemProperty property = new ItemProperty();
			property.setIndex(index++);
			property.setLabel(label);
			property.setType(type);
			property.setTargetItemId(bookId);
			list.add(property);
		}

		return list;
	}

	@Required
	public void setItemDao(ItemDao itemDao) {
		this.itemDao = itemDao;
	}

	@Required
	public void setPeopleDao(PeopleDao peopleDao) {
		this.peopleDao = peopleDao;
	}

	@Required
	public void setItemHistoryDao(ItemHistoryDao itemHistoryDao) {
		this.itemHistoryDao = itemHistoryDao;
	}

	public void setItemService(ItemService itemService) {
		this.itemService = itemService;
	}

}
