package pantheon;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.transaction.annotation.Transactional;

import pantheon.dao.BookDao;
import pantheon.dao.ItemDao;
import pantheon.dao.ItemHistoryDao;
import pantheon.dao.ItemPropertyDao;
import pantheon.dao.ItemPropertyHistoryDao;
import pantheon.dao.UserDao;
import pantheon.model.Book;
import pantheon.model.Item;
import pantheon.model.ItemProperty;
import pantheon.model.ItemType;
import pantheon.model.Label;
import pantheon.model.User;

public class ItemServiceImpl extends AbstractDefaultReadOnlyService implements ItemService {
	private UserDao userDao;
	private ItemDao itemDao;
	private ItemHistoryDao itemHistoryDao;
	private BookDao bookDao;
	private ItemPropertyDao itemPropertyDao;
	private ItemPropertyHistoryDao itemPropertyHistoryDao;

	@Override
	public Item findById(long itemId) {
		return itemDao.findById(itemId);
	}

	@Override
	public Item findByIdWithProperties(long itemId) {
		Item item = itemDao.findById(itemId);
		item.setPropertyList(itemPropertyDao.findAllById(itemId));
		return item;
	}

	@Override
	public Item findById(long itemId, int version) {
		return itemHistoryDao.findById(itemId, version);
	}

	@Override
	public Item findByIdWithProperties(long itemId, int version) {
		Item item = itemHistoryDao.findById(itemId, version);
		item.setPropertyList(itemPropertyHistoryDao.findAllById(itemId, version));
		return item;
	}

	@Override
	public Item findByName(String name) {
		return itemDao.findByName(name);

	}

	@Override
	public List<Book> findAllBook() {
		return bookDao.findAll();
	}

	@Override
	public List<Item> findAllItem(ItemType itemType) {
		return itemDao.findAllItem(itemType);
	}

	@Override
	public List<Item> findAllTargetItem(long itemId, Label label) {
		return itemDao.findAllTargetItem(itemId, label);
	}

	@Override
	public List<Item> findAllTargetItem(long itemId, Label label, int version) {
		return itemHistoryDao.findAllTargetItem(itemId, label, version);
	}

	@Override
	public List<Item> findAllItemByTargetItem(Label label, long tagetId) {
		return itemDao.findAllItemByTargetItem(label, tagetId);
	}

	@Override
	public List<Item> findAllHistory(long itemId) {

		List<User> userList = userDao.findAllUsersInPeopleHistory(itemId);
		Map<Long, User> userMap = new HashMap<Long, User>();
		for (User user : userList) {
			userMap.put(user.getId(), user);
		}

		List<Item> itemList = itemDao.findAllHistory(itemId);
		for (Item item : itemList) {
			item.setUpdateUser(userMap.get(item.getUpdateUserId()));
		}
		return itemList;
	}

	@Override
	@Transactional
	public long insert(Item item, long userId) {
		long itemId = itemDao.insert(item, userId);
		item.setId(itemId);
		item.setVersion(Item.I_FIRST_VIRSION);
		insertAllProperties(item);
		return itemId;
	}

	@Override
	@Transactional
	public void update(Item item, long userId) {
		int newVersion = itemDao.update(item, userId);
		item.setVersion(newVersion);
		insertAllProperties(item);
	}

	@Override
	public List<ItemProperty> findAllPropertyByItemId(long itemId) {
		return itemPropertyDao.findAllById(itemId);
	}

	@Override
	public List<ItemProperty> findAllPropertyByItemId(long itemId, int version) {
		return itemPropertyHistoryDao.findAllById(itemId, version);
	}

	@Override
	@Transactional
	public void insertAllProperties(Item item) {
		itemPropertyDao.deleteAll(item.getId());

		int index = 0;
		if (CollectionUtils.isNotEmpty(item.getPropertyList())) {
			for (ItemProperty property : item.getPropertyList()) {
				property.setItemId(item.getId());
				property.setIndex(index++);
				itemPropertyDao.insert(property);
			}
		}
		itemPropertyHistoryDao.backup(item.getId(), item.getVersion());
	}

	@Required
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}

	@Required
	public void setItemDao(ItemDao itemDao) {
		this.itemDao = itemDao;
	}

	@Required
	public void setItemHistoryDao(ItemHistoryDao itemHistoryDao) {
		this.itemHistoryDao = itemHistoryDao;
	}

	@Required
	public void setBookDao(BookDao bookDao) {
		this.bookDao = bookDao;
	}

	@Required
	public void setItemPropertyDao(ItemPropertyDao itemPropertyDao) {
		this.itemPropertyDao = itemPropertyDao;
	}

	@Required
	public void setItemPropertyHistoryDao(ItemPropertyHistoryDao itemPropertyHistoryDao) {
		this.itemPropertyHistoryDao = itemPropertyHistoryDao;
	}
}
