<?php

/*
 *  tiny OAI-PMH data provider framework for php4(or higher)
 *  Yamamoto.T
 *  
 *  2006.9.7  ver. 0.85
 *  2006.9.15 ver. 0.86 bugfix
 *  2006.9.28 ver. 0.87 add comment
 *            ver. 0.88 add initialize/finalize
 *  2007.8.3  ver. 0.89 bugfix(parameter 'until' doesn't work well)
 *
 *  this script cannot run by itself. write your script that 'include' this.
 *    example:
 *      <?php
 *          include 'tinyoai.php.inc';
 *            ... define your functions ...
 *          oai_mainroutine();
 *      ?>
 *
 *  in your script, you must implement your original functions below:
 *    * fw_identify (optional)
 *    * fw_listsets (optional)
 *    * fw_listmetadataformats
 *    * fw_getrecord
 *    * fw_getrecords
 *    * fw_initialize / fw_finalize (if you need)
 *
 *  read README for details.
 */

define ('OAI_ERROR_NO_SET', 1);
define ('OAI_ERROR_NO_FORMAT', 2);
define ('OAI_ERROR_NO_HIT', 3);

// set default constants if not defined
function oai_check_constants() {

  $default = array(
    'REPOSITORY_NAME' => 'My Repository',
    'ADMIN_EMAIL' => 'anonymous@localhost',
    'DELETED_RECORD' => 'transient',
    'EARLIEST_DATE' => '1900-01-01',
    'LIST_MAX' =>50,
  );
  $default['BASEURL'] = "http://" . $_SERVER['SERVER_NAME'] . $_SERVER['SCRIPT_NAME'];

  foreach($default as $key => $value) {
    if (!defined($key)) { define ($key, $value); }
  }
}

// build XML segment (header element)
function oai_array2recordheader(&$rec) {

  $buf = ($rec['deleted'] ? '<header status="deleted">' : '<header>');
  $buf .= "<identifier>" . $rec['id'] . "</identifier>";
  $buf .= "<datestamp>" . $rec['datestamp'] . "</datestamp>";
  if (is_array($rec['set'])) {
    foreach ($rec['set'] as $i) { $buf .= "<setSpec>".$i."</setSpec>"; }
  }
  $buf .= "</header>";
  return $buf;
}

// build XML segment (record element)
function oai_array2fullrecord(&$rec) {

  $buf = "<record>";
  $buf .= oai_array2recordheader(&$rec);
  if (!$rec['deleted']) {
    $buf .= "<metadata>" . $rec['metadata'] . "</metadata>";
  }
  $buf .= "</record>";
  return $buf;
}

// validate date string (granurality YYYY-MM-DD only)
function oai_validate_date($value) {
  if (! $value) {
    return TRUE;
  }
  if (strlen($value) != 10) {
    return FALSE;
  }
  $a = sscanf($value, "%04d-%02d-%02d");
  if (! checkdate($a[1], $a[2], $a[0])) {
    return FALSE;
  }
  return TRUE;
}

function oai_do_identify(&$r) {

  if ($r['identifier'] or $r['from'] or $r['until'] or $r['set'] or $r['resumptionToken']) {
    return array('badArgument', 'illegal argument for Identify');
  }

  $r['oai_body'] =
       "<repositoryName>".REPOSITORY_NAME."</repositoryName>" .
       "<baseURL>".BASEURL."</baseURL>" .
       "<protocolVersion>2.0</protocolVersion>" .
       "<earliestDatestamp>".EARLIEST_DATE."</earliestDatestamp>" .
       "<deletedRecord>".DELETED_RECORD."</deletedRecord>" .
       "<granularity>YYYY-MM-DD</granularity>" .
       "<adminEmail>".ADMIN_EMAIL."</adminEmail>";

  if (function_exists("fw_identify")) {
    $r['oai_body'] .= fw_identify();
  }

  return NULL;
}

function oai_do_listmetadataformats(&$r) {

  if ($r['from'] or $r['until'] or $r['set'] or $r['resumptionToken']) {
    return array('badArgument', 'illegal argument for ListMetadataFormat');
  }

  if (!function_exists("fw_listmetadataformats")) {
    return array('noMetadataFormats', 'ListMetadataFormats not implemented');
  }

  $r['oai_body'] = fw_listmetadataformats($r['identifier']);

  if ($r['oai_body'] == OAI_ERROR_NO_FORMAT) {
    return array('noMetadataFormats', 'no format available');
  } elseif ($r['oai_body'] == OAI_ERROR_NO_HIT) {
    return array('idDoesNotExist', 'no such identifier available');
  }

  return NULL;
}

function oai_do_listsets(&$r) {

  if ($r['identifier'] or $r['from'] or $r['until'] or $r['set'] or $r['resumptionToken']) {
    return array('badArgument', 'illegal argument for ListSets');
  }

  if (!function_exists("fw_listsets")) {
    return array('noSetHierarchy', 'ListSets not implemented');
  }

  $r['oai_body'] = fw_listsets();

  if ($r['oai_body'] == OAI_ERROR_NO_SET) {
    return array('noSetHierarchy', 'no set');
  }

  return NULL;
}

function oai_do_getrecord(&$r) {

  if (!$r['identifier'] or !$r['metadataPrefix']) {
    return array('badArgument', 'some argument missing');
  }

  if ($r['from'] or $r['until'] or $r['set'] or $r['resumptionToken']) {
    return array('badArgument', 'illegal argument for GetRecord');
  }

  if (strstr($r['identifier'], "\"")) {
    return array('badArgument', 'malformed identifier');
  }

  if (!function_exists("fw_getrecord")) {
    return array('idDoesNotExist ', 'GetRecord not implemented');
  }

  $rec = fw_getrecord($r['identifier'], $r['metadataPrefix']);

  if ($rec == OAI_ERROR_NO_FORMAT) {
    return array('noMetadataFormats', 'no format available');
  } elseif ($rec == OAI_ERROR_NO_HIT) {
    return array('idDoesNotExist', 'no such identifier available');
  }

  $r['oai_body'] = oai_array2fullrecord(&$rec);
  return NULL;
}

function oai_do_listrecords(&$r, $mode) {

  if ($r['identifier']) {
    return array('badArgument', 'illegal argument for ListRecords(ListIdentifiers)');
  }

  if ($r['resumptionToken']) {
    if ($r['metadataPrefix'] or $r['from'] or $r['until'] or $r['set']) {
      return array('badArgument', 'illegal argument (query redundant)');
    } else {
      list($r['metadataPrefix'], $r['offset'], $r['from'], $r['until'], $r['set']) = explode("!", $r['resumptionToken']);
      if (! $r['offset']) {
        return array('badResumptionToken', 'malformed resumptionToken');
      }
      $r['resumptionToken'] = NULL;
    }
  } else {
    if (!$r['metadataPrefix']) {
      return array('badArgument', 'metadataPrefix missing');
    }
    $r['offset'] = 1;
  }

  if (! oai_validate_date($r['from'])) {
     return array('badArgument', 'malformed date');
  }

  if (! oai_validate_date($r['until'])) {
     return array('badArgument', 'malformed date');
  }

  if ($r['until']) {
    if ($r['until'] < EARLIEST_DATE) {
      return array('noRecordsMatch', 'too old record requested');
    }
  }

  if (!function_exists("fw_getrecords")) {
    return array('noRecordsMatch', 'ListRecords(ListIdentifiers) not implemented');
  }

  $recs = fw_getrecords($r['metadataPrefix'], $r['from'], $r['until'], $r['set'], $r['offset'], LIST_MAX+1);

  if ($recs == OAI_ERROR_NO_SET) {
    return array('noSetHierarchy', 'no set');
  } elseif ($recs == OAI_ERROR_NO_HIT) {
    return array('noRecordsMatch', 'no records hit');
  } elseif ($recs == OAI_ERROR_NO_FORMAT) {
    return array('noMetadataFormats', 'no such format available');
  }

  if (count($recs) == 0) {
    return array('noRecordsMatch', 'no records hit');
  }

  $max = (count($recs) > LIST_MAX ? LIST_MAX : count($recs));

  for ($i = 0; $i < $max; $i++) {
    if ($mode == 'i') {
      $r['oai_body'] .= oai_array2recordheader(&$recs[$i]);
    } else { // mode == 'r'
      $r['oai_body'] .= oai_array2fullrecord(&$recs[$i]);
    }
  }

  if (count($recs) > LIST_MAX) {
    $r['oai_body'] .= "<resumptionToken>"
          . rtrim(join("!", array($r['metadataPrefix'], $r['offset'] + LIST_MAX, $r['from'], $r['until'], $r['set'])),"!")
          . "</resumptionToken>";
  } elseif ($r['offset'] > 1) {
    $r['oai_body'] .= "<resumptionToken/>";
  }

  return NULL;
}

// MAIN ROUTINE
function oai_mainroutine() {

  if (function_exists("fw_initialize")) {
    fw_initialize();
  }

  oai_check_constants();

  $r = array();

  foreach (array('verb', 'set', 'identifier', 'from', 'until', 'metadataPrefix', 'resumptionToken') as $i) {
    if ($_POST[$i]) { $_GET[$i] = $_POST[$i]; }
    $r[$i] = $_GET[$i];
  }

  $r['oai_body'] = '';

  switch($r['verb']) {
  case 'Identify':
    $err = oai_do_identify(&$r);
    break;
  case 'ListMetadataFormats':
    $err = oai_do_listmetadataformats(&$r);
    break;
  case 'ListSets':
    $err = oai_do_listsets(&$r);
    break;
  case 'ListIdentifiers':
    $err = oai_do_listrecords(&$r, "i");
    break;
  case 'GetRecord':
    $err = oai_do_getrecord(&$r);
    break;
  case 'ListRecords':
    $err = oai_do_listrecords(&$r, "r");
    break;
  default:
    $r['verb'] = NULL;
    $err = array('badVerb', 'illegal OAI verb');
  }

  if (function_exists("fw_finalize")) {
    fw_finalize();
  }

  header("Content-type: text/xml");

  echo <<<END_HEREDOC
<?xml version="1.0" encoding="utf-8"?>
<OAI-PMH xmlns="http://www.openarchives.org/OAI/2.0/"
         xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:schemaLocation="http://www.openarchives.org/OAI/2.0/
         http://www.openarchives.org/OAI/2.0/OAI-PMH.xsd">
END_HEREDOC;

  echo "<responseDate>" . gmdate("Y-m-d\TH:i:s\Z", time()) . "</responseDate>";

  if ($r['verb']) {
    echo "<request verb=\"" . $r['verb'] . "\"";
    if (! $err) {
      foreach (array('set', 'identifier', 'from', 'until', 'metadataPrefix', 'resumptionToken') as $i) {
        if ($_GET[$i]) { echo " $i=\"" . htmlspecialchars($_GET[$i]) . "\""; }
      }
    }
    echo ">".BASEURL."</request>";
  } else {
    echo "<request>".BASEURL."</request>";
  }

  if ($err) {
    echo "<error code=\"" . $err[0] . "\">" . $err[1] . "</error>";
  } else {
    echo "<" . $r['verb'] . ">" . $r['oai_body'] . "</" . $r['verb'] . ">";
  }

  echo "</OAI-PMH>";
}

?>
